#!/usr/share/ucs-test/runner pytest-3 -s -l -v
# -*- coding: utf-8 -*-
## desc: Test local expolit reading room file
## roles: [domaincontroller_master, domaincontroller_slave]
## tags: [apptest,ucsschool,ucsschool_base1]
## exposure: dangerous
## packages: [ucs-school-umc-computerroom]
## bugs: [43039, 50009]

import logging
import os
import sys
from unittest import TestCase, main

import pytest

import univention.testing.strings as uts
import univention.testing.ucr as ucr_test
from univention.management.console.modules import UMC_Error
from univention.management.console.modules.computerroom import _readRoomInfo


class TestComputerRoom(TestCase):
    local_file_path = ""
    text_in_local_file = ""
    ucr = None
    logger = None

    @classmethod
    def setUpClass(cls):
        logging.basicConfig(stream=sys.stderr, level=logging.DEBUG)
        cls.ucr = ucr_test.UCSTestConfigRegistry()
        cls.ucr.load()
        cls.local_file_path = "/etc/{}".format(uts.random_username())
        cls.logger = logging.getLogger("TestLog")
        cls.logger.info("Test file is %r.", cls.local_file_path)

    @classmethod
    def tearDownClass(cls):
        cls.ucr = None
        os.remove(cls.local_file_path)
        cls.logger.info("Removed %r.", cls.local_file_path)

    @classmethod
    def setUp(self):
        self.text_in_local_file = uts.random_string(30)
        with open(self.local_file_path, "w") as fp:
            fp.write(self.text_in_local_file)
        self.logger.info("Wrote to %r:\n%s\n------------", self.local_file_path, self.text_in_local_file)

    def _run_readRoomInfo(self, room_dn, expected_exception=None):
        self.logger.info(
            'Trying to read info for "room DN" %r, expecting %s exception...',
            room_dn,
            "an" if expected_exception else "no",
        )
        if expected_exception:
            with pytest.raises(expected_exception):
                info = _readRoomInfo(
                    "cn={},cn=groups,ou=foo,{}".format(self.local_file_path, self.ucr["ldap/base"])
                )
                self.logger.error("No exception raised. _readRoomInfo() returned: %r", info)
        else:
            info = _readRoomInfo(room_dn)
            self.logger.info("OK: no exception raised. _readRoomInfo() returned: %r", info)
            assert info == {}
        with open(self.local_file_path) as fp:
            txt = fp.read()
        assert txt == self.text_in_local_file

    def test_bad_dn(self):
        room_dn = "cn={},cn={},cn=groups,ou=foo,{}".format(
            self.local_file_path,
            self.ucr.get("ucsschool/ldap/default/container/rooms", "raeume"),
            self.ucr["ldap/base"],
        )
        self._run_readRoomInfo(room_dn, UMC_Error)

    def test_good_dn(self):
        room_dn = "cn={},cn={},cn=groups,ou=foo,{}".format(
            uts.random_groupname(),
            self.ucr.get("ucsschool/ldap/default/container/rooms", "raeume"),
            self.ucr["ldap/base"],
        )
        self._run_readRoomInfo(room_dn)

    def test_umlaute_dn(self):
        room_dn = "cn={},cn={},cn=groups,ou=foo,{}".format(
            uts.random_groupname() + "öüä" + uts.random_groupname(),
            self.ucr.get("ucsschool/ldap/default/container/rooms", "raeume"),
            self.ucr["ldap/base"],
        )
        self._run_readRoomInfo(room_dn)


if __name__ == "__main__":
    main(verbosity=2)
