import uuid

import pytest
from fastapi import HTTPException, status

import provisioning_plugin.dependencies
from id_broker_common.utils import ldap_settings
from provisioning_plugin.routes.v1 import schools
from provisioning_plugin.utils import remove_school_objects
from ucsschool.kelvin.client import NoObject as KelvinNoObject
from udm_rest_client import NoObject

from ..utils import DEFAULT_SCHOOL_AUTHORITY, KelvinSessionMock, kelvin_school


def test_school_by_name(func_client, broker_school):
    _client = func_client(
        custom_overrides={
            provisioning_plugin.dependencies.get_school_by_id: kelvin_school,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    response = _client.get(
        url=f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/schools/{broker_school.name}"
    )
    assert response.status_code == 200
    assert response.json() == broker_school


@pytest.mark.parametrize("ou_exists", (True, False))
def test_school_get_by_name(func_client, broker_school, ou_exists: bool):
    async def _get_school_by_id():  # -> KelvinSchool
        print("*** _get_school_by_id() ***")
        if ou_exists:
            return kelvin_school()
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND)

    _client = func_client(
        custom_overrides={
            provisioning_plugin.dependencies.get_school_by_id: _get_school_by_id,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    ou_name = broker_school.name if ou_exists else "fake_ou123"  # TODO
    response = _client.get(url=f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/schools/{ou_name}")
    assert response.status_code == 200 if ou_exists else 404
    if ou_exists:
        assert response.json() == broker_school


@pytest.mark.parametrize("ou_exists", (True, False))
def test_school_exists_by_name(func_client, broker_school, ou_exists: bool):
    async def _find_school() -> str:
        if ou_exists:
            return broker_school.name
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND)

    _client = func_client(
        custom_overrides={
            provisioning_plugin.dependencies.find_school_name_by_id_in_primary_ldap: _find_school,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    ou_name = broker_school.name if ou_exists else "fake_ou123"  # TODO
    response = _client.head(url=f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/schools/{ou_name}")
    assert response.status_code == 200 if ou_exists else 404


def test_post_school(func_client, broker_school, monkeypatch, async_iterable, mocker):
    _client = func_client(
        custom_overrides={
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    monkeypatch.setattr(
        schools,
        "get_pseudonyms_for_providers",
        async_iterable,
    )
    kelvinSessionMock = KelvinSessionMock()
    kelvinSessionMock.configure_mock(
        **{
            "post.return_value": kelvin_school(),
            "__aenter__.return_value": kelvinSessionMock,
        }
    )
    mocker.patch(
        "provisioning_plugin.routes.v1.schools.IDBrokerKelvinSession",
        lambda *args, **kwargs: kelvinSessionMock,
    )
    response = _client.post(
        f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/schools", json=broker_school
    )
    assert response.status_code == 201
    assert response.json() == broker_school


@pytest.mark.parametrize(
    "ou_exists", (True, False), ids=["ou_exists", "ou_not_existing"]
)
def test_delete_school(func_client, broker_school, ou_exists, mocker):
    async def _get_school_by_id():
        if ou_exists:
            return kelvin_school()
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND)

    mocker.patch(
        "provisioning_plugin.routes.v1.schools.remove_school_objects",
    )
    _client = func_client(
        custom_overrides={
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
            provisioning_plugin.dependencies.get_school_by_id: _get_school_by_id,
        }
    )
    response = _client.delete(
        url=f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/schools/{broker_school.name}"
    )
    if ou_exists:
        assert response.status_code == 204
    else:
        assert response.status_code == 404


class MockObject:
    deleted_objs = []

    def __init__(self, dn: str = None):
        if dn:
            self.obj_id = dn
        else:
            self.obj_id = uuid.uuid4()
        self.name = self.obj_id

    async def delete(self):
        self.deleted_objs.append(self.obj_id)


class UDMMock:
    def __init__(self, **kwargs):
        pass

    async def __aenter__(self):
        return self

    async def __aexit__(self, exc_type, exc_val, exc_tb):
        return self

    def get(self, module):
        return {
            "groups/group": UDMModuleMock(),
            "container/ou": UDMModuleMock(),
        }[module]


class UDMModuleMock:
    deleted_objs = []

    async def get(self, dn):
        return MockObject(dn)


@pytest.fixture(autouse=True)
def set_envs(monkeypatch):
    monkeypatch.setenv("LDAP_BASE", "dc=ldap,dc=base")
    monkeypatch.setenv("LDAP_HOSTDN", "provisioning1.broker.test")
    monkeypatch.setenv("LDAP_SERVER_PORT", "7389")
    monkeypatch.setenv("LDAP_MASTER", "idbroker-primary.broker.test")
    monkeypatch.setenv("LDAP_MASTER_PORT", "7389")
    monkeypatch.setenv("LDAP_SERVER_NAME", "provisioning1.broker.test")


@pytest.mark.asyncio
async def test_remove_school_objects(mocker, monkeypatch):
    users = [MockObject() for _ in range(10)]
    groups = [MockObject() for _ in range(10)]
    ldap_base = "dc=ldap,dc=base"
    ldap_settings.cache_clear()
    monkeypatch.setenv("ldap_base", "dc=ldap,dc=base")
    mocker.patch("provisioning_plugin.utils.get_admin_connection_kwargs")
    mocker.patch(
        "provisioning_plugin.utils.users_of_school",
        return_value=users,
    )
    mocker.patch(
        "provisioning_plugin.utils.groups_of_school",
        return_value=groups,
    )
    mocker.patch(
        "provisioning_plugin.utils.UDM",
        UDMMock,
    )
    kelvin_session_mock = KelvinSessionMock()
    await remove_school_objects(school="MySchool", session=kelvin_session_mock)
    expected_user_ids = {user.obj_id for user in users}
    expected_group_ids = {group.obj_id for group in groups}
    assert expected_user_ids.issubset(set(MockObject.deleted_objs))
    assert expected_group_ids.issubset(set(MockObject.deleted_objs))
    school = "MySchool"
    assert {
        f"cn=OU{school}-Member-Verwaltungsnetz,cn=ucsschool,cn=groups,{ldap_base}",
        f"cn=OU{school}-Member-Edukativnetz,cn=ucsschool,cn=groups,{ldap_base}",
        f"cn=OU{school}-Klassenarbeit,cn=ucsschool,cn=groups,{ldap_base}",
        f"cn=OU{school}-DC-Verwaltungsnetz,cn=ucsschool,cn=groups,{ldap_base}",
        f"cn=OU{school}-DC-Edukativnetz,cn=ucsschool,cn=groups,{ldap_base}",
        f"cn=admins-{school},cn=ouadmins,cn=groups,{ldap_base}",
    }.issubset(MockObject.deleted_objs)
    assert f"ou={school},{ldap_base}" in MockObject.deleted_objs


class UDMModuleAlreadyDeletedMock:
    deleted_objs = []

    async def get(self, dn):
        raise NoObject(dn)


class UDMModuleAlreadyDeletedObjectsMock(UDMMock):
    def get(self, module):
        return {
            "groups/group": UDMModuleAlreadyDeletedMock(),
            "container/ou": UDMModuleAlreadyDeletedMock(),
        }[module]


@pytest.mark.asyncio
async def test_remove_school_objects_previous_unfinished_removal(mocker):
    mocker.patch("provisioning_plugin.utils.get_admin_connection_kwargs")
    mocker.patch(
        "provisioning_plugin.utils.users_of_school",
        return_value=[],
    )
    mocker.patch(
        "provisioning_plugin.utils.groups_of_school",
        return_value=[],
    )
    mocker.patch(
        "provisioning_plugin.utils.UDM",
        UDMModuleAlreadyDeletedObjectsMock,
    )
    kelvin_session_mock = KelvinSessionMock()
    await remove_school_objects(school="MySchool", session=kelvin_session_mock)


class AlreadyRemovedKelvinObject(MockObject):
    async def delete(self):
        raise KelvinNoObject()


@pytest.mark.asyncio
async def test_already_deleted_objects_raise_no_errors(mocker):
    mocker.patch("provisioning_plugin.utils.get_admin_connection_kwargs")
    mocker.patch(
        "provisioning_plugin.utils.users_of_school",
        return_value=[AlreadyRemovedKelvinObject("not-longer-existing-user")],
    )
    mocker.patch(
        "provisioning_plugin.utils.groups_of_school",
        return_value=[AlreadyRemovedKelvinObject("not-longer-existing-group")],
    )
    mocker.patch(
        "provisioning_plugin.utils.UDM",
        UDMModuleAlreadyDeletedObjectsMock,
    )
    kelvin_session_mock = KelvinSessionMock()
    await remove_school_objects(school="MySchool", session=kelvin_session_mock)
