import logging
import time

from fastapi import APIRouter, Depends, Path

from id_broker_common.utils import remove_prefix
from self_disclosure_plugin.dependencies import (
    connecting_user_pseudonym_from_token,
    convert_pks_to_basic_users,
    get_connected_user,
    get_group_by_id,
    sddb_rest_client,
    service_provider_from_token,
    token_dep,
    username_from_user,
)
from self_disclosure_plugin.models import CountedGroup, NoStarStr, User, UsersGroups
from self_disclosure_plugin.sddb_client import (
    APIQueueEntryType,
    NotFoundError,
    SddbGroup,
    SddbUser,
)
from ucsschool.apis.opa import OPAClient, opa_instance
from ucsschool.apis.utils import get_logger

router = APIRouter(tags=["users"], dependencies=[Depends(token_dep)])


@router.get("/{id}/metadata", response_model=User)
async def user_metadata(
    pseudonym_of_requested_user: NoStarStr = Path(
        ...,
        alias="id",
        description="Pseudonym of user for which to retrieve data.",
        title="Pseudonym",
    ),
    pseudonym_of_connecting_user: str = Depends(connecting_user_pseudonym_from_token),
    connected_user: SddbUser = Depends(get_connected_user),
    policy_instance: OPAClient = Depends(opa_instance),
    service_provider: str = Depends(service_provider_from_token),
    logger: logging.Logger = Depends(get_logger),
) -> User:
    logger.debug(
        "pseudonym_of_requested_user=%r pseudonym_of_connecting_user=%r connected_user=%r",
        pseudonym_of_requested_user,
        pseudonym_of_connecting_user,
        connected_user,
    )
    await policy_instance.check_policy_true_or_raise(
        "self_disclosure_plugin/users_self_info",
        {
            "sub": pseudonym_of_connecting_user,
            "pseudonym": pseudonym_of_requested_user,
            "user_roles": connected_user.data["roles"],
        },
    )
    school_authority_prefix = f"{connected_user.school_authority}-"
    return User(
        user_id=pseudonym_of_requested_user,
        username=username_from_user(connected_user),
        firstname=connected_user.data["firstname"],
        lastname=connected_user.data["lastname"],
        type=", ".join(connected_user.data["roles"]),  # TODO what exactly to use here?
        school_id=remove_prefix(connected_user.school, school_authority_prefix),
        school_authority=connected_user.school_authority,
        legal_guardians=convert_pks_to_basic_users(
            pseudonym_of_requested_user,
            getattr(connected_user, "legal_guardians", []),
            service_provider,
        ),
        legal_wards=convert_pks_to_basic_users(
            pseudonym_of_requested_user,
            getattr(connected_user, "legal_wards", []),
            service_provider,
        ),
    )


@router.get("/{id}/groups", response_model=UsersGroups)
async def user_groups(
    pseudonym_of_requested_user: NoStarStr = Path(
        ...,
        alias="id",
        description="Pseudonym of user for which to retrieve data.",
        title="Pseudonym",
    ),
    pseudonym_of_connecting_user: str = Depends(connecting_user_pseudonym_from_token),
    connected_user: SddbUser = Depends(get_connected_user),
    service_provider: str = Depends(service_provider_from_token),
    policy_instance: OPAClient = Depends(opa_instance),
    logger: logging.Logger = Depends(get_logger),
) -> UsersGroups:
    logger.debug(
        "pseudonym_of_requested_user=%r pseudonym_of_connecting_user=%r connected_user=%r "
        "connected_user.data['school_classes']=%r, connected_user.data['workgroups']=%r, service_provider=%r",
        pseudonym_of_requested_user,
        pseudonym_of_connecting_user,
        connected_user,
        connected_user.data["school_classes"],
        connected_user.data["workgroups"],
        service_provider,
    )
    await policy_instance.check_policy_true_or_raise(
        "self_disclosure_plugin/users_self_info",
        {
            "sub": pseudonym_of_connecting_user,
            "pseudonym": pseudonym_of_requested_user,
            "user_roles": connected_user.data["roles"],
        },
    )

    groups = []
    t0 = time.time()
    for group_id in connected_user.school_classes + connected_user.workgroups:
        try:
            group: SddbGroup = get_group_by_id(
                group_id=group_id, service_provider=service_provider
            )
        except NotFoundError:
            logger.warning("Ignoring school group not found with ID %r.", group_id)
            sddb_rest_client().enqueue(
                service_provider=service_provider,
                entry_type=APIQueueEntryType.user,
                pseudonym=pseudonym_of_requested_user,
            )
            sddb_rest_client().enqueue(
                service_provider=service_provider,
                entry_type=APIQueueEntryType.group,
                entry_uuid=group_id,
            )
            continue

        groups.append(
            CountedGroup(
                group_id=group.pseudonym,
                name=group.name,
                school_id=group.school,
                school_authority=group.school_authority,
                student_count=len(group.students),
                type=group.type,
            )
        )
    logger.debug(
        "Retrieved %d groups in %0.3f sec from Redis.",
        len(connected_user.school_classes + connected_user.workgroups),
        time.time() - t0,
    )
    return UsersGroups(groups=groups)
