from typing import List

from pydantic import BaseModel, constr

# Order of models is defined by the order of occurrence in the self_disclosure plugin

NonEmptyStr = constr(min_length=1)
NoStarStr = constr(regex=r"^[^*]+$")


class BasicUser(BaseModel):  # response model
    user_id: str
    username: str
    firstname: str
    lastname: str


class User(BasicUser):  # response model
    type: str
    school_id: str
    school_authority: str
    legal_guardians: List[BasicUser]
    legal_wards: List[BasicUser]

    class Config:
        schema_extra = {
            "example": {
                "user_id": "d8aa843b-9628-42b8-9637-d198c745efa5",
                "username": "m.mustermann",
                "firstname": "Max",
                "lastname": "Mustermann",
                "type": "teacher",
                "school_id": "School1",
                "school_authority": "SchoolAuthority1",
                "legal_guardians": [],
                "legal_wards": [],
            }
        }


class Group(BaseModel):
    group_id: str
    type: str
    name: str
    school_id: str
    school_authority: str


class CountedGroup(Group):
    student_count: int


class UsersGroups(BaseModel):  # response model
    groups: List[CountedGroup]

    class Config:
        schema_extra = {
            "example": {
                "groups": [
                    {
                        "group_id": "503f1278-9cda-49f6-a749-3af022a7d32b",
                        "name": "1a",
                        "school_id": "School1",
                        "school_authority": "SchoolAuthority1",
                        "student_count": 2,
                        "type": "school_class",
                    },
                    {
                        "group_id": "3be3c78b-6263-4c42-9d5e-14ec95f0c849",
                        "name": "2b",
                        "school_id": "School1",
                        "school_authority": "SchoolAuthority1",
                        "student_count": 4,
                        "type": "workgroup",
                    },
                ]
            }
        }


class Teacher(BaseModel):
    user_id: str
    username: str


class Student(Teacher):
    firstname: str
    lastname: str
    legal_guardians: List[BasicUser]


class LegalGuardian(Teacher):
    firstname: str
    lastname: str
    legal_wards: List[BasicUser]


class GroupMembers(BaseModel):  # response model
    students: List[Student]
    teachers: List[Teacher]
    legal_guardians: List[LegalGuardian]

    class Config:
        schema_extra = {
            "example": {
                "students": [
                    {
                        "user_id": "0e49f8b0-9c98-4716-9fe1-23d4466d9af8",
                        "username": "jane.doe",
                        "firstname": "Jane",
                        "lastname": "Doe",
                        "legal_guardians": [],
                    },
                    {
                        "user_id": "86f753e0-50b3-44cb-807e-2c2556b0e6ca",
                        "username": "john.doe",
                        "firstname": "John",
                        "lastname": "Doe",
                        "legal_guardians": [
                            {
                                "user_id": "616890bf-746e-44a1-8820-03a479c03531",
                                "username": "victor.vorlage",
                                "firstname": "Victor",
                                "lastname": "Vorlage",
                            }
                        ],
                    },
                ],
                "teachers": [
                    {
                        "user_id": "d8aa843b-9628-42b8-9637-d198c745efa5",
                        "username": "m.mustermann",
                    }
                ],
                "legal_guardians": [
                    {
                        "user_id": "8d3484ce-cfa2-4478-86fe-267592c7adf3",
                        "username": "berta.beispiel",
                        "firstname": "Berta",
                        "lastname": "Beispiel",
                        "legal_wards": [],
                    },
                    {
                        "user_id": "616890bf-746e-44a1-8820-03a479c03531",
                        "username": "victor.vorlage",
                        "firstname": "Victor",
                        "lastname": "Vorlage",
                        "legal_wards": [
                            {
                                "user_id": "86f753e0-50b3-44cb-807e-2c2556b0e6ca",
                                "username": "john.doe",
                                "firstname": "John",
                                "lastname": "Doe",
                            }
                        ],
                    },
                ],
            }
        }


class Groups(BaseModel):  # response model
    groups: List[Group]


class UdmProperties(BaseModel):
    brokerID: str
