import html
import json
import os
import re
from urllib.parse import urljoin

import requests

UCS_ENV_TEST_APP_FQDN = os.environ.get("UCS_ENV_TEST_APP_FQDN")
UCS_ENV_KEYCLOAK_FQDN = os.environ.get("UCS_ENV_KEYCLOAK_FQDN")
UCS_ENV_TEST_APP_SECRET = os.environ.get("UCS_ENV_TEST_APP_SECRET")

if (
    not UCS_ENV_TEST_APP_FQDN
    or not UCS_ENV_KEYCLOAK_FQDN
    or not UCS_ENV_TEST_APP_SECRET
):
    raise ValueError(
        "ENV not set, check: UCS_ENV_TEST_APP_FQDN, UCS_ENV_KEYCLOAK_FQDN and UCS_ENV_TEST_APP_SECRET"
    )

TESTAPP_URL = f"https://{UCS_ENV_TEST_APP_FQDN}/univention-test-app/"
TESTAPP_PARAM = "login?kc_idp_hint={idp_hint}&pkce=y"


class LoginError(Exception):
    """Custom error for everything Login related"""


def _extract_relay_state(page):
    relay_state = re.search(
        'name="RelayState" value="([^"]+)"', page.content.decode("utf-8")
    )
    if relay_state is None:
        raise LoginError("could not extract relay state")
    return relay_state.group(1)


def _extract_saml_msg(page):
    saml_message = re.search(
        'name="SAMLRequest" value="([^"]+)"', page.content.decode("utf-8")
    )
    if saml_message is None:
        raise LoginError("could not extract saml message\n%s" % page.text)
    return saml_message.group(1)


def _extract_saml_response(page):
    saml_response = re.search(
        'name="SAMLResponse" value="([^"]+)"', page.content.decode("utf-8")
    )
    if saml_response is None:
        raise LoginError("could not extract saml response")
    return saml_response.group(1)


def _extract_kc_url(page):
    url = re.search(
        'form method="post"[ \r\n]* action="([^"]+)"', page.content.decode("utf-8")
    )
    if url is None:
        error = re.search(
            "Nutzername oder Passwort falsch", page.content.decode("utf-8")
        )
        if error is None:
            raise LoginError("could not extract keycloak url")
        else:
            raise LoginError("invalid user")
    return url.group(1)


def _extract_redirect_url(page):
    url = re.search('method="post" action="([^"]+)"', page.content.decode("utf-8"))
    if url is None:
        raise LoginError("could not extract redirect url")
    return url.group(1)


def _extract_auth_state(page):
    auth_state = re.search(
        'name="AuthState" value="([^"]+)"', page.content.decode("utf-8")
    )
    if not auth_state:
        raise LoginError("could not extract auth state")
    return auth_state.group(1)


def _extract_access_token(page):
    token = re.search(
        '(<h2>Token endpoint response:</h2>\n<pre style="white-space: pre-wrap; word-wrap: break-word;">)'
        + "(.*)(</pre><br>\n<h2>Encoded access Token:</h2>)",
        page.content.decode("utf-8"),
        flags=re.DOTALL,
    )

    if token is None:
        raise LoginError("could not extract token\n%s" % page.text)

    token = token.group(2)
    token = html.unescape(token)
    token = token.replace("'", '"')
    token = json.loads(token)

    return token


def check_page(page):
    if page is None:
        raise LoginError("page is None")


def _extract_pseudonym(page):
    pseudonym = re.search("(<p id=pseudonym>)(.*)(</p>)", page.content.decode("utf-8"))
    if not pseudonym:
        raise LoginError("could not extract auth state")
    return pseudonym.group(2)


def get_access_token_and_pseudonym(username, password, idp_hint, sess=None):
    if sess is None:
        session = requests.Session()
    else:
        session = sess

    param = TESTAPP_PARAM.format(idp_hint=idp_hint)
    url = urljoin(TESTAPP_URL, param)

    if sess:
        page = session.get(url, verify=False, name=TESTAPP_PARAM)
    else:
        page = session.get(url, verify=False)

    check_page(page)

    try:
        saml_msg = _extract_saml_msg(page)
    except LoginError as exc:  # user already logged in?
        try:
            token = _extract_access_token(page)
            pseudonym = _extract_pseudonym(page)
            return token, pseudonym
        except LoginError:
            raise exc

    relay_state = _extract_relay_state(page)
    redirect_url = _extract_redirect_url(page)

    if sess:
        page = session.post(
            redirect_url,
            data={"SAMLResponse": saml_msg, "RelayState": relay_state},
            verify=False,
            name="/simplesamlphp/module.php/core/loginuserpass.php-1",
        )
    else:
        page = session.post(
            redirect_url,
            data={"SAMLResponse": saml_msg, "RelayState": relay_state},
            verify=False,
        )

    check_page(page)

    auth_state = _extract_auth_state(page)
    data = {"username": username, "password": password, "AuthState": auth_state}

    if sess:
        page = session.post(
            page.url,
            data=data,
            verify=False,
            name="/simplesamlphp/module.php/core/loginuserpass.php-2",
        )
    else:
        page = session.post(page.url, data=data, verify=False)

    check_page(page)

    kc_url = _extract_kc_url(page)
    saml_msg = _extract_saml_response(page)
    relay_state = _extract_relay_state(page)

    if sess:
        page = session.post(
            kc_url,
            data={"SAMLResponse": saml_msg, "RelayState": relay_state},
            name="/auth/realms/ID-Broker/broker/{traeger}/endpoint",
        )
    else:
        page = session.post(
            kc_url, data={"SAMLResponse": saml_msg, "RelayState": relay_state}
        )

    check_page(page)

    token = _extract_access_token(page)
    pseudonym = _extract_pseudonym(page)

    return token, pseudonym


def refresh_token(token):
    url = f"https://{UCS_ENV_KEYCLOAK_FQDN}/auth/realms/ID-Broker/protocol/openid-connect/token"
    data = {
        "client_id": "univention-test-app",
        "client_secret": UCS_ENV_TEST_APP_SECRET,
        "grant_type": "refresh_token",
        "refresh_token": token["refresh_token"],
    }
    page = requests.post(url, data=data, verify=False)
    try:
        token = json.loads(page.text)
        token["access_token"]
    except json.JSONDecodeError as exc:
        raise LoginError("could not decode token") from exc
    return token
