import logging
import random
from typing import Dict, List, Tuple

from faker import Faker
from locust import events, task

from performance_tests.common import (
    ProvisioningClient,
    ProvisioningSchoolContext,
    ResponseError,
    TestData,
    UsedSchools,
    get_member_data,
    get_provisioning_api_admins,
    get_role_data,
    get_users,
)

school_authorities: List[str] = []  # ["Traeger1", "Traeger"]
user_credentials: Dict[str, Tuple[str, str]] = {}  # [(username, password), ..]
id_broker_school_users: Dict[
    str, Dict[str, Dict[str, str]]
] = {}  # {sch_auth: {ou: {uid: uuid, ...}}}
# {sch_auth: {ou: {group: [uid, ...]}}}:
id_broker_class_members: Dict[str, Dict[str, Dict[str, List[str]]]] = {}
# {sch_auth: {ou: {uid: ["student"]}}}:
id_broker_user_roles: Dict[str, Dict[str, Dict[str, List[str]]]] = {}
fake = Faker()
NUM_SCHOOLS_PER_SCHOOL_AUTHORITY = 5  # loading all OUs from test DB takes to long
used_schools = UsedSchools()


@events.init.add_listener
def on_init(environment, **kwargs):
    data = TestData()
    school_authorities.extend(data.school_authorities)
    user_credentials.update(get_provisioning_api_admins(data))
    id_broker_school_users.update(get_users(data, NUM_SCHOOLS_PER_SCHOOL_AUTHORITY))
    schools = set()
    for school_authority in school_authorities:
        schools.update(id_broker_school_users[school_authority].keys())
    id_broker_class_members.update(get_member_data(data, schools))
    id_broker_user_roles.update(get_role_data(data, schools))


class ProvisioningAPIClient(ProvisioningClient):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.school_authority = random.choice(school_authorities)
        logging.info("Testing with school authority: %r", self.school_authority)
        self.auth_username, self.auth_password = user_credentials[self.school_authority]
        self.schools = list(id_broker_school_users[self.school_authority].keys())
        users = sum(
            len(v) for v in id_broker_school_users[self.school_authority].values()
        )
        logging.info(
            "Testing with schools: %r incl. %d users.", sorted(self.schools), users
        )

    @task
    def user_update(self) -> None:
        school = random.choice(self.schools)
        used_schools.add(self.school_authority, school)
        users = id_broker_school_users[self.school_authority][school]
        username, entry_uuid = random.choice(list(users.items()))
        classes = [
            class_name
            for class_name, members in id_broker_class_members[self.school_authority][
                school
            ].items()
            if username in members
        ]
        roles = id_broker_user_roles[self.school_authority][school][username]
        try:
            super().user_update(
                school_authority=self.school_authority,
                obj_id=entry_uuid,
                first_name=fake.first_name(),
                last_name=fake.last_name(),
                user_name=username,
                context={
                    school: ProvisioningSchoolContext(classes=classes, roles=roles)
                },
            )
        except ResponseError as exc:
            logging.error(
                "%s: school_authority=%r name=%r entry_uuid=%r -> %r",
                self.greenlet.name,
                self.school_authority,
                username,
                entry_uuid,
                exc,
            )


@events.test_stop.add_listener
def _(environment, **_kwargs):
    logging.info(
        "Used schools: %r", sorted(f"{s_a}-{ou}" for s_a, ou in used_schools.schools)
    )
    used_schools.persist()
