import logging
import random
from typing import Dict, List, Tuple

from faker import Faker
from locust import events, task

from performance_tests.common import (
    ProvisioningClient,
    ResponseError,
    TestData,
    UsedSchools,
    get_provisioning_api_admins,
)

school_authorities: List[str] = []  # ["Traeger1", "Traeger"]
user_credentials: Dict[str, Tuple[str, str]] = {}  # [(username, password), ..]
fake = Faker()
used_schools = UsedSchools()


@events.init.add_listener
def on_init(environment, **kwargs):
    data = TestData()
    school_authorities.extend(data.school_authorities)
    user_credentials.update(get_provisioning_api_admins(data))


class ProvisioningAPIClient(ProvisioningClient):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.school_authority = random.choice(school_authorities)
        logging.info("Testing with school authority: %r", self.school_authority)
        self.auth_username, self.auth_password = user_credentials[self.school_authority]

    @task
    def school_create(self) -> None:
        name = f"test_{fake.user_name()[:10]}{fake.pyint(1000, 9999)}"
        used_schools.add(self.school_authority, name)
        entry_uuid = fake.uuid4()
        try:
            super().school_create(
                school_authority=self.school_authority,
                obj_id=entry_uuid,
                name=name,
                display_name="Performance Test temp OU",
            )
        except ResponseError as exc:
            logging.error(
                "%s: school_authority=%r name=%r entry_uuid=%r -> %r",
                self.greenlet.name,
                self.school_authority,
                name,
                entry_uuid,
                exc,
            )


@events.test_stop.add_listener
def _(environment, **_kwargs):
    logging.info(
        "Used schools: %r", sorted(f"{s_a}-{ou}" for s_a, ou in used_schools.schools)
    )
    used_schools.persist()
