# -*- coding: utf-8 -*-
# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.
import json
import os.path
import subprocess

import pytest
from tenacity import (
    AsyncRetrying,
    retry_if_exception_type,
    stop_after_attempt,
    wait_fixed,
)

import ucsschool_id_connector.plugin_loader

# load ID Broker plugin
ucsschool_id_connector.plugin_loader.load_plugins()
id_broker = pytest.importorskip("idbroker")
pytestmark = pytest.mark.id_broker

from idbroker.initial_sync import CONFIG_PATH as SCHOOL_CONFIG_PATH, get_ous  # isort:skip  # noqa: E402

SCRIPT_PATH = (
    "/var/lib/univention-appcenter/apps/ucsschool-id-connector/conf/"
    "plugins/packages/idbroker/manage_schools_to_sync.py"
)


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "schools_arg",
    ("all_schools", "school_name", "partial_school_name", "*"),
    ids=lambda x: f"schools_arg-{x}",
)
async def test_add_schools(
    make_sender_user_special,
    kelvin_school_on_sender,
    make_kelvin_workgroup_on_id_connector,
    make_kelvin_school_class_on_id_connector,
    wait_for_kelvin_user_on_id_broker,
    wait_for_kelvin_school_class_on_id_broker,
    wait_for_kelvin_workgroup_on_id_broker,
    schools_arg,
    schedule_schools_delete,
    delete_school_on_id_broker,
    set_school_authority_on_id_connector,
):
    # create school with users & groups
    # remove users & groups from id broker
    # add school
    schools_on_sender = [school.name for school in await get_ous("*")]
    if schools_arg == "all_schools":
        school_filter = schools_on_sender
    elif schools_arg == "school_name":
        school_filter = [kelvin_school_on_sender.name]
    elif schools_arg == "partial_school_name":
        school_filter = [
            f"{kelvin_school_on_sender.name[0]}*{kelvin_school_on_sender.name[2:]}"
        ]
    elif schools_arg == "*":
        school_filter = ["*"]
    else:
        raise ValueError(f"Invalid schools_arg {schools_arg}")
    school_authority = await set_school_authority_on_id_connector(schools=school_filter)
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    sender_user = await make_sender_user_special(
        ous=[kelvin_school_on_sender.name],
    )
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=sender_user["name"],
    )
    school_class = await make_kelvin_school_class_on_id_connector(
        school_name=kelvin_school_on_sender.name, users=[sender_user["name"]]
    )
    await wait_for_kelvin_school_class_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=school_class.name,
    )
    work_group = await make_kelvin_workgroup_on_id_connector(
        school_name=kelvin_school_on_sender.name, users=[sender_user["name"]]
    )
    await wait_for_kelvin_workgroup_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=work_group.name,
    )
    await delete_school_on_id_broker(
        config=school_authority, schools=[kelvin_school_on_sender.name]
    )
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=sender_user["name"],
        must_exist=False,
    )
    await wait_for_kelvin_school_class_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=school_class.name,
        must_exist=False,
    )
    await wait_for_kelvin_workgroup_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=work_group.name,
        must_exist=False,
    )
    add_school_args = [
        SCRIPT_PATH,
        "add_schools",
        "--num_tasks",
        "5",
        "--school_authority",
        school_authority.name,
    ]
    if schools_arg == "all_schools":
        add_school_args.append("--all_schools")
    elif schools_arg == "school_name":
        add_school_args.append(kelvin_school_on_sender.name)
    elif schools_arg == "partial_school_name":
        add_school_args.append(
            f"{kelvin_school_on_sender.name[0]}*{kelvin_school_on_sender.name[2:]}"
        )
    elif schools_arg == "*":
        add_school_args.append("*")
    else:
        raise ValueError(f"Unknown schools_arg {schools_arg}")
    # should fail, because there we already added the same school in set_school_authority_on_id_connector
    with pytest.raises(subprocess.CalledProcessError):
        subprocess.check_call(add_school_args)
    output = subprocess.check_output(
        add_school_args
        + [
            "--force",
        ]
    )
    print(output)
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=sender_user["name"],
    )
    async for attempt in AsyncRetrying(
        reraise=True,
        stop=stop_after_attempt(3),
        retry=retry_if_exception_type(AssertionError),
        wait=wait_fixed(5),
    ):
        with attempt:
            sc = await wait_for_kelvin_school_class_on_id_broker(
                s_a_name=school_authority.name,
                school=kelvin_school_on_sender.name,
                name=school_class.name,
            )
            assert sc.users == [f"{school_authority.name}-{sender_user['name']}"]
    wg = await wait_for_kelvin_workgroup_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=work_group.name,
    )
    assert wg.users == [f"{school_authority.name}-{sender_user['name']}"]
    with open(os.path.join(SCHOOL_CONFIG_PATH, f"{school_authority.name}.json")) as fin:
        config = json.load(fin)
        assert sorted(school_filter) == sorted(
            config["plugin_configs"]["id_broker"]["schools"]
        )


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "schools_arg",
    ("all_schools", "school_name", "partial_school_name", "*"),
    ids=lambda x: f"schools_arg-{x}",
)
async def test_remove_schools(
    make_sender_user_special,
    kelvin_school_on_sender,
    make_kelvin_workgroup_on_id_connector,
    make_kelvin_school_class_on_id_connector,
    wait_for_kelvin_user_on_id_broker,
    wait_for_kelvin_school_class_on_id_broker,
    wait_for_kelvin_workgroup_on_id_broker,
    schools_arg,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    schools_on_sender = [school.name for school in await get_ous("*")]
    if schools_arg == "all_schools":
        school_filter = schools_on_sender
    elif schools_arg == "school_name":
        school_filter = [kelvin_school_on_sender.name]
    elif schools_arg == "partial_school_name":
        school_filter = [
            f"{kelvin_school_on_sender.name[0]}*{kelvin_school_on_sender.name[2:]}"
        ]
    elif schools_arg == "*":
        school_filter = ["*"]
    else:
        raise ValueError(f"Invalid schools_arg {schools_arg}")
    school_authority = await set_school_authority_on_id_connector(schools=school_filter)
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    work_group = await make_kelvin_workgroup_on_id_connector(
        school_name=kelvin_school_on_sender.name
    )
    await wait_for_kelvin_workgroup_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=work_group.name,
    )
    user = await make_sender_user_special(
        ous=[kelvin_school_on_sender.name],
    )
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=user["name"],
    )
    school_class = await make_kelvin_school_class_on_id_connector(
        school_name=kelvin_school_on_sender.name, users=[user["name"]]
    )
    await wait_for_kelvin_school_class_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=school_class.name,
    )
    remove_school_args = [
        SCRIPT_PATH,
        "remove_schools",
        "--school_authority",
        school_authority.name,
    ]
    if schools_arg == "all_schools":
        remove_school_args.append("--all_schools")
    elif schools_arg == "school_name":
        remove_school_args.append(kelvin_school_on_sender.name)
    elif schools_arg == "partial_school_name":
        remove_school_args.append(
            f"{kelvin_school_on_sender.name[0]}*{kelvin_school_on_sender.name[2:]}"
        )
    elif schools_arg == "*":
        remove_school_args.append("*")
    else:
        raise ValueError(f"Unknown schools_arg {schools_arg}")
    subprocess.check_output(remove_school_args)
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=user["name"],
        must_exist=False,
    )
    await wait_for_kelvin_school_class_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=school_class.name,
        must_exist=False,
    )
    await wait_for_kelvin_workgroup_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=work_group.name,
        must_exist=False,
    )
    with open(os.path.join(SCHOOL_CONFIG_PATH, f"{school_authority.name}.json")) as fin:
        config = json.load(fin)
        for entry in school_filter:
            assert entry not in config["plugin_configs"]["id_broker"]["schools"]
    with pytest.raises(subprocess.CalledProcessError) as exc:
        subprocess.check_call(remove_school_args)
        assert exc.code == 1


@pytest.mark.asyncio
async def test_remove_schools_skip_configured(
    id_broker_school_auth_conf,
    make_school_authority,
    make_sender_user_special,
    kelvin_schools_on_sender,
    make_kelvin_workgroup_on_id_connector,
    make_kelvin_school_class_on_id_connector,
    set_school_authority_on_id_connector,
    wait_for_kelvin_user_on_id_broker,
    wait_for_kelvin_school_class_on_id_broker,
    wait_for_kelvin_workgroup_on_id_broker,
    schedule_schools_delete,
):
    school1, school2 = await kelvin_schools_on_sender(2)
    school_authority = await set_school_authority_on_id_connector(
        schools=[school1.name, "*"]
    )
    await schedule_schools_delete(
        school_authority=school_authority, schools=[school1.name, school2.name]
    )
    user1 = await make_sender_user_special(
        ous=[school1.name],
    )
    user2 = await make_sender_user_special(
        ous=[school2.name],
    )
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=school_authority.name, school=school1.name, name=user1["name"]
    )
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=school_authority.name, school=school2.name, name=user2["name"]
    )
    remove_school_args = [
        SCRIPT_PATH,
        "remove_schools",
        "--school_authority",
        school_authority.name,
        "*",
    ]
    output = subprocess.check_output(remove_school_args)
    print(output)
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=school_authority.name,
        school=school2.name,
        name=user2["name"],
        must_exist=False,
    )
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=school_authority.name,
        school=school1.name,
        name=user1["name"],
        must_exist=True,
    )
    with open(os.path.join(SCHOOL_CONFIG_PATH, f"{school_authority.name}.json")) as fin:
        config = json.load(fin)
        assert "*" not in config["plugin_configs"]["id_broker"]["schools"]
        assert school1.name in config["plugin_configs"]["id_broker"]["schools"]
    with pytest.raises(subprocess.CalledProcessError) as exc:
        subprocess.check_call(remove_school_args)
        assert exc.code == 1


@pytest.mark.asyncio
async def test_remove_schools_dont_remove_schools_on_id_broker(
    make_sender_user_special,
    kelvin_school_on_sender,
    make_kelvin_workgroup_on_id_connector,
    make_kelvin_school_class_on_id_connector,
    wait_for_kelvin_user_on_id_broker,
    wait_for_kelvin_school_class_on_id_broker,
    wait_for_kelvin_workgroup_on_id_broker,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    school_authority = await set_school_authority_on_id_connector(
        schools=[kelvin_school_on_sender.name]
    )
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    work_group = await make_kelvin_workgroup_on_id_connector(
        school_name=kelvin_school_on_sender.name
    )
    user = await make_sender_user_special(
        ous=[kelvin_school_on_sender.name],
    )
    school_class = await make_kelvin_school_class_on_id_connector(
        school_name=kelvin_school_on_sender.name, users=[user["name"]]
    )
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=user["name"],
    )
    await wait_for_kelvin_school_class_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=school_class.name,
    )
    await wait_for_kelvin_workgroup_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=work_group.name,
    )
    remove_school_args = [
        SCRIPT_PATH,
        "remove_schools",
        "--school_authority",
        school_authority.name,
        "--delete_schools",
        "false",
        kelvin_school_on_sender.name,
    ]
    output = subprocess.check_output(remove_school_args)
    print(output)
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=user["name"],
    )
    await wait_for_kelvin_school_class_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=school_class.name,
    )
    await wait_for_kelvin_workgroup_on_id_broker(
        s_a_name=school_authority.name,
        school=kelvin_school_on_sender.name,
        name=work_group.name,
    )
    with open(os.path.join(SCHOOL_CONFIG_PATH, f"{school_authority.name}.json")) as fin:
        config = json.load(fin)
        assert (
            kelvin_school_on_sender.name
            not in config["plugin_configs"]["id_broker"]["schools"]
        )


@pytest.mark.asyncio
@pytest.mark.parametrize("all_schools", (True, False), ids=lambda x: f"all_schools-{x}")
async def test_configure_schools(
    kelvin_school_on_sender,
    all_schools,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    school_authority = await set_school_authority_on_id_connector(schools=[])
    schools_on_sender = None
    if all_schools:
        schools_on_sender = [school.name for school in await get_ous("*")]
        configure_all_school_args = [
            SCRIPT_PATH,
            "add_schools",
            "--school_authority",
            school_authority.name,
            "--initial_sync",
            "false",
            "--all_schools",
        ]
    else:
        configure_all_school_args = [
            SCRIPT_PATH,
            "add_schools",
            "--school_authority",
            school_authority.name,
            "--initial_sync",
            "false",
            kelvin_school_on_sender.name,
        ]
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    output = subprocess.check_output(configure_all_school_args)
    print(output)
    with open(os.path.join(SCHOOL_CONFIG_PATH, f"{school_authority.name}.json")) as fin:
        config = json.load(fin)
        if all_schools:
            assert set(schools_on_sender) == set(
                config["plugin_configs"]["id_broker"]["schools"]
            )
        else:
            assert [kelvin_school_on_sender.name] == config["plugin_configs"][
                "id_broker"
            ]["schools"]
    # -> should fail, because there we try to add the same schools again.
    with pytest.raises(subprocess.CalledProcessError):
        subprocess.check_call(configure_all_school_args)
    # -> should pass with option force
    output = subprocess.check_output(
        configure_all_school_args
        + [
            "--force",
        ]
    )
    print(output)
