#!/usr/share/ucs-test/runner pytest-3 -s -l -v
## -*- coding: utf-8 -*-
## desc: Check traeger users are synced into the broker domain
## tags: [ucsschool-id-broker]
## packages: [id-broker-provisioning-api-plugin]
## exposure: safe

import operator
import os

import pytest
from tenacity import retry, stop_after_attempt, wait_fixed
from utils import get_user_from_traeger

from univention.id_broker.pseudonyms import generate_pseudonym
from univention.id_broker.pseudonyms_udm_host import (
    get_service_provider_attribute,
    get_service_provider_secret,
)


def assert_users(broker_user, traeger_user, traeger, service_providers):
    assert broker_user.props.firstname == traeger_user.firstname
    assert broker_user.props.lastname == traeger_user.lastname
    assert broker_user.props.displayName == traeger_user.displayName
    assert broker_user.props.gecos == traeger_user.displayName
    assert broker_user.props.ucsschoolSourceUID == traeger
    assert broker_user.props.ucsschoolRecordUID == traeger_user.entryUUID
    assert set(
        [role.replace(f":{traeger}-", ":") for role in broker_user.props.ucsschoolRole]
    ) == set(traeger_user.ucsschoolRole)
    for provider in service_providers:
        attribute = get_service_provider_attribute(provider)
        secret = get_service_provider_secret(provider)
        pseudonym = getattr(broker_user.props, attribute)
        expected_pseudonym = generate_pseudonym(
            secret, broker_user.props.ucsschoolRecordUID, traeger
        )
        assert pseudonym == expected_pseudonym


@pytest.mark.parametrize(
    "uid",
    [
        "Traeger1-stud1",
        "Traeger1-stud2",
        "Traeger1-stud3",
        "Traeger1-teach1",
        "Traeger1-teach2",
        "Traeger1-teach3",
        "Traeger2-stud1",
        "Traeger2-stud2",
        "Traeger2-stud3",
        "Traeger2-teach1",
        "Traeger2-teach2",
        "Traeger2-teach3",
        "Traeger2-parent1",
    ],
)
@retry(wait=wait_fixed(5), stop=stop_after_attempt(6))
def test_setup_users_are_synced(udm, uid, service_providers):
    assert os.environ.get("UCS_ENV_TRAEGER1_DOMAIN")
    assert os.environ.get("UCS_ENV_TRAEGER2_DOMAIN")
    traeger = uid.split("-")[0]
    udm_mod = udm.get("users/user")
    broker_user = list(udm_mod.search(f"uid={uid}"))
    assert len(broker_user) == 1, f"user {uid} not found in broker domain"
    broker_user = broker_user[0]
    traeger_user = list(
        get_user_from_traeger(
            traeger, f"(entryUUID={broker_user.props.ucsschoolRecordUID})"
        )
    )[0]
    assert_users(broker_user, traeger_user, traeger, service_providers)


@pytest.mark.parametrize(
    "traeger",
    [
        "Traeger1",
        "Traeger2",
    ],
)
@retry(wait=wait_fixed(5), stop=stop_after_attempt(6))
def test_all_traeger_users_are_synced(udm, traeger, service_providers):
    assert os.environ.get("UCS_ENV_TRAEGER1_DOMAIN")
    assert os.environ.get("UCS_ENV_TRAEGER2_DOMAIN")
    user_mod = udm.get("users/user")
    broker_users = list(user_mod.search(f"uid={traeger}-*"))
    traeger_users = list(
        get_user_from_traeger(
            traeger,
            "(&(ucsschoolRole=*)(objectClass=ucsschoolType)(uid=*)(!(givenName=Demo)))",
        )
    )
    # sort lists to reduce runtime-costs
    traeger_users = sorted(traeger_users, key=operator.attrgetter("entryUUID"))
    broker_users = sorted(
        broker_users, key=operator.attrgetter("props.ucsschoolRecordUID")
    )
    assert len(broker_users) == len(traeger_users)
    for index, (broker_user, traeger_user) in enumerate(
        zip(broker_users, traeger_users)
    ):
        assert_users(broker_user, traeger_user, traeger, service_providers)
