from typing import List

from fastapi import APIRouter

from id_broker_common.kelvin import kelvin_session_kwargs, load_kelvin_session_kwargs
from id_broker_common.utils import load_settings, remove_passwords
from ucsschool.apis.models import Plugin
from ucsschool.apis.utils import auth_manager, get_logger

from .routes.v1 import (
    school_classes as school_classes_routes_v1,
    schools as schools_routes_v1,
    users as users_routes_v1,
    workgroups as workgroups_v1,
)
from .settings import PLUGIN_SETTINGS_FILE

# the name of the plugin which will also determine the name of the resource
PLUGIN_NAME: str = "provisioning"
# the plugins current version. Not used for anything in the ucsschool-apis app
PLUGIN_VERSION: str = "1.3.0"
# the plugins tags:
PLUGIN_TAGS: List[str] = ["provisioning"]
# the router that will be mounted as the resource under the plugins name:
router: APIRouter = APIRouter()


def setup():
    logger = get_logger()
    logger.info(f"Setup of {PLUGIN_NAME} with version {PLUGIN_VERSION}")
    # load config now to raise exception about errors as early as possible
    settings = load_settings(PLUGIN_SETTINGS_FILE)
    logger.info(
        "Settings in '%s': %r", PLUGIN_SETTINGS_FILE, remove_passwords(settings)
    )
    kelvin_session_kwargs.update(load_kelvin_session_kwargs(PLUGIN_SETTINGS_FILE))
    auth_manager.tags[PLUGIN_NAME] = "Auth tag for all provisioning API endpoints"


router.include_router(users_routes_v1.router, prefix="/v1/{school_authority}")
router.include_router(schools_routes_v1.router, prefix="/v1/{school_authority}")
router.include_router(school_classes_routes_v1.router, prefix="/v1/{school_authority}")
router.include_router(workgroups_v1.router, prefix="/v1/{school_authority}")

# This is the object that is referenced in the pyproject.toml as the plugin object:
ProvisioningPlugin = Plugin(PLUGIN_NAME, PLUGIN_VERSION, PLUGIN_TAGS, router, setup)
