# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.

test_kelvin_and_uas_apis_apps_installed () {
  if  ! univention-app status ucsschool-kelvin-rest-api 1>/dev/null 2>/dev/null  ||  ! univention-app status ucsschool-apis 1>/dev/null 2>/dev/null; then
    echo "ERROR: -------------------------------------------------------------------------"
    echo "ERROR: The apps 'UCS@school Kelvin Rest API' (ucsschool-kelvin-rest-api) and"
    echo "ERROR: 'UCS@school APIs' (ucsschool-apis) have to be installed."
    echo "ERROR: Please install the apps manually:"
    echo "ERROR: root@$(hostname -s):~# univention-app install ucsschool-kelvin-rest-api"
    echo "ERROR: root@$(hostname -s):~# univention-app install ucsschool-apis"
    echo "ERROR: -------------------------------------------------------------------------"
    return 1
  fi
  return 0
}

test_kelvin_app_installed () {
  if  ! univention-app status ucsschool-kelvin-rest-api 1>/dev/null 2>/dev/null; then
    echo "ERROR: -------------------------------------------------------------------------"
    echo "ERROR: The app 'UCS@school Kelvin Rest API' (ucsschool-kelvin-rest-api) has"
    echo "ERROR: to be installed."
    echo "ERROR: Please install the app manually:"
    echo "ERROR: root@$(hostname -s):~# univention-app install ucsschool-kelvin-rest-api"
    echo "ERROR: -------------------------------------------------------------------------"
    return 1
  fi
  return 0
}

test_uas_apis_apps_installed () {
  if  ! univention-app status ucsschool-apis 1>/dev/null 2>/dev/null; then
    echo "ERROR: -------------------------------------------------------------------------"
    echo "ERROR: The app 'UCS@school APIs' (ucsschool-apis) has to be installed."
    echo "ERROR: Please install the app manually:"
    echo "ERROR: root@$(hostname -s):~# univention-app install ucsschool-apis"
    echo "ERROR: -------------------------------------------------------------------------"
    return 1
  fi
  return 0
}

add_prop_to_kelvin_mapped_udm_properties () {
  local cls="${1:-}" prop="${2:-}" path="${3:-}"
  python3 -c "
import json
conf = json.load(open('$path'))
if '$cls' not in conf:
  conf['$cls'] = ['$prop']
else:
  conf_cls = set(conf['$cls'])
  conf_cls.add('$prop')
  conf['$cls'] = sorted(conf_cls)
json.dump(conf, open('$path', 'w'), indent=4, sort_keys=True)
"
}

prop_in_kelvin_mapped_udm_properties () {
  local cls="${1:-}" prop="${2:-}" path="${3:-}"
  python3 -c "
import json, sys
conf = json.load(open('$path'))
sys.exit(int('$prop' not in conf.get('$cls', [])))
"
}

add_prop_to_kelvin_mapped_udm_properties_if_missing () {
  local cls="${1:-}" prop="${2:-}" path="${3:-}"
  if prop_in_kelvin_mapped_udm_properties "$cls" "$prop" "$path";
  then
    echo "Property '$prop' for '$cls' already in Kelvin REST APIs configuration file '$path'."
  else
    echo "Adding '$prop' for '$cls' to Kelvin REST APIs configuration file '$path'..."
    add_prop_to_kelvin_mapped_udm_properties "$cls" "$prop" "$path" || return 1
  fi
}

add_all_kelvin_mapped_udm_properties_if_missing () {
  local KELVIN_MAPPED_CONFIG=/etc/ucsschool/kelvin/mapped_udm_properties.json

for cls in "school" "school_class" "workgroup"; do
    add_prop_to_kelvin_mapped_udm_properties_if_missing "$cls" "ucsschoolRecordUID" "$KELVIN_MAPPED_CONFIG" || die
    add_prop_to_kelvin_mapped_udm_properties_if_missing "$cls" "ucsschoolSourceUID" "$KELVIN_MAPPED_CONFIG" || die
  done
for cls in "school" "school_class" "workgroup" "user"; do
    for index in $(seq -w 30); do
      add_prop_to_kelvin_mapped_udm_properties_if_missing "$cls" "idBrokerPseudonym00${index}" "$KELVIN_MAPPED_CONFIG" || die
    done
  done
}

plugin_enabled_in_ucsschool_apis () {
  local plugin="${1:-}" path="${2:-}"
  python3 -c "
import json, sys
conf = json.load(open('$path'))
sys.exit(int('$plugin' not in conf.get('enabled_plugins', [])))
"
}

enable_plugin_in_ucsschool_apis () {
  local plugin="${1:-}" path="${2:-}"
  python3 -c "
import json
conf = json.load(open('$path'))
conf_plugins = set(conf.get('enabled_plugins', []))
conf_plugins.add('$plugin')
conf['enabled_plugins'] = sorted(conf_plugins)
json.dump(conf, open('$path', 'w'), indent=4, sort_keys=True)
"
}

idp_added_in_ucsschool_apis () {
  local idp_name="${1:-}" path="${2:-}"
  python3 -c "
import json, sys
with open('$path') as fd:
  conf = json.load(fd)
sys.exit(int('$idp_name' not in conf.get('auth', {}).get('idps', {})))
"
}

add_idp_in_ucsschool_apis() {
  local idp_name="${1:-}" idp_value="${2:-}" path="${3:-}"
  python3 -c "
import json
idp_conf = json.loads('$idp_value')
with open('$path', 'r') as fd:
  conf = json.load(fd)
auth_conf = conf.get('auth', {})
auth_conf['idps'] = auth_conf.get('idps', {})
auth_conf['idps']['$idp_name'] = idp_conf
conf['auth'] = auth_conf
with open('$path', 'w') as fd:
  json.dump(conf, fd, indent=4, sort_keys=True)
"
}

add_auth_tag_mapping_in_ucsschool_apis() {
  local tag="${1:-}" idp_name="${2:-}" path="${3:-}"
  python3 -c "
import json
with open('$path', 'r') as fd:
  conf = json.load(fd)
auth_conf = conf.get('auth', {})
auth_conf['tag_idp_mapping'] = auth_conf.get('tag_idp_mapping', {})
current_idps = set(auth_conf['tag_idp_mapping'].get('$tag', []))
current_idps.add('$idp_name')
auth_conf['tag_idp_mapping']['$tag'] = list(current_idps)
conf['auth'] = auth_conf
with open('$path', 'w') as fd:
  json.dump(conf, fd, indent=4, sort_keys=True)
"
}

enable_plugin_in_ucsschool_apis_if_missing () {
  local plugin="${1:-}" path="${2:-}"
  if plugin_enabled_in_ucsschool_apis "$plugin" "$path";
  then
    echo "Plugin '$plugin' already active in UCS@school APIs configuration file '$path'."
  else
    echo "Adding '$plugin' to UCS@school APIs configuration file '$path'..."
    enable_plugin_in_ucsschool_apis "$plugin" "$path" || return 1
  fi
}

restart_ucsschool_apis_app () {
  echo "Restarting UCS@school APIs app..."
  univention-app shell ucsschool-apis sh -c 'for cmd in stop zap start; do echo "# ucsschool-apis $cmd"; /etc/init.d/ucsschool-apis $cmd; done'
}

write_secret () {
  local secret="${1:-}" filename="${2:-}"
  mkdir -p "$(dirname $filename)"
  touch "$filename"
  chmod 0600 "$filename"
  chown root:root "$filename"
  echo "$secret" >"$filename"
}

create_empty_settings_file_if_missing () {
  local path="${1:-}"
  if [ ! -f "$path" ]; then
    echo "Creating empty settings file '$path'..."
    mkdir -p -m 750 -v "$(dirname $path)"
    echo "{}" > "$path"
    chown root:root "$path"
    chmod 640 "$path"
  fi
}

set_max_username_length_for_kelvin () {
  local num="${1:-}"
  univention-app shell ucsschool-kelvin-rest-api ucr set "ucsschool/username/max_length=$num"
}

restart_kelvin () {
  echo "Restarting UCS@school Kelvin REST API app..."
  univention-app restart ucsschool-kelvin-rest-api
}

create_ext_attr_for_pseudonym () {
  local attr_name="${1:-}" sp_name="${2:-}" udm_module="${3:-}" tab_position="${4:-}"
  shift; shift; shift; shift
  module_word="$(echo "$udm_module" | cut -d '/' -f 2)"

  univention-directory-manager settings/extended_attribute create "$@" --ignore_exists \
      --position "cn=IDBroker,cn=custom attributes,cn=univention,$ldap_base" \
      --set name="${attr_name}${module_word}" \
      --set CLIName="$attr_name" \
      --set module="$udm_module" \
      --set tabName="ID Broker" \
      --set tabPosition="$tab_position" \
      --set groupName="Pseudonyms" \
      --set translationGroupName='"de_DE" "Pseudonyme"' \
      --set groupPosition="2" \
      --set shortDescription="ID Broker pseudonym for ${sp_name}." \
      --set longDescription="ID Broker pseudonym for ${sp_name}." \
      --set translationShortDescription="\"de_DE\" \"ID Broker Pseudonym für ${sp_name}.\"" \
      --set translationLongDescription="\"de_DE\" \"ID Broker Pseudonym für ${sp_name}.\"" \
      --set syntax=ascii \
      --set mayChange=1 \
      --set multivalue=0 \
      --set ldapMapping="$attr_name" \
      --set objectClass="idBrokerPseudonymsType" || die
}

add_indices_for_attrs () {
  local last_attr="$(echo $@ | awk '{print $NF}')"

  if [ -x /etc/init.d/slapd ] &&
     [ -x /usr/share/univention-ldap/ldap_setup_index ] &&
     [ -e /var/univention-join/joined ] &&
     ! ucr get ldap/index/eq | tr , '\n' | grep -Fxq "$last_attr" &&
     univention-ldapsearch -LLL -o ldif-wrap=no -b cn=Subschema -s base attributeTypes | grep -Fq "NAME '$last_attr'"
  then
    echo "Updating LDAP indices..."
    for attr_name in $@; do
      index_args="$index_args --add-eq ${attr_name}"
    done
    unset stop_and_start_slapd
    pidof slapd >/dev/null 2>/dev/null && stop_and_start_slapd=yes
    [ -n "$stop_and_start_slapd" ] && service slapd stop
    /usr/share/univention-ldap/ldap_setup_index --verbose $index_args || die
    [ -n "$stop_and_start_slapd" ] && service slapd start
    echo "Finished updating LDAP indices."
  else
    echo "Not updating LDAP indices (LDAP server not running or index exists)."
  fi
}

create_ext_attr_for_record_and_source_uid () {
  local attr_name="${1:-}" udm_module="${2:-}" tab_position="${3:-}" description_en="${4:-}" description_de="${5:-}"
  shift; shift; shift; shift; shift
  module_word="$(echo "$udm_module" | cut -d '/' -f 2)"

  univention-directory-manager settings/extended_attribute create "$@" --ignore_exists \
      --position "cn=IDBroker,cn=custom attributes,cn=univention,$ldap_base" \
      --set name="${attr_name}${module_word}" \
      --set CLIName="$attr_name" \
      --set module="$udm_module" \
      --set tabName="ID Broker" \
      --set tabPosition="$tab_position" \
      --set groupName="Origin" \
      --set translationGroupName='"de_DE" "Herkunft"' \
      --set groupPosition=1 \
      --set shortDescription="$description_en" \
      --set longDescription="$description_en" \
      --set translationShortDescription="\"de_DE\" \"$description_de\"" \
      --set translationLongDescription="\"de_DE\" \"$description_de\"" \
      --set syntax=ascii \
      --set mayChange=1 \
      --set multivalue=0 \
      --set ldapMapping="$attr_name" \
      --set objectClass="idBrokerOriginIds" \
      --set disableUDMWeb=0 || die
}

set_udm_rest_api_multi_process () {
  ucr set directory/manager/rest/processes=0
  systemctl restart univention-directory-manager-rest
}

set_kelvin_rest_api_multi_process () {
  univention-app configure ucsschool-kelvin-rest-api --set ucsschool/kelvin/processes=0
}

rebuild_udm_rest_client_of_kelvin () {
  systemctl restart univention-directory-manager-rest.service  # in case ext attr were installed
  sleep 10  # fixme: wait for the UDM REST API process to exist
  univention-app shell ucsschool-kelvin-rest-api pip --no-cache-dir install -U pip wheel
  # workaround for random failed uninstall when nothing is there to uninstall:
  univention-app shell ucsschool-kelvin-rest-api pip uninstall --yes openapi-client-udm
  univention-app shell ucsschool-kelvin-rest-api /var/lib/univention-appcenter/apps/ucsschool-kelvin-rest-api/data/update_openapi_client || die
}
