#!/usr/bin/env python3

# -*- coding: utf-8 -*-

# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.

import asyncio
import datetime
import re
import time
from functools import lru_cache
from typing import (
    Awaitable,
    Callable,
    Dict,
    Iterable,
    Iterator,
    List,
    Optional,
    Tuple,
    TypeVar,
    Union,
)

import click
from ldap3 import Entry
from ldap3.core.exceptions import LDAPInvalidFilterError

from ucsschool_id_connector.group_scheduler import GroupScheduler
from ucsschool_id_connector.ldap_access import LDAPAccess
from ucsschool_id_connector.school_scheduler import SchoolScheduler
from ucsschool_id_connector.user_scheduler import UserScheduler

try:
    # works in IDE, but not on the cli
    from config_utils import CONFIG_PATH, read_id_broker_config

    from .id_broker_client import (
        IDBrokerError,
        IDBrokerNotFoundError,
        IDBrokerObject,
        IDBrokerSchool,
        IDBrokerSchoolClass,
        IDBrokerUser,
        IDBrokerWorkGroup,
        ProvisioningAPIClient,
        School,
        SchoolClass,
        SchoolContext,
        User,
        WorkGroup,
    )
    from .utils import get_configured_schools, school_in_configured_schools
except ImportError:
    # works on the cli in ID Connector Docker container
    # make `idbroker` package importable
    import sys

    sys.path.append(
        "/var/lib/univention-appcenter/apps/ucsschool-id-connector/conf/plugins/packages"
    )

    from idbroker.config_utils import CONFIG_PATH, read_id_broker_config
    from idbroker.id_broker_client import (
        IDBrokerError,
        IDBrokerNotFoundError,
        IDBrokerObject,
        IDBrokerSchool,
        IDBrokerSchoolClass,
        IDBrokerUser,
        IDBrokerWorkGroup,
        ProvisioningAPIClient,
        School,
        SchoolClass,
        SchoolContext,
        User,
        WorkGroup,
    )
    from idbroker.utils import get_configured_schools, school_in_configured_schools


SCHOOL_ATTRS = ["entryUUID", "ou", "displayName"]
GROUP_ATTRS = ["entryUUID", "cn", "description", "uniqueMember"]
USER_ATTRS = [
    "entryUUID",
    "uid",
    "givenName",
    "sn",
    "memberOf",
    "ucsschoolRole",
    "ucsschoolLegalGuardian",
    "ucsschoolLegalWard",
]
SCHOOL_CLASS_DN_PATTERN = re.compile(
    r"^cn=(?P<cn>.+?),cn=klassen,cn=schueler,cn=groups,ou=(?P<ou>.+?),dc=.+$"
)
WORKGROUP_DN_PATTERN = re.compile(
    r"^cn=(?P<cn>.+?),cn=schueler,cn=groups,ou=(?P<ou>.+?),dc=.+$",
    flags=re.IGNORECASE,
)

PARALLELISM_DEFAULT = 4

_dn_to_id_cache: Dict[str, str] = {}

T = TypeVar("T")

ContextDictType = Dict[str, Dict[str, Iterable[str]]]


def dt_str() -> str:
    return datetime.datetime.now().isoformat(" ")


@click.group()
def cli():
    pass


@cli.command("users")
@click.argument("ldap_filter", required=False)
@click.option(
    "-p",
    "--parallelism",
    type=int,
    default=PARALLELISM_DEFAULT,
    show_default=True,
    help="Parallel requests.",
)
@click.option("-a", "--school_authority_name", required=True, type=str)
@click.option("--no-classes", is_flag=True, help="Do not compare classes.")
@click.option("--fix", is_flag=True, help="Queue objects that failed for resync")
def verify_users(
    ldap_filter: str,
    parallelism: int,
    no_classes: bool,
    school_authority_name: str,
    fix: bool,
) -> None:
    """
    Verify users:
    '(&(|(ucsschoolRole=teacher:*)(ucsschoolRole=student:*)(ucsschoolRole=legal_guardian:*))(LDAP_FILTER))'
    """
    role_filter = "(|(ucsschoolRole=teacher:*)(ucsschoolRole=student:*)(ucsschoolRole=legal_guardian:*))"
    if ldap_filter:
        ldap_filter = f"(&{role_filter}({ldap_filter}))"
    else:
        ldap_filter = role_filter
    try:
        ldap_objects: List[Entry] = asyncio.run(
            get_ldap_objects(ldap_filter, USER_ATTRS)
        )
    except LDAPInvalidFilterError:
        raise click.ClickException(
            f"Your LDAP filter is invalid, the complete filter that was used: {ldap_filter}"
        )
    ldap_objects.sort(key=lambda obj: obj.entry_dn)
    school_authority = read_id_broker_config(
        config_path=CONFIG_PATH, school_authority_name=school_authority_name
    )
    configured_schools = get_configured_schools(school_authority)
    user_objects = [
        user
        for user in asyncio.run(ldap_users_to_id_broker_users(ldap_objects))
        if any(
            school_in_configured_schools(school.lower(), configured_schools)
            for school in user.context.keys()
        )
    ]
    id_broker_user = IDBrokerUser(school_authority, "id_broker")
    asyncio.run(
        get_remote_objects_and_compare(
            user_objects,
            id_broker_user,
            Compare(school_authority, no_classes=no_classes, fix=fix).compare_user,
            parallelism,
            fix=fix,
        )
    )


@cli.command("school_classes")
@click.argument("ldap_filter", required=False)
@click.option(
    "-p",
    "--parallelism",
    type=int,
    default=PARALLELISM_DEFAULT,
    show_default=True,
    help="Parallel requests.",
)
@click.option("-a", "--school_authority_name", required=True, type=str)
@click.option("--fix", is_flag=True, help="Queue objects that failed for resync")
def verify_school_classes(
    ldap_filter: str, parallelism: int, school_authority_name: str, fix: bool
) -> None:
    """Verify school classes: '(&(ucsschoolRole=school_class:*)(LDAP_FILTER))'"""
    if ldap_filter:
        ldap_filter = f"(&(ucsschoolRole=school_class:*)({ldap_filter}))"
    else:
        ldap_filter = "(ucsschoolRole=school_class:*)"
    try:
        ldap_objects: List[Entry] = asyncio.run(
            get_ldap_objects(ldap_filter, GROUP_ATTRS)
        )
    except LDAPInvalidFilterError:
        raise click.ClickException(
            f"Your LDAP filter is invalid, the complete filter that was used: {ldap_filter}"
        )
    ldap_objects.sort(key=lambda obj: obj.entry_dn)
    school_authority = read_id_broker_config(
        config_path=CONFIG_PATH, school_authority_name=school_authority_name
    )
    configured_schools = get_configured_schools(school_authority)
    school_class_objects = [ldap_object_to_school_class(o) for o in ldap_objects]
    school_class_objects = [
        o
        for o in school_class_objects
        if o is not None
        and school_in_configured_schools(o.school.lower(), configured_schools)
    ]
    id_broker_school_class = IDBrokerSchoolClass(school_authority, "id_broker")
    asyncio.run(
        get_remote_objects_and_compare(
            school_class_objects,
            id_broker_school_class,
            Compare(school_authority, fix=fix).compare_school_group,
            parallelism,
        )
    )


@cli.command("workgroups")
@click.argument("ldap_filter", required=False)
@click.option(
    "-p",
    "--parallelism",
    type=int,
    default=PARALLELISM_DEFAULT,
    show_default=True,
    help="Parallel requests.",
)
@click.option("-a", "--school_authority_name", required=True, type=str)
@click.option("--fix", is_flag=True, help="Queue objects that failed for resync")
def verify_workgroups(
    ldap_filter: str, parallelism: int, school_authority_name: str, fix: bool
) -> None:
    """Verify work groups: '(&(ucsschoolRole=workgroups:*)(LDAP_FILTER))'"""
    if ldap_filter:
        ldap_filter = f"(&(ucsschoolRole=workgroup:*)({ldap_filter}))"
    else:
        ldap_filter = "(ucsschoolRole=workgroup:*)"
    try:
        ldap_objects: List[Entry] = asyncio.run(
            get_ldap_objects(ldap_filter, GROUP_ATTRS)
        )
    except LDAPInvalidFilterError:
        raise click.ClickException(
            f"Your LDAP filter is invalid, the complete filter that was used: {ldap_filter}"
        )
    ldap_objects.sort(key=lambda obj: obj.entry_dn)
    school_authority = read_id_broker_config(
        config_path=CONFIG_PATH, school_authority_name=school_authority_name
    )
    configured_schools = get_configured_schools(school_authority)
    _workgroup_objects = [ldap_object_to_workgroup(o) for o in ldap_objects]
    workgroup_objects = [
        o
        for o in _workgroup_objects
        if o is not None
        and school_in_configured_schools(o.school.lower(), configured_schools)
    ]
    id_broker_workgroup = IDBrokerWorkGroup(school_authority, "id_broker")
    asyncio.run(
        get_remote_objects_and_compare(
            workgroup_objects,
            id_broker_workgroup,
            Compare(school_authority, fix=fix).compare_school_group,
            parallelism,
        )
    )


@cli.command("schools")
@click.argument("ldap_filter", required=False)
@click.option(
    "-p",
    "--parallelism",
    type=int,
    default=PARALLELISM_DEFAULT,
    show_default=True,
    help="Parallel requests.",
)
@click.option("-a", "--school_authority_name", required=True, type=str)
@click.option("--fix", is_flag=True, help="Queue objects that failed for resync")
def verify_schools(
    ldap_filter: str, parallelism: int, school_authority_name: str, fix: bool
) -> None:
    """Verify schools: '(&(ucsschoolRole=school:*)(LDAP_FILTER))'"""
    if ldap_filter:
        ldap_filter = f"(&(ucsschoolRole=school:*)({ldap_filter}))"
    else:
        ldap_filter = "(ucsschoolRole=school:*)"
    try:
        ldap_objects: List[Entry] = asyncio.run(
            get_ldap_objects(ldap_filter, SCHOOL_ATTRS)
        )
    except LDAPInvalidFilterError:
        raise click.ClickException(
            f"Your LDAP filter is invalid, the complete filter that was used: {ldap_filter}"
        )

    ldap_objects.sort(key=lambda obj: obj.entry_dn)
    school_authority = read_id_broker_config(
        config_path=CONFIG_PATH, school_authority_name=school_authority_name
    )
    configured_schools = get_configured_schools(school_authority)
    school_objects = [ldap_object_to_school(o) for o in ldap_objects]
    school_objects = [
        s
        for s in school_objects
        if school_in_configured_schools(s.name.lower(), configured_schools)
    ]
    id_broker_school = IDBrokerSchool(school_authority, "id_broker")
    asyncio.run(
        get_remote_objects_and_compare(
            school_objects,
            id_broker_school,
            Compare(school_authority, fix=fix).compare_school,
            parallelism,
        )
    )


@lru_cache(maxsize=1)
def ldap_access() -> LDAPAccess:
    return LDAPAccess()


async def get_ldap_objects(ldap_filter: str, attributes: List[str]) -> List[Entry]:
    click.echo(f"{dt_str()} Searching LDAP with filter {ldap_filter!r}...")
    t0 = time.time()
    results = await ldap_access().search(filter_s=ldap_filter, attributes=attributes)
    t1 = time.time()
    click.echo(f"{dt_str()} Found {len(results)} objects in {t1 - t0:.2f} seconds.")
    return results


@lru_cache(maxsize=1024)
def role_split(ucsschool_role: str) -> Tuple[str, str, str]:
    """Cached: same ucsschool_role string results in same tuple."""
    role, context_type, context_name = ucsschool_role.split(":")
    return role, context_type, context_name


async def dn_to_id(dn: str) -> str:
    if dn not in _dn_to_id_cache:
        results = await ldap_access().search(
            base=dn, filter_s="(cn=*)", attributes="entryUUID"
        )
        if len(results) != 1:
            raise ValueError(f"Cannot find object in LDAP: {dn!r}")
        _dn_to_id_cache[dn] = results[0].entryUUID.value
    return _dn_to_id_cache[dn]


async def ldap_users_to_id_broker_users(ldap_objects: List[Entry]) -> List[User]:
    user_objects = [await ldap_object_to_user(o) for o in ldap_objects]
    return user_objects


def ldap_object_to_school(ldap_object: Entry) -> School:
    return School(
        id=ldap_object.entryUUID.value,
        name=ldap_object.ou.value,
        display_name=ldap_object.displayName.value,
    )


def ldap_object_to_school_class(ldap_object: Entry) -> Optional[SchoolClass]:
    if isinstance(ldap_object.uniqueMember.value, str):
        members = [ldap_object.uniqueMember.value]
    elif ldap_object.uniqueMember.value is None:
        members = []
    else:
        members = ldap_object.uniqueMember.value
    school = SCHOOL_CLASS_DN_PATTERN.match(ldap_object.entry_dn).groupdict()["ou"]
    try:
        ou, class_name = ldap_object.cn.value.split("-", 1)
    except ValueError:
        click.secho(
            f"{dt_str()} Name error in school class {ldap_object.cn.value!r}s"
            f" dn: {ldap_object.entry_dn!r}",
            fg="red",
        )
        class_name = ldap_object.cn.value
    else:
        if ou != school:
            click.secho(
                f"{dt_str()} OU in school class name and DN do not match: cn={ldap_object.cn.value!r}s "
                f"dn={ldap_object.entry_dn!r}",
                fg="red",
            )
    try:
        return SchoolClass(
            id=ldap_object.entryUUID.value,
            name=class_name,
            description=ldap_object.description.value,
            school=school,
            members=members,  # convert to entryUUIDs in compare_school_class(), so it'll be done in parallel
            # and only if the remote object exists
        )
    except ValueError as exc:
        click.secho(
            f"{dt_str()} ERROR creating 'SchoolClass' object from ldap_object={ldap_object!r} and "
            f"members={members!r}: {str(exc)}",
            fg="red",
            bold=True,
        )
        return None


def ldap_object_to_workgroup(ldap_object: Entry) -> Optional[WorkGroup]:
    if isinstance(ldap_object.uniqueMember.value, str):
        members = [ldap_object.uniqueMember.value]
    elif ldap_object.uniqueMember.value is None:
        members = []
    else:
        members = ldap_object.uniqueMember.value
    school = WORKGROUP_DN_PATTERN.match(ldap_object.entry_dn).groupdict()["ou"]
    try:
        ou, name = ldap_object.cn.value.split("-", 1)
    except ValueError:
        click.secho(
            f"{dt_str()} Name error in work group {ldap_object.cn.value!r}s"
            f" dn: {ldap_object.entry_dn!r}",
            fg="red",
        )
        name = ldap_object.cn.value
    else:
        if ou != school:
            click.secho(
                f"{dt_str()} OU in work group name and DN do not match: cn={ldap_object.cn.value!r}s "
                f"dn={ldap_object.entry_dn!r}",
                fg="red",
            )
    try:
        return WorkGroup(
            id=ldap_object.entryUUID.value,
            name=name,
            description=ldap_object.description.value,
            school=school,
            members=members,  # convert to entryUUIDs in compare_school_class(), so it'll be done in parallel
            # and only if the remote object exists
        )
    except ValueError as exc:
        click.secho(
            f"{dt_str()} ERROR creating 'WorkGroup' object from ldap_object={ldap_object!r} and "
            f"members={members!r}: {str(exc)}",
            fg="red",
            bold=True,
        )
        return None


async def ldap_object_to_user(ldap_object: Entry) -> User:
    if isinstance(ldap_object.ucsschoolRole.value, str):
        ucsschool_roles = [ldap_object.ucsschoolRole.value]
    elif not ldap_object.ucsschoolRole.value:
        click.secho(
            f"{dt_str()} User {ldap_object.uid.value!r} has empty ucsschoolRole attribute.",
            fg="red",
        )
        ucsschool_roles = []
    else:
        ucsschool_roles = ldap_object.ucsschoolRole.value
    context = {}
    for role, context_type, school in [role_split(u) for u in ucsschool_roles]:
        if context_type != "school":
            continue
        if role == "school_admin":
            click.secho(
                f"{dt_str()} Ignoring 'school_admin' role of user {ldap_object.uid.value!r}.",
                fg="white",
            )
            continue
        if school in context:
            context[school].roles.append(role)
        else:
            context[school] = SchoolContext(classes=[], roles=[role], workgroups=[])
    for dn in ldap_object.memberOf.value:
        if not ucsschool_roles:
            continue  # context will be empty
        if m := SCHOOL_CLASS_DN_PATTERN.match(dn):
            cn = m.groupdict()["cn"]
            try:
                ou, class_name = cn.split("-", 1)
            except ValueError:
                click.secho(
                    f"{dt_str()} Name error in user {ldap_object.uid.value!r}s school class: {dn!r}",
                    fg="red",
                )
                continue
            school = m.groupdict()["ou"]
            if ou != school:
                click.secho(
                    f"{dt_str()} Name error in user {ldap_object.uid.value!r}s school class: {dn!r}",
                    fg="red",
                )
                continue
            try:
                context[school].classes.append(class_name)
            except KeyError:
                click.secho(
                    f"{dt_str()} User {ldap_object.uid.value!r} is member of a school class of "
                    f"school {school!r} which is missing in their ucsschoolRole attribute.",
                    fg="red",
                )
        elif m := WORKGROUP_DN_PATTERN.match(dn):
            cn = m.groupdict()["cn"]
            try:
                ou, workgroup_name = cn.split("-", 1)
            except ValueError:
                click.secho(
                    f"{dt_str()} Name error in user {ldap_object.uid.value!r}s workgroup: {dn!r}",
                    fg="red",
                )
                continue
            school = m.groupdict()["ou"]
            if ou != school:
                click.secho(
                    f"{dt_str()} Name error in user {ldap_object.uid.value!r}s workgroup: {dn!r}",
                    fg="red",
                )
                continue
            try:
                context[school].workgroups.append(workgroup_name)
            except KeyError:
                click.secho(
                    f"{dt_str()} User {ldap_object.uid.value!r} is member of a workgroup of "
                    f"school {school!r} which is missing in their ucsschoolRole attribute.",
                    fg="red",
                )
    legal_guardians_dns = ldap_object.ucsschoolLegalGuardian.value
    legal_wards_dns = ldap_object.ucsschoolLegalWard.value
    legal_guardians_dns = (
        [legal_guardians_dns]
        if isinstance(legal_guardians_dns, str)
        else legal_guardians_dns
    )
    legal_wards_dns = (
        [legal_wards_dns] if isinstance(legal_wards_dns, str) else legal_wards_dns
    )

    return User(
        id=ldap_object.entryUUID.value,
        first_name=ldap_object.givenName.value or "N/A",
        last_name=ldap_object.sn.value or "N/A",
        user_name=ldap_object.uid.value,
        legal_guardians=[await dn_to_id(dn) for dn in legal_guardians_dns]
        if legal_guardians_dns
        else [],
        legal_wards=[await dn_to_id(dn) for dn in legal_wards_dns]
        if legal_wards_dns
        else [],
        context=context,
    )


def get_name(obj: IDBrokerObject) -> str:
    name = obj.user_name if isinstance(obj, User) else obj.name
    name = (
        f"{obj.school}-{name}"
        if (isinstance(obj, SchoolClass) or isinstance(obj, WorkGroup))
        else name
    )
    return f"{obj.__class__.__name__}({name!r} | {obj.id})"


def _empty_classes(context: ContextDictType) -> ContextDictType:
    for school in context:
        for class_name in context[school].classes:
            context[school].classes.remove(class_name)
    return context


class Compare:
    def __init__(self, school_authority, no_classes: bool = False, fix: bool = False):
        self.school_authority = school_authority
        self.no_classes = no_classes
        self.fix = fix
        self.userScheduler = UserScheduler()
        self.groupScheduler = GroupScheduler()
        self.schoolScheduler = SchoolScheduler()
        self.school_authority = school_authority

    async def compare_school(self, local: School, remote: Optional[School]) -> str:
        if not remote:
            if self.fix:
                await self.schoolScheduler.queue_school(local.name)
                return (
                    f"FIX  : {get_name(local)} local={local.dict()!r} remote={remote}"
                )
            return f"FAIL : {get_name(local)}: NOT FOUND on ID Broker server."
        if local == remote:
            return f"OK   : {get_name(local)}"
        else:
            if self.fix:
                await self.schoolScheduler.queue_school(local.name)
                return f"FIX  : {get_name(local)} local={local.dict()!r} remote={remote.dict()!r}"
            return f"FAIL : {get_name(local)} local={local.dict()!r} remote={remote.dict()!r}"

    async def compare_school_group(
        self,
        local: Union[SchoolClass, WorkGroup],
        remote: Optional[Union[SchoolClass, WorkGroup]],
    ) -> str:
        if not remote:
            if self.fix:
                await self.groupScheduler.queue_group(f"{local.school}-{local.name}")
                return (
                    f"FIX  : {get_name(local)} local={local.dict()!r} remote={remote}"
                )
            return f"FAIL : {get_name(local)}: NOT FOUND on ID Broker server."
        local_ori_name = local.name
        local_ori_members = local.members.copy()
        local.members = []
        t0 = time.time()
        for dn in local_ori_members:
            try:
                local.members.append(await dn_to_id(dn))
            except ValueError as exc:
                click.secho(
                    f"{dt_str()} Resolving member of SchoolClass {local.name!r}: {exc!s}",
                    fg="red",
                )
        t1 = time.time()
        t_diff = t1 - t0
        click.secho(
            f"{dt_str()} Converted {len(local.members)} member DNs of {local.name!r} to entryUUIDs in "
            f"{t_diff:.2f} seconds ({len(local.members) / t_diff:.2f}/s).",
            fg="white",
        )
        if local == remote:
            local.name = local_ori_name
            return f"OK   : {get_name(local)}"
        else:
            if self.fix:
                await self.groupScheduler.queue_group(f"{local.school}-{local.name}")
                return f"FIX  : {get_name(local)} local={local.dict()!r} remote={remote.dict()!r}"
            local.name = local_ori_name
            return f"FAIL : {get_name(local)} local={local.dict()!r} remote={remote.dict()!r}"

    async def compare_user(self, local: User, remote: Optional[User]) -> str:
        if not remote:
            if self.fix:
                await self.userScheduler.queue_user(local.user_name)
                return (
                    f"FIX  : {get_name(local)} local={local.dict()!r} remote={remote}"
                )
            return f"FAIL : {get_name(local)}: NOT FOUND on ID Broker server."
        # ignore classes, e.g initial mode
        if self.no_classes:
            local.context = _empty_classes(local.context)
            remote.context = _empty_classes(remote.context)

        # ignore local context of schools that should not be synced
        configured_schools = get_configured_schools(self.school_authority)
        for context_school in [str(s) for s in local.context.keys()]:
            if not school_in_configured_schools(
                context_school.lower(), configured_schools
            ):
                del local.context[context_school]

        if local == remote:
            return f"OK   : {get_name(local)}"
        else:
            if self.fix:
                await self.userScheduler.queue_user(local.user_name)
                return f"FIX  : {get_name(local)} local={local.dict()!r} remote={remote.dict()!r}"
            return f"FAIL : {get_name(local)} local={local.dict()!r} remote={remote.dict()!r}"


def split_list(li: List[T], chunk_size: int) -> Iterator[List[T]]:
    """Split a list into chunks of equal size (except last chunk, which is <= `chunk_size`)."""
    len_li = len(li)
    for i in range(0, len_li, chunk_size):
        yield li[i : i + chunk_size]


async def get_remote_objects_and_compare(
    local_objects: List[IDBrokerObject],
    client: ProvisioningAPIClient,
    compare_func: Callable[[IDBrokerObject, Optional[IDBrokerObject]], Awaitable[str]],
    parallelism: int,
    fix=False,
) -> List[str]:
    results: List[str] = []
    t0 = time.time()
    for local_object_list in split_list(local_objects, parallelism):
        partial_results = await asyncio.gather(
            *[
                get_remote_object_and_compare(
                    local_object, client, compare_func, fix=fix
                )
                for local_object in local_object_list
            ]
        )
        results.extend(partial_results)
        print_results(partial_results)
    t1 = time.time()
    click.secho(
        f"{dt_str()} Retrieved and compared {len(local_objects)} objects (parallelism: {parallelism}) "
        f"in {t1 - t0:.2f} seconds ({len(local_objects) / (t1 - t0):.2f}/s).",
        fg="white",
    )
    return results


async def get_remote_object_and_compare(
    local_obj: IDBrokerObject,
    client: ProvisioningAPIClient,
    compare_func: Callable[[IDBrokerObject, Optional[IDBrokerObject]], Awaitable[str]],
    fix=False,
) -> str:
    name = get_name(local_obj)

    click.secho(f"{dt_str()} Fetching {name} from ID Broker...", fg="white")
    try:
        remote_obj = await client.get(local_obj.id)
    except IDBrokerNotFoundError:
        return await compare_func(local_obj, None)  # Trigger schedule if fix is True
    except IDBrokerError as exc:
        if fix and exc.status == 500:
            try:
                # We assume that there is data corruption on the ID Broker side,
                # so we try to delete the object and reschedule.
                await client.delete(local_obj.id)
                return await compare_func(local_obj, None)
            except IDBrokerError as inner_exc:
                return f"ERROR: {name}: Connection error: {inner_exc!s}"
        else:
            return f"ERROR: {name}: Connection error: {exc!s}"
    return await compare_func(local_obj, remote_obj)


def print_results(results: Iterable[Union[str, Exception]]) -> None:
    for result in results:
        if isinstance(result, Exception):
            result = f"ERROR: {result!s}"
        color = "green" if result.startswith("OK") else "red"
        bold = result.startswith("ERROR")
        click.secho(f"{dt_str()} {result}", fg=color, bold=bold)


if __name__ == "__main__":
    cli()
