import datetime
import os
from typing import Dict, List, Sequence
from unittest.mock import MagicMock, call

import pytest
from fastapi.testclient import TestClient
from pytest_mock import MockerFixture

# REDIS_OM_URL will be used by self_disclosure_plugin.redis_sddb.get_redis_url() during the import.
old_redis_env_value = os.environ.get("REDIS_OM_URL")
os.environ["REDIS_OM_URL"] = "redis://127.0.0.1:12345"

from id_broker_common.models import RawUser  # noqa: E402
from id_broker_common.utils import remove_prefix  # noqa: E402
from self_disclosure_plugin import dependencies  # noqa: E402
from self_disclosure_plugin.models import GroupMembers, LegalGuardian, Student, Teacher  # noqa: E402
from self_disclosure_plugin.rest_sddb import SdDBRestClientImpl  # noqa: E402
from self_disclosure_plugin.routes.v1 import (  # noqa: E402
    groups as groups_module,
    users as users_module,
)  # noqa: E402
from self_disclosure_plugin.routes.v1.groups import router as groups_router  # noqa: E402
from self_disclosure_plugin.routes.v1.users import router as users_router  # noqa: E402
from self_disclosure_plugin.sddb_client import APIQueueEntryType  # noqa: E402
from self_disclosure_plugin.sddb_client import NotFoundError  # noqa: E402
from self_disclosure_plugin.sddb_client import SddbGroup, SddbInternalUser, SddbUser  # noqa: E402
from ucsschool.apis.models import OAuth2Token  # noqa: E402
from ucsschool.apis.opa import opa_instance  # noqa: E402

from ..utils import Dictable, DictObject  # noqa: E402

# restore old environment
if old_redis_env_value:
    os.environ["REDIS_OM_URL"] = old_redis_env_value
else:
    del os.environ["REDIS_OM_URL"]


@pytest.fixture
def client(app, dependency_overrides, fake_opa, async_dict):
    def _func(router, sub: str, client_id: str):
        app.include_router(router)
        overrides = {
            opa_instance: fake_opa,
            dependencies.token_dep: lambda: OAuth2Token(
                iss="",
                sub=sub,
                aud=[],
                exp=datetime.datetime.now() + datetime.timedelta(hours=3),
                client_id=client_id,
                iat=datetime.datetime.now() - datetime.timedelta(seconds=1),
                jti="",
            ),
            dependencies.connecting_user_pseudonym_from_token: lambda: sub,
            dependencies.service_provider_from_token: lambda: client_id,
        }
        dependency_overrides(overrides)
        return TestClient(app)

    return _func


service_provider = "EinServiceProvider"
traeger1 = "EinTraeger"
service_provider_attribute = "idBrokerPseudonym0001"
unused_sp_attr = "idBrokerPseudonym0002"

student1 = DictObject(
    __name__=f"{traeger1}-student1",  # needed for parametrize
    udm_properties={
        service_provider_attribute: "stu1@pseudo",
        unused_sp_attr: None,
        "entryUUID": "stu1-entry-uuid",
    },
    name=f"{traeger1}-student1",
    roles=["student"],
    firstname="stu1",
    lastname="dent",
    school_classes={f"{traeger1}-demoschool": ["class1"]},
    workgroups={f"{traeger1}-demoschool": ["workgroup1"]},
    school=f"{traeger1}-demoschool",
    schools=[f"{traeger1}-demoschool"],
    source_uid=traeger1,
    ucsschool_roles=[f"student:school:{traeger1}-demoschool"],
    legal_guardians=[],
)
student2 = DictObject(
    __name__=f"{traeger1}-student2",  # needed for parametrize
    udm_properties={
        service_provider_attribute: "stu2@pseudo",
        unused_sp_attr: None,
        "entryUUID": "stu2-entry-uuid",
    },
    name=f"{traeger1}-student2",
    roles=["student"],
    firstname="stu2",
    lastname="dent",
    school_classes={f"{traeger1}-demoschool": ["class1"]},
    workgroups={f"{traeger1}-demoschool": ["workgroup1"]},
    school=f"{traeger1}-demoschool",
    schools=[f"{traeger1}-demoschool"],
    source_uid=traeger1,
    ucsschool_roles=[f"student:school:{traeger1}-demoschool"],
    legal_guardians=[],
)
student3 = DictObject(
    __name__=f"{traeger1}-student3",  # needed for parametrize
    udm_properties={
        service_provider_attribute: "stu3@pseudo",
        unused_sp_attr: None,
        "entryUUID": "stu3-entry-uuid",
    },
    name=f"{traeger1}-student3",
    roles=["student"],
    firstname="stu3",
    lastname="dent",
    school_classes={f"{traeger1}-demoschool": ["class2"]},
    workgroups={f"{traeger1}-demoschool": ["workgroup2"]},
    school=f"{traeger1}-demoschool",
    schools=[f"{traeger1}-demoschool"],
    source_uid=traeger1,
    ucsschool_roles=[f"student:school:{traeger1}-demoschool"],
    legal_guardians=[],
)
student4 = DictObject(
    __name__=f"{traeger1}-student4",  # needed for parametrize
    udm_properties={
        service_provider_attribute: None,  # no pseudonym!
        unused_sp_attr: None,
        "entryUUID": "stu4-entry-uuid",
    },
    name=f"{traeger1}-student4",
    roles=["student"],
    firstname="stu4",
    lastname="dent",
    school_classes={f"{traeger1}-demoschool": ["class1", "class2"]},
    workgroups={f"{traeger1}-demoschool": ["workgroup1", "workgroup2"]},
    school=f"{traeger1}-demoschool",
    schools=[f"{traeger1}-demoschool"],
    source_uid=traeger1,
    ucsschool_roles=[f"student:school:{traeger1}-demoschool"],
    legal_guardians=[],
)
teacher1 = Dictable(
    __name__=f"{traeger1}-teacher1",  # needed for parametrize
    udm_properties={
        service_provider_attribute: "tea1@Pseudonym",
        unused_sp_attr: None,
        "entryUUID": "tea1-entry-uuid",
    },
    name=f"{traeger1}-teacher1",
    roles=["teacher"],
    firstname="tea1",
    lastname="cher",
    school_classes={f"{traeger1}-demoschool": ["class1"]},
    workgroups={f"{traeger1}-demoschool": ["workgroup1"]},
    school=f"{traeger1}-demoschool",
    schools=[f"{traeger1}-demoschool"],
    source_uid=traeger1,
    ucsschool_roles=[f"teacher:school:{traeger1}-demoschool"],
)
legal_guardian1 = Dictable(
    __name__=f"{traeger1}-legal_guardian1",  # needed for parametrize
    udm_properties={
        service_provider_attribute: "le1@Pseudonym",
        unused_sp_attr: None,
        "entryUUID": "le1-entry-uuid",
    },
    name=f"{traeger1}-legal_guardian1",
    roles=["legal_guardian"],
    firstname="le1",
    lastname="gal",
    school_classes={f"{traeger1}-demoschool": ["class1"]},
    workgroups={f"{traeger1}-demoschool": ["workgroup1"]},
    school=f"{traeger1}-demoschool",
    schools=[f"{traeger1}-demoschool"],
    source_uid=traeger1,
    ucsschool_roles=[f"legal_guardian:school:{traeger1}-demoschool"],
    legal_wards=[],
)
class1 = Dictable(
    name="class1",
    type="school_class",
    users=[
        student1.name,
        student2.name,
        teacher1.name,
        legal_guardian1.name,
    ],
    school=f"{traeger1}-demoschool",
    udm_properties={
        service_provider_attribute: "class1@pseudo",
        unused_sp_attr: None,
        "ucsschoolRecordUID": "class1@record",
        "entryUUID": "class1-entry-uuid",
    },
)
class2 = Dictable(
    name="class2",
    type="school_class",
    users=[student3.name, student4.name],
    school=f"{traeger1}-demoschool",
    udm_properties={
        service_provider_attribute: "class2@pseudo",
        unused_sp_attr: None,
        "ucsschoolRecordUID": "class2@record",
        "entryUUID": "class2-entry-uuid",
    },
)
workgroup1 = Dictable(
    name="workgroup1",
    type="workgroup",
    users=[
        student1.name,
        student2.name,
        teacher1.name,
        legal_guardian1.name,
    ],
    school=f"{traeger1}-demoschool",
    udm_properties={
        service_provider_attribute: "workgroup1@pseudo",
        unused_sp_attr: None,
        "ucsschoolRecordUID": "workgroup1@record",
        "entryUUID": "workgroup1-entry-uuid",
    },
)
workgroup2 = Dictable(
    name="workgroup2",
    type="workgroup",
    users=[student3.name, student4.name],
    school=f"{traeger1}-demoschool",
    udm_properties={
        service_provider_attribute: "workgroup2@pseudo",
        unused_sp_attr: None,
        "ucsschoolRecordUID": "workgroup2@record",
        "entryUUID": "workgroup2-entry-uuid",
    },
)
all_test_objects: Dict[str, DictObject] = {
    student1.name: student1,
    student2.name: student2,
    student3.name: student3,
    student4.name: student4,
    teacher1.name: teacher1,
    legal_guardian1.name: legal_guardian1,
    class1.name: class1,
    class2.name: class2,
    workgroup1.name: workgroup1,
    workgroup2.name: workgroup2,
}


def get_obj_by_name(ou, obj_name):
    for obj in all_test_objects.values():
        if obj["name"] == obj_name and obj["school"] == ou:
            return obj
    raise KeyError


def to_flat_groups(ou_groups):
    return [
        get_obj_by_name(ou, group_name)
        for ou, groups in ou_groups.items()
        for group_name in groups
    ]


def kelvin_mock_user_to_sddb_user(user) -> SddbUser:
    school_classes_uuid = [
        group["udm_properties"]["entryUUID"]
        for group in to_flat_groups(user["school_classes"])
    ]
    workgroups_uuid = [
        group["udm_properties"]["entryUUID"]
        for group in to_flat_groups(user["workgroups"])
    ]
    return SddbUser(
        pseudonym=user["udm_properties"][service_provider_attribute],
        api_version=1,
        name=user["name"],
        school=user["school"],
        schools=[user["school"]],
        school_authority=user["school"].split("-", 1)[0],
        school_classes=school_classes_uuid,
        workgroups=workgroups_uuid,
        modifyTimestamp=datetime.datetime.now(),
        legal_guardians=user.get("legal_guardians", []),
        legal_wards=user.get("legal_wards", []),
        data=user,
    )


def kelvin_mock_group_to_sddb_group(group) -> SddbGroup:
    students = [
        all_test_objects[name]["udm_properties"]["entryUUID"]
        for name in group["users"]
        if "student" in all_test_objects[name].roles
    ]
    teachers = [
        all_test_objects[name]["udm_properties"]["entryUUID"]
        for name in group["users"]
        if "teacher" in all_test_objects[name].roles
    ]
    legal_guardians = [
        all_test_objects[name]["udm_properties"]["entryUUID"]
        for name in group["users"]
        if "legal_guardian" in all_test_objects[name].roles
    ]
    return SddbGroup(
        pseudonym=group["udm_properties"][service_provider_attribute],
        api_version=1,
        type=group.type,
        name=group["name"],
        school=group["school"].split("-", 1)[1],
        school_authority=group["school"].split("-", 1)[0],
        modifyTimestamp=datetime.datetime.now(),
        students=students,
        teachers=teachers,
        legal_guardians=legal_guardians,
        data=group,
    )


class SDDBClientMock:
    def get_users(self, pks: List[str]):  # -> List[Optional[SddbInternalUser]]:
        internal_users = []
        for pk in pks:
            try:
                user = [
                    obj
                    for obj in all_test_objects.values()
                    if pk in obj["udm_properties"].values()
                ][0]
            except IndexError:
                internal_users.append(None)
            else:
                sddb_user = kelvin_mock_user_to_sddb_user(user)
                internal_user = SddbInternalUser(id="foo", **sddb_user._asdict())
                internal_user["p0001"] = user["udm_properties"][
                    service_provider_attribute
                ]
                internal_user["p0002"] = user["udm_properties"][unused_sp_attr]
                internal_users.append(internal_user)
        return internal_users


def username_from_user(obj):
    return f"{obj.firstname}.{obj.lastname}"


def user_to_raw_user(user: DictObject) -> RawUser:
    return RawUser(
        dn="foo",
        firstname=user.firstname,
        lastname=user.lastname,
        record_uid="foo",
        schools=user.schools,
        source_uid=user.source_uid,
        synonyms={
            k: v
            for k, v in user.udm_properties.items()
            if k.startswith("idBrokerPseudonym00")
        },
        ucsschool_roles=user.ucsschool_roles,
        username=user.name,
    )


async def ldap_get_users_mock(user_names: Sequence[str]) -> List[RawUser]:
    return [user_to_raw_user(all_test_objects[name]) for name in user_names]


def student_from_dict_object(obj: DictObject) -> Student:
    return Student(
        user_id=obj.udm_properties[service_provider_attribute],
        username=username_from_user(obj),
        firstname=obj.firstname,
        lastname=obj.lastname,
        legal_guardians=obj.legal_guardians,
    )


def teacher_from_dict_object(obj: DictObject) -> Teacher:
    return Teacher(
        user_id=obj.udm_properties[service_provider_attribute],
        username=username_from_user(obj),
    )


def legal_guardian_from_dict_object(obj: DictObject) -> LegalGuardian:
    return LegalGuardian(
        user_id=obj.udm_properties[service_provider_attribute],
        username=username_from_user(obj),
        firstname=obj.firstname,
        lastname=obj.lastname,
        legal_wards=obj.legal_wards,
    )


def get_user_mock(pseudonym: str, service_provider: str) -> SddbUser:
    try:
        user = [
            obj
            for obj in all_test_objects.values()
            if pseudonym in obj["udm_properties"].values()
        ][0]
    except IndexError:
        raise NotFoundError
    return kelvin_mock_user_to_sddb_user(user)


def test_users_metadata(client, monkeypatch):
    monkeypatch.setattr(
        dependencies,
        "get_user",
        MagicMock(return_value=kelvin_mock_user_to_sddb_user(student1)),
    )
    test_client = client(
        router=users_router,
        sub=student1.udm_properties[service_provider_attribute],
        client_id=service_provider,
    )
    response = test_client.get(
        f"/{student1.udm_properties[service_provider_attribute]}/metadata"
    )
    assert response.status_code == 200, response.__dict__
    json_response = response.json()
    school_authority_prefix = f"{student1.source_uid}-"
    expected_user = {
        "user_id": student1.udm_properties[service_provider_attribute],
        "username": username_from_user(student1),
        "firstname": student1.firstname,
        "lastname": student1.lastname,
        "type": student1.roles[0],
        "school_id": remove_prefix(student1.school, school_authority_prefix),
        "school_authority": student1.source_uid,
        "legal_guardians": [],
        "legal_wards": [],
    }
    assert json_response == expected_user


@pytest.mark.parametrize(
    "user", [student1, student2, student3, student4, teacher1, legal_guardian1]
)
def test_user_groups(client, monkeypatch, user):
    def get_group_by_id_mock(group_id, service_provider):
        for obj in all_test_objects.values():
            if obj["udm_properties"]["entryUUID"] == group_id:
                return kelvin_mock_group_to_sddb_group(obj)
        raise KeyError

    monkeypatch.setattr(
        dependencies,
        "get_user",
        MagicMock(return_value=kelvin_mock_user_to_sddb_user(user)),
    )
    monkeypatch.setattr(
        dependencies,
        "get_service_provider_attribute",
        MagicMock(return_value=service_provider_attribute),
    )
    monkeypatch.setattr(
        users_module,
        "get_group_by_id",
        get_group_by_id_mock,
    )
    test_client = client(
        router=users_router,
        sub=teacher1.udm_properties[service_provider_attribute],
        client_id=service_provider,
    )
    response = test_client.get(
        f"/{user.udm_properties[service_provider_attribute]}/groups"
    )
    assert response.status_code == 200, response.__dict__
    json_response = response.json()
    school_authority, school_id = class1.school.split("-", 1)
    expected_groups = {"groups": []}
    for group in to_flat_groups(user.school_classes) + to_flat_groups(user.workgroups):
        group = kelvin_mock_group_to_sddb_group(group)
        expected_groups["groups"].append(
            {
                "group_id": group.pseudonym,
                "name": group.name,
                "type": group.type,
                "school_id": school_id,
                "school_authority": school_authority,
                "student_count": 2,
            }
        )
    assert json_response == expected_groups


@pytest.mark.parametrize("group", [class1, workgroup1])
def test_group_users(group, client, monkeypatch):
    monkeypatch.setattr(
        dependencies,
        "get_service_provider_attribute",
        MagicMock(return_value=service_provider_attribute),
    )
    monkeypatch.setattr(
        dependencies,
        "get_connected_user",
        MagicMock(return_value=kelvin_mock_user_to_sddb_user(teacher1)),
    )

    monkeypatch.setattr(dependencies, "get_user", get_user_mock)
    monkeypatch.setattr(
        groups_module,
        "get_group_by_pseudonym",
        MagicMock(return_value=kelvin_mock_group_to_sddb_group(group)),
    )
    monkeypatch.setattr(dependencies, "sddb_client", SDDBClientMock)

    test_client = client(
        router=groups_router,
        sub=teacher1.udm_properties[service_provider_attribute],
        client_id=service_provider,
    )
    response = test_client.get(
        f"/{group.udm_properties[service_provider_attribute]}/users"
    )
    assert response.status_code == 200, response.__dict__
    json_response = response.json()
    expected_group = {
        "students": [
            {
                "user_id": student1.udm_properties[service_provider_attribute],
                "username": username_from_user(student1),
                "firstname": student1.firstname,
                "lastname": student1.lastname,
                "legal_guardians": [],
            },
            {
                "user_id": student2.udm_properties[service_provider_attribute],
                "username": username_from_user(student2),
                "firstname": student2.firstname,
                "lastname": student2.lastname,
                "legal_guardians": [],
            },
        ],
        "teachers": [
            {
                "user_id": teacher1.udm_properties[service_provider_attribute],
                "username": username_from_user(teacher1),
            }
        ],
        "legal_guardians": [
            {
                "user_id": legal_guardian1.udm_properties[service_provider_attribute],
                "username": username_from_user(legal_guardian1),
                "firstname": legal_guardian1.firstname,
                "lastname": legal_guardian1.lastname,
                "legal_wards": [],
            }
        ],
    }
    assert json_response == expected_group


@pytest.mark.parametrize("group", [class1, workgroup1])
def test_extract_members(group, monkeypatch):
    monkeypatch.setattr(
        dependencies,
        "get_service_provider_attribute",
        MagicMock(return_value=service_provider_attribute),
    )
    monkeypatch.setattr(dependencies, "sddb_client", SDDBClientMock)
    classmembers = dependencies.extract_members(
        kelvin_mock_group_to_sddb_group(group), service_provider
    )
    assert classmembers == GroupMembers(
        students=[
            student_from_dict_object(student1),
            student_from_dict_object(student2),
        ],
        teachers=[teacher_from_dict_object(teacher1)],
        legal_guardians=[legal_guardian_from_dict_object(legal_guardian1)],
    )


@pytest.mark.parametrize("user", [student1, teacher1, legal_guardian1])
def test_roles_in_school(user):
    roles = dependencies.roles_in_school(user.ucsschool_roles, user.school)
    assert roles == set(user.roles)


def test_get_user_groups_enqueues_missing_in_sddb(
    client, mocker: MockerFixture, monkeypatch
):
    mocker.patch.object(
        dependencies,
        "get_service_provider_attribute",
        MagicMock(return_value=service_provider_attribute),
    )
    monkeypatch.setattr(
        users_module,
        "get_group_by_id",
        MagicMock(side_effect=NotFoundError),
    )
    monkeypatch.setattr(
        dependencies,
        "get_user",
        MagicMock(return_value=kelvin_mock_user_to_sddb_user(student1)),
    )
    sddb_rest_client_mock = mocker.MagicMock(spec_set=SdDBRestClientImpl)
    mocker.patch(
        "self_disclosure_plugin.routes.v1.users.sddb_rest_client",
        return_value=sddb_rest_client_mock,
    )

    test_client = client(
        router=users_router,
        sub=student1.udm_properties[service_provider_attribute],
        client_id=service_provider,
    )
    response = test_client.get(
        f"/{student1.udm_properties[service_provider_attribute]}/groups"
    )
    assert response.status_code == 200, response.__dict__
    sddb_rest_client_mock.enqueue.assert_has_calls(
        [
            call(
                entry_type=APIQueueEntryType.user,
                pseudonym=student1.udm_properties[service_provider_attribute],
                service_provider=service_provider,
            ),
            call(
                entry_type=APIQueueEntryType.group,
                service_provider=service_provider,
                entry_uuid=class1.udm_properties["entryUUID"],
            ),
            call(
                entry_type=APIQueueEntryType.group,
                service_provider=service_provider,
                entry_uuid=workgroup1.udm_properties["entryUUID"],
            ),
        ],
        any_order=True,
    )


@pytest.mark.parametrize("user_type", ["student", "teacher", "legal_guardian"])
@pytest.mark.parametrize("group", [class1, workgroup1])
def test_get_group_members_enqueues_missing_in_sddb(
    user_type, group, client, mocker: MockerFixture, monkeypatch
):
    monkeypatch.setattr(
        dependencies,
        "get_service_provider_attribute",
        MagicMock(return_value=service_provider_attribute),
    )
    monkeypatch.setattr(
        dependencies,
        "get_connected_user",
        MagicMock(return_value=kelvin_mock_user_to_sddb_user(teacher1)),
    )

    monkeypatch.setattr(dependencies, "get_user", get_user_mock)
    sddb_group = kelvin_mock_group_to_sddb_group(group)
    if user_type == "student":
        sddb_group.students.extend(["s_missing"])
    else:
        sddb_group.teachers.extend(["t_missing"])
    monkeypatch.setattr(
        groups_module,
        "get_group_by_pseudonym",
        MagicMock(return_value=sddb_group),
    )
    monkeypatch.setattr(dependencies, "sddb_client", SDDBClientMock)
    sddb_rest_client_mock = mocker.MagicMock(spec_set=SdDBRestClientImpl)
    mocker.patch.object(
        dependencies, "sddb_rest_client", return_value=sddb_rest_client_mock
    )

    test_client = client(
        router=groups_router,
        sub=teacher1.udm_properties[service_provider_attribute],
        client_id=service_provider,
    )
    response = test_client.get(
        f"/{group.udm_properties[service_provider_attribute]}/users"
    )
    assert response.status_code == 200, response.__dict__
    sddb_rest_client_mock.enqueue.assert_has_calls(
        [
            call(
                entry_type=APIQueueEntryType.group,
                pseudonym=group.udm_properties[service_provider_attribute],
                service_provider=service_provider,
            ),
        ]
    )


def test_get_connected_user_enqueues_user_missing_in_sddb(
    client, mocker: MockerFixture, monkeypatch
):
    mocker.patch.object(
        dependencies,
        "get_service_provider_attribute",
        MagicMock(return_value=service_provider_attribute),
    )
    sddb_rest_client_mock = mocker.MagicMock(spec_set=SdDBRestClientImpl)
    mocker.patch.object(
        dependencies, "sddb_rest_client", return_value=sddb_rest_client_mock
    )
    mocker.patch("redis_om.model.migrations.migrator.Migrator.detect_migrations")
    monkeypatch.setenv("REDIS_OM_URL", "redis://foo")
    mocker.patch("redis_om.connections.get_redis_connection")
    mocker.patch("redis.commands.core.ManagementCommands.ping")
    mocker.patch("redis_om.model.model.has_redis_json")
    mocker.patch(
        "redis_om.model.model.RedisModel.find", MagicMock(side_effect=NotFoundError())
    )

    test_client = client(
        router=users_router,
        sub=student1.udm_properties[service_provider_attribute],
        client_id=service_provider,
    )
    response = test_client.get(
        f"/{student1.udm_properties[service_provider_attribute]}/metadata"
    )
    assert response.status_code == 401, response.__dict__
    sddb_rest_client_mock.enqueue.assert_called_once()
    assert sddb_rest_client_mock.enqueue.call_args.kwargs == {
        "entry_type": APIQueueEntryType.user,
        "pseudonym": student1.udm_properties[service_provider_attribute],
        "service_provider": service_provider,
    }
