import pytest
import requests

from provisioning_plugin.models import WorkGroup
from ucsschool.kelvin.client import NoObject, WorkGroupResource

from ..utils import DEFAULT_SCHOOL_AUTHORITY, DEFAULT_SCHOOL_NAME, must_run_in_docker

pytestmark = must_run_in_docker


@pytest.mark.asyncio
async def test_workgroups_get(
    create_school, create_workgroup, create_user, url_fragment, auth_headers
):
    await create_school()
    user = await create_user()
    workgroup = await create_workgroup(descr="WORKGROUP", members=[user.name])
    response = requests.get(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/"
        f"workgroups/{workgroup.udm_properties.get('ucsschoolRecordUID')}",
        headers=await auth_headers(DEFAULT_SCHOOL_AUTHORITY),
    )
    assert response.status_code == 200, response.__dict__
    assert response.json() == {
        "name": workgroup.name,
        "school": DEFAULT_SCHOOL_NAME,
        "members": [user.record_uid],
        "description": "WORKGROUP",
        "id": workgroup.udm_properties.get("ucsschoolRecordUID"),
    }, response.__dict__


@pytest.mark.asyncio
async def test_workgroups_head(
    create_school, create_workgroup, create_user, url_fragment, auth_headers
):
    await create_school()
    user = await create_user()
    workgroup = await create_workgroup(descr="WORKGROUP", members=[user.name])
    response = requests.head(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/"
        f"workgroups/{workgroup.udm_properties.get('ucsschoolRecordUID')}",
        headers=await auth_headers(DEFAULT_SCHOOL_AUTHORITY),
    )
    assert response.status_code == 200, response.__dict__
    assert response.content == b""


@pytest.mark.asyncio
async def test_workgroups_get_wrong_auth(
    create_school, create_workgroup, url_fragment, auth_headers
):
    await create_school(DEFAULT_SCHOOL_AUTHORITY)
    await create_school("TEST2")
    workgroup = await create_workgroup(descr="DESCR", members=[])
    response = requests.get(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/"
        f"workgroups/{workgroup.udm_properties.get('ucsschoolRecordUID')}",
        headers=await auth_headers("TEST2"),
    )
    assert response.status_code == 403, response.__dict__


@pytest.mark.asyncio
async def test_workgroups_post(
    create_school, create_user, url_fragment, auth_headers, kelvin_session_obj, faker
):
    await create_school()
    u1 = await create_user()
    u2 = await create_user(roles=["student"])
    workgroup_name = faker.pystr(1, 7)
    workgroup_id = faker.pystr(1, 7)
    response = requests.post(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/workgroups/",
        data=WorkGroup(
            id=workgroup_id,
            name=workgroup_name,
            description="WORKGROUP",
            school=DEFAULT_SCHOOL_NAME,
            members={u1.record_uid, u2.record_uid},
        ).json(),
        headers={
            "Content-Type": "application/json",
            **await auth_headers(DEFAULT_SCHOOL_AUTHORITY),
        },
    )
    assert response.status_code == 201, response.__dict__
    workgroup = await WorkGroupResource(session=kelvin_session_obj).get(
        name=workgroup_name, school=f"{DEFAULT_SCHOOL_AUTHORITY}-{DEFAULT_SCHOOL_NAME}"
    )
    assert workgroup.name == workgroup_name
    assert workgroup.description == "WORKGROUP"
    assert workgroup.school == f"{DEFAULT_SCHOOL_AUTHORITY}-{DEFAULT_SCHOOL_NAME}"
    assert set(workgroup.users) == {u1.name, u2.name}
    assert workgroup.udm_properties.get("ucsschoolRecordUID") == workgroup_id
    assert (
        workgroup.udm_properties.get("ucsschoolSourceUID") == DEFAULT_SCHOOL_AUTHORITY
    )


@pytest.mark.asyncio
async def test_workgroups_put(
    create_school,
    create_user,
    create_workgroup,
    url_fragment,
    auth_headers,
    kelvin_session_obj,
):
    await create_school()
    u1 = await create_user()
    u2 = await create_user(roles=["student"])
    workgroup = await create_workgroup(descr="DESCR")
    new_workgroup = WorkGroup.without_prefix(
        await WorkGroup.from_kelvin_workgroup(
            workgroup.as_dict(), session=kelvin_session_obj
        ),
        DEFAULT_SCHOOL_AUTHORITY,
    )
    new_workgroup.members = {u1.record_uid, u2.record_uid}
    response = requests.put(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/workgroups/"
        f"{workgroup.udm_properties.get('ucsschoolRecordUID')}",
        data=new_workgroup.json(),
        headers={
            "Content-Type": "application/json",
            **await auth_headers(DEFAULT_SCHOOL_AUTHORITY),
        },
    )
    assert response.status_code == 200, response.__dict__
    kelvin_workgroup = WorkGroup.from_kelvin_workgroup(
        (
            await WorkGroupResource(session=kelvin_session_obj).get(
                name=workgroup.name, school=workgroup.school
            )
        ).as_dict(),
        session=kelvin_session_obj,
    )
    assert await kelvin_workgroup == WorkGroup.with_prefix(
        new_workgroup, DEFAULT_SCHOOL_AUTHORITY
    )


@pytest.mark.asyncio
async def test_workgroups_change_school_fails_put(
    create_school,
    create_user,
    create_workgroup,
    url_fragment,
    auth_headers,
    kelvin_session_obj,
):
    await create_school()
    await create_school(name="otherschool")
    u1 = await create_user()
    u2 = await create_user(roles=["student"])
    workgroup = await create_workgroup(descr="DESCR")
    new_workgroup = WorkGroup.without_prefix(
        await WorkGroup.from_kelvin_workgroup(
            workgroup.as_dict(), session=kelvin_session_obj
        ),
        DEFAULT_SCHOOL_AUTHORITY,
    )
    new_workgroup.school = "otherschool"
    new_workgroup.members = {u1.record_uid, u2.record_uid}
    response = requests.put(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/workgroups/"
        f"{workgroup.udm_properties.get('ucsschoolRecordUID')}",
        data=new_workgroup.json(),
        headers={
            "Content-Type": "application/json",
            **await auth_headers(DEFAULT_SCHOOL_AUTHORITY),
        },
    )
    assert response.status_code == 422, response.__dict__
    assert response.json() == {
        "detail": "The school must not be changed after creation."
    }


@pytest.mark.asyncio
async def test_workgroups_delete(
    create_school, create_workgroup, url_fragment, auth_headers, kelvin_session_obj
):
    await create_school()
    workgroup = await create_workgroup(descr="DESCR")
    response = requests.delete(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/"
        f"workgroups/{workgroup.udm_properties.get('ucsschoolRecordUID')}",
        headers=await auth_headers(DEFAULT_SCHOOL_AUTHORITY),
    )
    assert response.status_code == 204, response.__dict__
    with pytest.raises(NoObject):
        await WorkGroupResource(session=kelvin_session_obj).get(
            name=workgroup.name,
            school=f"{DEFAULT_SCHOOL_AUTHORITY}-{DEFAULT_SCHOOL_NAME}",
        )
