#!/usr/share/ucs-test/runner pytest-3 -s -l -v
## -*- coding: utf-8 -*-
## desc: Verify removing school authority using python script
## roles: [domaincontroller_master]
## tags: [ucsschool-id-broker, ucsschool-id-broker-primary-only]
## packages: [id-broker-common]
## exposure: dangerous

import os
import subprocess
from copy import deepcopy
from datetime import datetime
from importlib.machinery import SourceFileLoader
from importlib.util import module_from_spec, spec_from_loader
from unittest import mock

import pytest
from keycloak import KeycloakAdmin
from keycloak.exceptions import KeycloakGetError, KeycloakPostError

from ucsschool.lib.models.school import School
from ucsschool.lib.models.utils import ucr
from univention.admin.uexceptions import noObject as AdminNoObject
from univention.admin.uldap import access
from univention.testing import utils
from univention.udm import UDM
from univention.udm.exceptions import CreateError, NoObject

# This allows us to import the script as a python module
# and do full branch testing
spec = spec_from_loader(
    "remove_school_authority",
    SourceFileLoader("remove_school_authority", "/usr/bin/remove-school-authority"),
)
remove_school_authority = module_from_spec(spec)
spec.loader.exec_module(remove_school_authority)

rma_main = remove_school_authority.main
parse_cmdline = remove_school_authority.parse_cmdline
SchoolAuthority = remove_school_authority.SchoolAuthority
SchoolAuthCleaner = remove_school_authority.SchoolAuthCleaner
domainAdmin = utils.UCSTestDomainAdminCredentials()


# Keycloak settings
KEYCLOAK_URL = "https://kc.broker.test/auth/"
KEYCLOAK_USER = domainAdmin.username
KEYCLOAK_PASSWORD = domainAdmin.bindpw
KEYCLOAK_ADMIN_REALM = "master"
KEYCLOAK_REALM = "ID-Broker"


# Because we can't collect password input on the command line,
# we're going to mock it for all tests
mock_password = KEYCLOAK_PASSWORD
mock_user_password_input = mock.Mock()
mock_user_password_input.return_value = mock_password
remove_school_authority.user_password_input = mock_user_password_input


def get_idp_payload(alias: str) -> dict:
    return {
        "alias": alias,
        "providerId": "saml",
        "config": {
            "clientId": "test-22-remove-school-authority",
            "clientSecret": "test",
        },
    }


def create_idp(keycloak_connection: KeycloakAdmin, idp_name: str):
    try:
        keycloak_connection.create_idp(get_idp_payload(idp_name))
    except (KeycloakGetError, KeycloakPostError):
        # IDP already exists; ignore
        pass
    return idp_name


def create_idp_user(udm_connection: UDM, idp_name: str):
    user = udm_connection.get("users/user").new()
    uid = f"provisioning-{idp_name}"
    user.props.username = uid
    user.props.lastname = idp_name
    user.props.password = "univention"
    user.props.unixhome = f"/home/provisioning-{idp_name}"
    try:
        user.save()
    except CreateError:
        pass

    ldap_base = udm_connection.connection.base
    return udm_connection.get("users/user").get(dn=f"uid={uid},cn=users,{ldap_base}")


def create_idp_school(ldap_connection: access, idp_name: str, school_name: str):
    school_name = f"{idp_name}-{school_name}"
    school = School(name=school_name)
    school.create(ldap_connection)
    return School.from_dn(
        f"ou={school_name},{ldap_connection.base}",
        None,
        ldap_connection,
    )


class TestScript:
    """
    Tests for running the script directly.
    Unfortunately, we can't test the normal interactive mode;
    please see other unit tests for INTERACTIVE_MODE coverage.
    """

    DEFAULT_KEYCLOAK_CONFIG = {
        "server_url": KEYCLOAK_URL,
        "username": KEYCLOAK_USER,
        "password": KEYCLOAK_PASSWORD,
        "realm_name": KEYCLOAK_REALM,
        "user_realm_name": KEYCLOAK_ADMIN_REALM,
    }

    def test_dry_run(self):
        pwd_filepath = f"/tmp/{datetime.now().strftime('%Y%m%d%H%M')}-dry-run-test.txt"
        with open(pwd_filepath, "w") as pwd_file:
            pwd_file.write(f"{KEYCLOAK_PASSWORD}\n")

        keycloak_connection = remove_school_authority.get_keycloak_connection(
            **self.DEFAULT_KEYCLOAK_CONFIG
        )
        ldap_connection = remove_school_authority.get_ldap_admin_connection()
        udm_connection = remove_school_authority.get_udm_connection()

        idp_name = "tscriptdr1"
        create_idp(keycloak_connection, idp_name)
        user = create_idp_user(udm_connection, idp_name)
        school = create_idp_school(ldap_connection, idp_name, "School1")

        # pre-delete checks
        assert idp_name in [idp["alias"] for idp in keycloak_connection.get_idps()]
        try:
            School.from_dn(school.dn, None, ldap_connection)
        except AdminNoObject:
            assert False, "School doesn't exist"
        try:
            udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

        cmd = (
            "remove-school-authority "
            f"--keycloak-admin-name {KEYCLOAK_USER} "
            f"--keycloak-admin-realm {KEYCLOAK_ADMIN_REALM} "
            f"--keycloak-authority-realm {KEYCLOAK_REALM} "
            f"--keycloak-url {KEYCLOAK_URL} "
            f"--school-authority-name {idp_name} "
            f"--keycloak-admin-password-file {pwd_filepath} "
            "--dry-run"
        )
        try:
            exit_code = subprocess.call(cmd, shell=True)
            assert exit_code == 0
        finally:
            os.remove(pwd_filepath)

        # nothing should have been deleted
        assert idp_name in [idp["alias"] for idp in keycloak_connection.get_idps()]
        try:
            School.from_dn(school.dn, None, ldap_connection)
        except AdminNoObject:
            assert False, "School was deleted"
        try:
            udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User was deleted"

    def test_force(self):
        pwd_filepath = f"/tmp/{datetime.now().strftime('%Y%m%d%H%M')}-force-test.txt"
        with open(pwd_filepath, "w") as pwd_file:
            pwd_file.write(f"{KEYCLOAK_PASSWORD}\n")

        keycloak_connection = remove_school_authority.get_keycloak_connection(
            **self.DEFAULT_KEYCLOAK_CONFIG
        )
        ldap_connection = remove_school_authority.get_ldap_admin_connection()
        udm_connection = remove_school_authority.get_udm_connection()

        idp_name = "tscriptforce1"
        create_idp(keycloak_connection, idp_name)
        user = create_idp_user(udm_connection, idp_name)
        school = create_idp_school(ldap_connection, idp_name, "School1")

        # pre-delete checks
        assert idp_name in [idp["alias"] for idp in keycloak_connection.get_idps()]
        try:
            School.from_dn(school.dn, None, ldap_connection)
        except AdminNoObject:
            assert False, "School doesn't exist"
        try:
            udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

        cmd = (
            "remove-school-authority "
            f"--keycloak-admin-name {KEYCLOAK_USER} "
            f"--keycloak-admin-realm {KEYCLOAK_ADMIN_REALM} "
            f"--keycloak-authority-realm {KEYCLOAK_REALM} "
            f"--keycloak-url {KEYCLOAK_URL} "
            f"--school-authority-name {idp_name} "
            f"--keycloak-admin-password-file {pwd_filepath} "
            "--force"
        )
        try:
            exit_code = subprocess.call(cmd, shell=True)
            assert exit_code == 0
        finally:
            os.remove(pwd_filepath)

        # everything should have been deleted
        assert idp_name not in [idp["alias"] for idp in keycloak_connection.get_idps()]
        try:
            School.from_dn(school.dn, None, ldap_connection)
            assert False, "School still exists"
        except AdminNoObject:
            # School1 was deleted
            pass
        try:
            udm_connection.get("users/user").get(dn=user.dn)
            assert False, "User not deleted"
        except NoObject:
            # user was deleted
            pass


class TestMain:
    """Tests for main"""

    def test_script_wont_run_if_not_primary(self):
        original_ucr = remove_school_authority.ucr
        mock_ucr = mock.MagicMock()
        mock_ucr.get.return_value = "fake_server"
        remove_school_authority.ucr = mock_ucr

        with pytest.raises(
            SystemExit,
            match=(
                "This script is only intended to be run on primary servers. "
                "The current system has role: 'fake_server' instead of "
                "'domaincontroller_master'."
            ),
        ):
            rma_main()

        remove_school_authority.ucr = original_ucr

    def test_args_passed_to_school_auth_cleaner(self):
        original_sac = remove_school_authority.SchoolAuthCleaner
        mock_sac_cls = mock.MagicMock()
        mock_sac = mock.MagicMock()
        mock_sac_cls.return_value = mock_sac
        remove_school_authority.SchoolAuthCleaner = mock_sac_cls

        original_pc = remove_school_authority.parse_cmdline
        mock_pc = mock.MagicMock()
        mock_pc.return_value = {
            "keycloak_config": {
                "server_url": "http://foo.com",
                "username": "foo",
                "password": "bar",
                "realm_name": "baz",
                "user_realm_name": "buz",
            },
            "school_authority_name": "fake",
            "mode": 0,
        }
        remove_school_authority.parse_cmdline = mock_pc

        try:
            rma_main()
            mock_sac_cls.assert_called_once_with(**mock_pc.return_value)
            mock_sac.clean.assert_called_once()
        finally:
            remove_school_authority.SchoolAuthCleaner = original_sac
            remove_school_authority.parse_cmdline = original_pc


class TestParseCmdline:
    """Tests for parse_cmdline"""

    def test_minimum_arguments(self):
        sysargs = [
            "remove-school-authority",
            "--keycloak-admin-name",
            KEYCLOAK_USER,
            "--keycloak-admin-realm",
            KEYCLOAK_ADMIN_REALM,
            "--keycloak-authority-realm",
            KEYCLOAK_REALM,
            "--keycloak-url",
            KEYCLOAK_URL,
            "--school-authority-name",
            "TestSchool",
        ]
        expected_args = {
            "keycloak_config": {
                "server_url": KEYCLOAK_URL,
                "username": KEYCLOAK_USER,
                "password": mock_password,
                "realm_name": KEYCLOAK_REALM,
                "user_realm_name": KEYCLOAK_ADMIN_REALM,
            },
            "school_authority_name": "TestSchool",
            "mode": 0,
        }
        with mock.patch("sys.argv", sysargs):
            args = parse_cmdline()
            assert args == expected_args

    def test_keycloak_admin_name_required(self):
        sysargs = [
            "remove-school-authority",
            "--keycloak-admin-realm",
            KEYCLOAK_ADMIN_REALM,
            "--keycloak-authority-realm",
            KEYCLOAK_REALM,
            "--keycloak-url",
            KEYCLOAK_URL,
            "--school-authority-name",
            "TestSchool",
        ]
        with mock.patch("sys.argv", sysargs):
            with pytest.raises(SystemExit, match="2"):
                parse_cmdline()

    def test_keycloak_admin_realm_required(self):
        sysargs = [
            "remove-school-authority",
            "--keycloak-admin-name",
            KEYCLOAK_USER,
            "--keycloak-authority-realm",
            KEYCLOAK_REALM,
            "--keycloak-url",
            KEYCLOAK_URL,
            "--school-authority-name",
            "TestSchool",
        ]
        with mock.patch("sys.argv", sysargs):
            with pytest.raises(SystemExit, match="2"):
                parse_cmdline()

    def test_keycloak_authority_realm_required(self):
        sysargs = [
            "remove-school-authority",
            "--keycloak-admin-name",
            KEYCLOAK_USER,
            "--keycloak-admin-realm",
            KEYCLOAK_ADMIN_REALM,
            "--keycloak-url",
            KEYCLOAK_URL,
            "--school-authority-name",
            "TestSchool",
        ]
        with mock.patch("sys.argv", sysargs):
            with pytest.raises(SystemExit, match="2"):
                parse_cmdline()

    def test_keycloak_url_required(self):
        sysargs = [
            "remove-school-authority",
            "--keycloak-admin-name",
            KEYCLOAK_USER,
            "--keycloak-admin-realm",
            KEYCLOAK_ADMIN_REALM,
            "--keycloak-authority-realm",
            KEYCLOAK_REALM,
            "--school-authority-name",
            "TestSchool",
        ]
        with mock.patch("sys.argv", sysargs):
            with pytest.raises(SystemExit, match="2"):
                parse_cmdline()

    def test_school_authority_name_required(self):
        sysargs = [
            "remove-school-authority",
            "--keycloak-admin-name",
            KEYCLOAK_USER,
            "--keycloak-admin-realm",
            KEYCLOAK_ADMIN_REALM,
            "--keycloak-authority-realm",
            KEYCLOAK_REALM,
            "--keycloak-url",
            KEYCLOAK_URL,
        ]
        with mock.patch("sys.argv", sysargs):
            with pytest.raises(SystemExit, match="2"):
                parse_cmdline()

    def test_args_with_keycloak_admin_password_file(self):
        pwd_filepath = (
            f"/tmp/{datetime.now().strftime('%Y%m%d%H%M')}-fake-keycloak-pwd.txt"
        )
        pwd = "9876543Plm"
        with open(pwd_filepath, "w") as pwd_file:
            pwd_file.write(f"{pwd}\n")

        sysargs = [
            "remove-school-authority",
            "--keycloak-admin-name",
            KEYCLOAK_USER,
            "--keycloak-admin-realm",
            KEYCLOAK_ADMIN_REALM,
            "--keycloak-authority-realm",
            KEYCLOAK_REALM,
            "--keycloak-url",
            KEYCLOAK_URL,
            "--school-authority-name",
            "TestSchool",
            "--keycloak-admin-password-file",
            pwd_filepath,
        ]
        expected_args = {
            "keycloak_config": {
                "server_url": KEYCLOAK_URL,
                "username": KEYCLOAK_USER,
                "password": pwd,
                "realm_name": KEYCLOAK_REALM,
                "user_realm_name": KEYCLOAK_ADMIN_REALM,
            },
            "school_authority_name": "TestSchool",
            "mode": 0,
        }
        try:
            with mock.patch("sys.argv", sysargs):
                args = parse_cmdline()
                assert args == expected_args
        finally:
            os.remove(pwd_filepath)

    def test_args_with_dry_run(self):
        sysargs = [
            "remove-school-authority",
            "--keycloak-admin-name",
            KEYCLOAK_USER,
            "--keycloak-admin-realm",
            KEYCLOAK_ADMIN_REALM,
            "--keycloak-authority-realm",
            KEYCLOAK_REALM,
            "--keycloak-url",
            KEYCLOAK_URL,
            "--school-authority-name",
            "TestSchool",
            "--dry-run",
        ]
        expected_args = {
            "keycloak_config": {
                "server_url": KEYCLOAK_URL,
                "username": KEYCLOAK_USER,
                "password": mock_password,
                "realm_name": KEYCLOAK_REALM,
                "user_realm_name": KEYCLOAK_ADMIN_REALM,
            },
            "school_authority_name": "TestSchool",
            "mode": 1,
        }
        with mock.patch("sys.argv", sysargs):
            args = parse_cmdline()
            assert args == expected_args

    def test_args_with_force(self):
        sysargs = [
            "remove-school-authority",
            "--keycloak-admin-name",
            KEYCLOAK_USER,
            "--keycloak-admin-realm",
            KEYCLOAK_ADMIN_REALM,
            "--keycloak-authority-realm",
            KEYCLOAK_REALM,
            "--keycloak-url",
            KEYCLOAK_URL,
            "--school-authority-name",
            "TestSchool",
            "--force",
        ]
        expected_args = {
            "keycloak_config": {
                "server_url": KEYCLOAK_URL,
                "username": KEYCLOAK_USER,
                "password": mock_password,
                "realm_name": KEYCLOAK_REALM,
                "user_realm_name": KEYCLOAK_ADMIN_REALM,
            },
            "school_authority_name": "TestSchool",
            "mode": 2,
        }
        with mock.patch("sys.argv", sysargs):
            args = parse_cmdline()
            assert args == expected_args

    def test_cannot_use_both_force_and_dry_run(self):
        sysargs = [
            "remove-school-authority",
            "--keycloak-admin-name",
            KEYCLOAK_USER,
            "--keycloak-admin-realm",
            KEYCLOAK_ADMIN_REALM,
            "--keycloak-authority-realm",
            KEYCLOAK_REALM,
            "--keycloak-url",
            KEYCLOAK_URL,
            "--school-authority-name",
            "TestSchool",
            "--dry-run",
            "--force",
        ]
        with mock.patch("sys.argv", sysargs):
            with pytest.raises(
                SystemExit,
                match="Cannot use both --force and --dry-run flags together",
            ):
                parse_cmdline()


class TestSchoolAuthCleaner:
    """Tests for SchoolAuthCleaner"""

    SCHOOL_AUTHORITY = "SACTestSchool1"
    DEFAULT_CONFIG = {
        "keycloak_config": {
            "server_url": KEYCLOAK_URL,
            "username": KEYCLOAK_USER,
            "password": KEYCLOAK_PASSWORD,
            "realm_name": KEYCLOAK_REALM,
            "user_realm_name": KEYCLOAK_ADMIN_REALM,
        },
        "school_authority_name": SCHOOL_AUTHORITY,
    }

    def test_init(self):
        sac = SchoolAuthCleaner(**self.DEFAULT_CONFIG)
        assert sac.keycloak_config == self.DEFAULT_CONFIG["keycloak_config"]
        assert sac.school_authority_name == self.DEFAULT_CONFIG["school_authority_name"]
        assert sac.ldap_base == ucr.get("ldap/base")
        assert sac.mode == 0
        assert sac._keycloak_connection is None
        assert sac._udm_connection is None
        assert sac._ldap_connection is None

    def test_keycloak_connection(self):
        sac = SchoolAuthCleaner(**self.DEFAULT_CONFIG)
        assert sac._keycloak_connection is None

        # Initialize keycloak connection
        keycloak_connection = sac.keycloak_connection
        assert keycloak_connection is not None
        assert sac._keycloak_connection == keycloak_connection

        # It should not initialize it multiple times
        keycloak_connection2 = sac.keycloak_connection
        assert keycloak_connection == keycloak_connection2

    def test_keycloak_connection_initialization_with_auth_errors(self):
        config = deepcopy(self.DEFAULT_CONFIG)
        config["keycloak_config"]["password"] = "badpassword"
        sac = SchoolAuthCleaner(**config)

        with pytest.raises(
            SystemExit,
            match="ERROR: Invalid Keycloak credentials",
        ):
            sac.keycloak_connection

    def test_keycloak_connection_initialization_with_errors(self):
        old_get_keycloak_connection = remove_school_authority.get_keycloak_connection
        mock_get_keycloak_connection = mock.Mock(side_effect=Exception("test error"))
        remove_school_authority.get_keycloak_connection = mock_get_keycloak_connection

        try:
            sac = SchoolAuthCleaner(**self.DEFAULT_CONFIG)
            with pytest.raises(
                SystemExit,
                match="ERROR: Keycloak connection failed: test error",
            ):
                sac.keycloak_connection
        finally:
            remove_school_authority.get_keycloak_connection = (
                old_get_keycloak_connection
            )

    def test_ldap_connection(self):
        sac = SchoolAuthCleaner(**self.DEFAULT_CONFIG)
        assert sac._ldap_connection is None

        # Initialize ldap connection
        ldap_connection = sac.ldap_connection
        assert ldap_connection is not None
        assert sac._ldap_connection == ldap_connection

        # It should not initialize it multiple times
        ldap_connection2 = sac.ldap_connection
        assert ldap_connection == ldap_connection2

    def test_lo_initialization_with_errors(self):
        old_get_ldap_admin_connection = (
            remove_school_authority.get_ldap_admin_connection
        )
        mock_get_ldap_admin_connection = mock.Mock(side_effect=Exception("test error"))
        remove_school_authority.get_ldap_admin_connection = (
            mock_get_ldap_admin_connection
        )

        try:
            sac = SchoolAuthCleaner(**self.DEFAULT_CONFIG)
            with pytest.raises(
                SystemExit,
                match="ERROR: Could not get ldap admin connection: test error",
            ):
                sac.ldap_connection
        finally:
            remove_school_authority.get_ldap_admin_connection = (
                old_get_ldap_admin_connection
            )

    def test_udm_connection(self):
        sac = SchoolAuthCleaner(**self.DEFAULT_CONFIG)
        assert sac._udm_connection is None

        # Initialize udm connection
        udm_connection = sac.udm_connection
        assert udm_connection is not None
        assert sac._udm_connection == udm_connection

        # It should not initialize it multiple times
        udm_connection2 = sac.udm_connection
        assert udm_connection == udm_connection2

    def test_udm_connection_initialization_with_errors(self):
        old_get_udm_connection = remove_school_authority.get_udm_connection
        mock_get_udm_connection = mock.Mock(side_effect=Exception("test error"))
        remove_school_authority.get_udm_connection = mock_get_udm_connection

        try:
            sac = SchoolAuthCleaner(**self.DEFAULT_CONFIG)
            with pytest.raises(
                SystemExit,
                match="ERROR: UDM connection failed: test error",
            ):
                sac.udm_connection
        finally:
            remove_school_authority.get_udm_connection = old_get_udm_connection

    def test_error(self):
        base_message = "This is a test"
        expected_message = f"ERROR: {base_message}"
        sac = SchoolAuthCleaner(**self.DEFAULT_CONFIG)
        assert sac._error(base_message) == expected_message

    def test_error_with_exception(self):
        base_message = "This is also a test"
        base_exception = "Horrible things happened!"
        expected_message = f"ERROR: {base_message}: {base_exception}"
        sac = SchoolAuthCleaner(**self.DEFAULT_CONFIG)
        assert sac._error(base_message, Exception(base_exception)) == expected_message

    def test_print_error(self):
        old_user_output = remove_school_authority.user_output
        mock_user_output = mock.MagicMock()
        remove_school_authority.user_output = mock_user_output

        try:
            base_message = "This is a test: foo bar"
            expected_message = (
                f"{remove_school_authority.TermColors.ERROR}",
                f"{base_message}",
                f"{remove_school_authority.TermColors.ENDC}",
            )
            sac = SchoolAuthCleaner(**self.DEFAULT_CONFIG)
            sac._print_error(base_message)
            assert mock_user_output.called_once_with(expected_message)
        finally:
            remove_school_authority.user_output = old_user_output

    def test_wrap_logged_output(self):
        old_user_output = remove_school_authority.user_output
        mock_user_output = mock.MagicMock()
        remove_school_authority.user_output = mock_user_output

        try:
            mock_func = mock.MagicMock()
            sac = SchoolAuthCleaner(**self.DEFAULT_CONFIG)
            sac._wrap_logged_output(mock_func, "foo", "bar", baz=1)

            assert mock_user_output.called_with(
                remove_school_authority.TermColors.LOG_OUTPUT,
                end="",
            )
            assert mock_func.called_once_with("foo", "bar", baz=1)
            assert mock_user_output.called_with(
                remove_school_authority.TermColors.ENDC,
                end="",
            )
        finally:
            remove_school_authority.user_output = old_user_output

    def test_wrap_logged_output_still_prints_closing_tag_if_error(self):
        old_user_output = remove_school_authority.user_output
        mock_user_output = mock.MagicMock()
        remove_school_authority.user_output = mock_user_output

        try:
            mock_func = mock.Mock(side_effect=Exception("TEST!"))
            sac = SchoolAuthCleaner(**self.DEFAULT_CONFIG)

            try:
                sac._wrap_logged_output(mock_func, "foo", "bar", baz=1)
            except Exception:
                assert mock_user_output.called_with(
                    remove_school_authority.TermColors.LOG_OUTPUT,
                    end="",
                )
                assert mock_func.called_once_with("foo", "bar", baz=1)
                assert mock_user_output.called_with(
                    remove_school_authority.TermColors.ENDC,
                    end="",
                )
        finally:
            remove_school_authority.user_output = old_user_output

    def test_list_identity_providers_integration_test(self):
        sac = SchoolAuthCleaner(**self.DEFAULT_CONFIG)
        idp_name = "test_list_identity_providers"
        create_idp(sac.keycloak_connection, idp_name)

        existing_idps = sac.keycloak_connection.get_idps()
        expected_idps = [idp["alias"] for idp in existing_idps]
        assert idp_name in expected_idps

        returned_idps = sac._list_identity_providers()
        assert set(expected_idps) == set(returned_idps)

    def test_get_keycloak_idp_alias(self):
        idp_name = "TestGetKeycloakIDPAlias"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name
        sac = SchoolAuthCleaner(**config)
        create_idp(sac.keycloak_connection, idp_name)
        assert sac._get_keycloak_idp_alias() == idp_name

    def test_get_keycloak_idp_alias_case_insensitive(self):
        idp_name = "TestGetKeycloakIDPAlias"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name.lower()
        sac = SchoolAuthCleaner(**config)
        create_idp(sac.keycloak_connection, idp_name)
        assert sac._get_keycloak_idp_alias() == idp_name

    def test_get_keycloak_idp_alias_not_found(self):
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = "does-not-exist"
        sac = SchoolAuthCleaner(**config)
        assert sac._get_keycloak_idp_alias() is None

    def test_get_school_authority(self):
        idp_name = "tgsa1"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name
        sac = SchoolAuthCleaner(**config)
        idp = create_idp(sac.keycloak_connection, idp_name)
        user = create_idp_user(sac.udm_connection, idp_name)
        school = create_idp_school(sac.ldap_connection, idp_name, "School2")

        school_authority = sac._get_school_authority()
        assert school_authority.identity_provider == idp
        assert school_authority.user.dn == user.dn
        assert [school.dn for school in school_authority.schools] == [school.dn]

    def test_get_school_authority_only_idp_found(self):
        idp_name = "tgsaoidpf1"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name
        sac = SchoolAuthCleaner(**config)
        idp = create_idp(sac.keycloak_connection, idp_name)

        school_authority = sac._get_school_authority()
        assert school_authority.identity_provider == idp
        assert school_authority.user is None
        assert school_authority.schools == []

    def test_get_school_authority_only_user_found(self):
        idp_name = "tgsaouserf1"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name
        sac = SchoolAuthCleaner(**config)
        user = create_idp_user(sac.udm_connection, idp_name)

        school_authority = sac._get_school_authority()
        assert school_authority.identity_provider is None
        assert school_authority.user.dn == user.dn
        assert school_authority.schools == []

    def test_get_school_authority_only_schools_found(self):
        idp_name = "tgsaoschoolsf1"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name
        sac = SchoolAuthCleaner(**config)
        school = create_idp_school(sac.ldap_connection, idp_name, "School1")

        school_authority = sac._get_school_authority()
        assert school_authority.identity_provider is None
        assert school_authority.user is None
        assert [school.dn for school in school_authority.schools] == [school.dn]

    def test_get_school_authority_nothing_found(self):
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = "does-not-exist"
        sac = SchoolAuthCleaner(**config)
        assert sac._get_school_authority() is None

    def test_delete(self):
        idp_name = "tdelete1"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name
        sac = SchoolAuthCleaner(**config)
        idp = create_idp(sac.keycloak_connection, idp_name)
        user = create_idp_user(sac.udm_connection, idp_name)
        school1 = create_idp_school(sac.ldap_connection, idp_name, "School1")
        school2 = create_idp_school(sac.ldap_connection, idp_name, "School2")
        school_authority = SchoolAuthority(
            user=user,
            schools=[school1, school2],
            identity_provider=idp,
        )

        # pre-delete checks
        assert idp_name in [idp["alias"] for idp in sac.keycloak_connection.get_idps()]
        assert school1.dn is not None
        assert school2.dn is not None
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

        sac._delete(school_authority)

        # post-delete checks
        assert idp_name not in [
            idp["alias"] for idp in sac.keycloak_connection.get_idps()
        ]
        assert school1.dn is None
        assert school2.dn is None
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
            assert False, "User not deleted"
        except NoObject:
            # user was deleted
            pass

    def test_delete_errors_during_idp_deletion(self):
        idp_name = "tdedidpdel1"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name
        sac = SchoolAuthCleaner(**config)
        idp = create_idp(sac.keycloak_connection, idp_name)
        user = create_idp_user(sac.udm_connection, idp_name)
        school1 = create_idp_school(sac.ldap_connection, idp_name, "School1")
        school2 = create_idp_school(sac.ldap_connection, idp_name, "School2")
        school_authority = SchoolAuthority(
            user=user,
            schools=[school1, school2],
            identity_provider=idp,
        )

        # pre-delete checks
        assert idp_name in [idp["alias"] for idp in sac.keycloak_connection.get_idps()]
        assert school1.dn is not None
        assert school2.dn is not None
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

        old_keycloak_connection = sac.keycloak_connection
        mock_delete_idp = mock.Mock(side_effect=Exception("IDP deletion error"))
        sac._keycloak_connection = mock.MagicMock()
        sac._keycloak_connection.delete_idp = mock_delete_idp
        with pytest.raises(SystemExit, match="1"):
            sac._delete(school_authority)
        sac._keycloak_connection = old_keycloak_connection

        # Script should still delete what it can
        assert school1.dn is None
        assert school2.dn is None
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
            assert False, "User not deleted"
        except NoObject:
            # user was deleted
            pass

        # We can still find what wasn't deleted
        assert idp_name in [idp["alias"] for idp in sac.keycloak_connection.get_idps()]

    def test_delete_errors_during_user_deletion(self):
        idp_name = "tdeduserdel1"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name
        sac = SchoolAuthCleaner(**config)
        idp = create_idp(sac.keycloak_connection, idp_name)
        user = create_idp_user(sac.udm_connection, idp_name)
        school1 = create_idp_school(sac.ldap_connection, idp_name, "School1")
        school2 = create_idp_school(sac.ldap_connection, idp_name, "School2")
        school_authority = SchoolAuthority(
            user=user,
            schools=[school1, school2],
            identity_provider=idp,
        )

        # pre-delete checks
        assert idp_name in [idp["alias"] for idp in sac.keycloak_connection.get_idps()]
        assert school1.dn is not None
        assert school2.dn is not None
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

        old_user_delete = user.delete
        user.delete = mock.Mock(side_effect=Exception("User deletion error"))
        with pytest.raises(SystemExit, match="1"):
            sac._delete(school_authority)
        user.delete = old_user_delete

        # Script should still delete what it can
        assert idp_name not in [
            idp["alias"] for idp in sac.keycloak_connection.get_idps()
        ]
        assert school1.dn is None
        assert school2.dn is None

        # We can still find what wasn't deleted
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

    def test_delete_errors_during_school_deletion(self):
        idp_name = "tdedschooldel1"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name
        sac = SchoolAuthCleaner(**config)
        idp = create_idp(sac.keycloak_connection, idp_name)
        user = create_idp_user(sac.udm_connection, idp_name)
        school1 = create_idp_school(sac.ldap_connection, idp_name, "School1")
        school2 = create_idp_school(sac.ldap_connection, idp_name, "School2")
        school_authority = SchoolAuthority(
            user=user,
            schools=[school1, school2],
            identity_provider=idp,
        )

        # pre-delete checks
        assert idp_name in [idp["alias"] for idp in sac.keycloak_connection.get_idps()]
        assert school1.dn is not None
        assert school2.dn is not None
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

        old_school1_remove = school1.remove
        school1.remove = mock.Mock(side_effect=Exception("School removal error"))
        with pytest.raises(SystemExit, match="1"):
            sac._delete(school_authority)
        school1.remove = old_school1_remove

        # Script should still delete what it can
        assert idp_name not in [
            idp["alias"] for idp in sac.keycloak_connection.get_idps()
        ]
        assert school2.dn is None
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
            assert False, "User not deleted"
        except NoObject:
            # user was deleted
            pass

        # We can still find what wasn't deleted
        assert school1.dn is not None

    def test_delete_user_ctrl_c(self):
        idp_name = "tdctrlc1"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name
        sac = SchoolAuthCleaner(**config)
        idp = create_idp(sac.keycloak_connection, idp_name)
        user = create_idp_user(sac.udm_connection, idp_name)
        school1 = create_idp_school(sac.ldap_connection, idp_name, "School1")
        school2 = create_idp_school(sac.ldap_connection, idp_name, "School2")
        school_authority = SchoolAuthority(
            user=user,
            schools=[school1, school2],
            identity_provider=idp,
        )

        # pre-delete checks
        assert idp_name in [idp["alias"] for idp in sac.keycloak_connection.get_idps()]
        assert school1.dn is not None
        assert school2.dn is not None
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

        old_keycloak_connection = sac.keycloak_connection
        mock_delete_idp = mock.Mock(side_effect=KeyboardInterrupt())
        sac._keycloak_connection = mock.MagicMock()
        sac._keycloak_connection.delete_idp = mock_delete_idp
        # This shouldn't throw an error, but nothing will get deleted
        sac._delete(school_authority)
        sac._keycloak_connection = old_keycloak_connection

        # post-delete checks
        assert idp_name in [idp["alias"] for idp in sac.keycloak_connection.get_idps()]
        assert school1.dn is not None
        assert school2.dn is not None
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

    def test_clean(self):
        idp_name = "tclean1"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name
        sac = SchoolAuthCleaner(**config)
        create_idp(sac.keycloak_connection, idp_name)
        user = create_idp_user(sac.udm_connection, idp_name)
        school1 = create_idp_school(sac.ldap_connection, idp_name, "School1")
        school2 = create_idp_school(sac.ldap_connection, idp_name, "School2")

        # pre-delete checks
        assert idp_name in [idp["alias"] for idp in sac.keycloak_connection.get_idps()]
        try:
            School.from_dn(school1.dn, None, sac.ldap_connection)
        except AdminNoObject:
            assert False, "School1 doesn't exist"
        try:
            School.from_dn(school2.dn, None, sac.ldap_connection)
        except AdminNoObject:
            assert False, "School2 doesn't exist"
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

        old_user_input = remove_school_authority.user_input
        mock_user_input = mock.MagicMock()
        mock_user_input.return_value = "y"
        remove_school_authority.user_input = mock_user_input
        try:
            sac.clean()
        finally:
            remove_school_authority.user_input = old_user_input

        # everything should have been deleted
        assert idp_name not in [
            idp["alias"] for idp in sac.keycloak_connection.get_idps()
        ]
        try:
            School.from_dn(school1.dn, None, sac.ldap_connection)
            assert False, "School1 still exists"
        except AdminNoObject:
            # School1 was deleted
            pass
        try:
            School.from_dn(school2.dn, None, sac.ldap_connection)
            assert False, "School2 still exists"
        except AdminNoObject:
            # School2 was deleted
            pass
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
            assert False, "User not deleted"
        except NoObject:
            # user was deleted
            pass

    def test_clean_user_aborts(self):
        idp_name = "tcleanua1"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name
        sac = SchoolAuthCleaner(**config)
        create_idp(sac.keycloak_connection, idp_name)
        user = create_idp_user(sac.udm_connection, idp_name)
        school1 = create_idp_school(sac.ldap_connection, idp_name, "School1")
        school2 = create_idp_school(sac.ldap_connection, idp_name, "School2")

        # pre-delete checks
        assert idp_name in [idp["alias"] for idp in sac.keycloak_connection.get_idps()]
        try:
            School.from_dn(school1.dn, None, sac.ldap_connection)
        except AdminNoObject:
            assert False, "School1 doesn't exist"
        try:
            School.from_dn(school2.dn, None, sac.ldap_connection)
        except AdminNoObject:
            assert False, "School2 doesn't exist"
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

        old_user_input = remove_school_authority.user_input
        mock_user_input = mock.MagicMock()
        mock_user_input.return_value = "n"
        remove_school_authority.user_input = mock_user_input
        try:
            sac.clean()
        finally:
            remove_school_authority.user_input = old_user_input

        # nothing should have been deleted
        assert idp_name in [idp["alias"] for idp in sac.keycloak_connection.get_idps()]
        try:
            School.from_dn(school1.dn, None, sac.ldap_connection)
        except AdminNoObject:
            assert False, "School1 was deleted"
        try:
            School.from_dn(school2.dn, None, sac.ldap_connection)
        except AdminNoObject:
            assert False, "School2 was deleted"
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

    def test_clean_nothing_to_delete(self):
        sac = SchoolAuthCleaner(**self.DEFAULT_CONFIG)
        # We shouldn't need to patch the deletion confirmation,
        # so this should just run
        sac.clean()

    def test_clean_dry_run(self):
        idp_name = "tcleandr1"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name
        config["mode"] = SchoolAuthCleaner.DRY_RUN_MODE
        sac = SchoolAuthCleaner(**config)
        create_idp(sac.keycloak_connection, idp_name)
        user = create_idp_user(sac.udm_connection, idp_name)
        school1 = create_idp_school(sac.ldap_connection, idp_name, "School1")
        school2 = create_idp_school(sac.ldap_connection, idp_name, "School2")

        # pre-delete checks
        assert idp_name in [idp["alias"] for idp in sac.keycloak_connection.get_idps()]
        try:
            School.from_dn(school1.dn, None, sac.ldap_connection)
        except AdminNoObject:
            assert False, "School1 doesn't exist"
        try:
            School.from_dn(school2.dn, None, sac.ldap_connection)
        except AdminNoObject:
            assert False, "School2 doesn't exist"
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

        # We shouldn't need to patch the deletion confirmation,
        # so this should just run
        sac.clean()

        # nothing should have been deleted
        assert idp_name in [idp["alias"] for idp in sac.keycloak_connection.get_idps()]
        try:
            School.from_dn(school1.dn, None, sac.ldap_connection)
        except AdminNoObject:
            assert False, "School1 was deleted"
        try:
            School.from_dn(school2.dn, None, sac.ldap_connection)
        except AdminNoObject:
            assert False, "School2 was deleted"
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

    def test_clean_force(self):
        idp_name = "tcleanforce1"
        config = deepcopy(self.DEFAULT_CONFIG)
        config["school_authority_name"] = idp_name
        config["mode"] = SchoolAuthCleaner.FORCE_MODE
        sac = SchoolAuthCleaner(**config)
        create_idp(sac.keycloak_connection, idp_name)
        user = create_idp_user(sac.udm_connection, idp_name)
        school1 = create_idp_school(sac.ldap_connection, idp_name, "School1")
        school2 = create_idp_school(sac.ldap_connection, idp_name, "School2")

        # pre-delete checks
        assert idp_name in [idp["alias"] for idp in sac.keycloak_connection.get_idps()]
        try:
            School.from_dn(school1.dn, None, sac.ldap_connection)
        except AdminNoObject:
            assert False, "School1 doesn't exist"
        try:
            School.from_dn(school2.dn, None, sac.ldap_connection)
        except AdminNoObject:
            assert False, "School2 doesn't exist"
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
        except NoObject:
            assert False, "User does not exist"

        # We shouldn't need to patch the deletion confirmation,
        # so this should just run
        sac.clean()

        # everything should have been deleted
        assert idp_name not in [
            idp["alias"] for idp in sac.keycloak_connection.get_idps()
        ]
        try:
            School.from_dn(school1.dn, None, sac.ldap_connection)
            assert False, "School1 still exists"
        except AdminNoObject:
            # School1 was deleted
            pass
        try:
            School.from_dn(school2.dn, None, sac.ldap_connection)
            assert False, "School2 still exists"
        except AdminNoObject:
            # School2 was deleted
            pass
        try:
            sac.udm_connection.get("users/user").get(dn=user.dn)
            assert False, "User not deleted"
        except NoObject:
            # user was deleted
            pass
