import logging

import httpx

from ucsschool.apis.utils import get_logger

from .sddb_client import APIQueueEntryType, SdDBRestClient

logger: logging.Logger = get_logger()


class SdDBRestClientImpl(SdDBRestClient):
    append_url_tmpl = (
        "https://{host}/ucsschool/id-broker-sddb-builder/v1/append/{queue}"
    )
    queue = "high"

    def __init__(self, username: str, password: str, host: str):
        self.basic_auth = httpx.BasicAuth(username=username, password=password)
        self.host = host
        self.append_url = self.append_url_tmpl.format(host=self.host, queue=self.queue)

    def enqueue(
        self,
        service_provider: str,
        entry_type: APIQueueEntryType,
        pseudonym: str = None,
        entry_uuid: str = None,
    ) -> None:
        if pseudonym:
            data = {
                "pseudonym": pseudonym,
                "service_provider": service_provider,
                "entry_type": entry_type.value,
            }
        elif entry_uuid:
            data = {
                "entry_uuid": entry_uuid,
                "entry_type": entry_type.value,
            }
        else:
            logger.error(
                "Either entry_uuid or pseudonym have to be passed to the sddb API."
            )
            return
        resp = httpx.post(url=self.append_url, auth=self.basic_auth, json=data)
        if resp.status_code == 201:
            logger.info(
                "[%d] Added to SDDB %r priority queue: %r.",
                resp.status_code,
                self.queue,
                data,
            )
        else:
            logger.error(
                "[%d] Sending %r to %r: %s",
                resp.status_code,
                data,
                self.append_url,
                resp.text,
            )
