import logging
import os
import re
import time
from functools import lru_cache
from typing import Dict, List, Sequence, Set, Tuple, Union

from fastapi import Depends, HTTPException, status
from ldap3.utils.dn import to_dn

from id_broker_common.exceptions import (
    IdBrokerPseudonymisationDataError,
    IdBrokerPseudonymisationNoMapping,
)
from id_broker_common.utils import load_settings
from ucsschool.apis.models import OAuth2Token
from ucsschool.apis.utils import auth_manager, get_logger

from .models import BasicUser, GroupMembers, LegalGuardian, Student, Teacher
from .redis_sddb import RedisSdDBClient
from .rest_sddb import SdDBRestClientImpl
from .sddb_client import (
    APIQueueEntryType,
    NotFoundError,
    SdDBClient,
    SddbGroup,
    SddbInternalUser,
    SdDBRestClient,
    SddbUser,
)
from .settings import PLUGIN_SETTINGS_FILE

logger: logging.Logger = get_logger()


# Extracted to variable to allow dep overrides in FastAPI, which cannot resolve anonymous dependencies
token_dep = auth_manager("self_disclosure", OAuth2Token)

_service_provider_mapping_cache: Dict[str, str] = {}


def username_from_user(user: Union[SddbUser, SddbInternalUser]) -> str:
    if isinstance(user, SddbUser):
        firstname = user.data["firstname"]
        lastname = user.data["lastname"]
    else:
        firstname = user["data"]["firstname"]
        lastname = user["data"]["lastname"]
    return f"{firstname}.{lastname}"


def service_provider_from_token(token: OAuth2Token = Depends(token_dep)) -> str:
    return token.client_id


def connecting_user_pseudonym_from_token(
    token: OAuth2Token = Depends(token_dep),
) -> str:
    return token.sub


@lru_cache(maxsize=1024)
def role_split(ucsschool_role: str) -> Tuple[str, str, str]:
    """Cached: same ucsschool_role string results in same tuple."""
    role, context_type, context_name = ucsschool_role.split(":")
    return role, context_type, context_name


@lru_cache(maxsize=1024)
def roles_in_ucsschool_roles(
    ucsschool_roles: Tuple[str, ...], school_name: str
) -> Set[str]:
    """Cached: same list of ucsschool_roles strings (hashable->sorted tuple) returns same set of roles."""
    return {
        role
        for role, context_type, context_name in [
            role_split(ucsschool_role) for ucsschool_role in ucsschool_roles
        ]
        if context_type == "school" and school_name == context_name
    }


def roles_in_school(ucsschool_roles: Sequence[str], school_name: str) -> Set[str]:
    """Helper to get only the roles that are defined for a user within a specific school"""
    try:
        return roles_in_ucsschool_roles(tuple(sorted(ucsschool_roles)), school_name)
    except ValueError:
        logger.error("Bad value in 'ucsschool_roles': %r", ucsschool_roles)
        return set()


def extract_members(group: SddbGroup, service_provider: str) -> GroupMembers:
    """
    Helper to get users of a group split into teachers, students and legal-guardians

    :raises IdBrokerPseudonymisationDataError: when `service_provider` does not exist in backend or the
        content of the backends data is not valid / unexpected
    :raises IdBrokerPseudonymisationNoMapping: when `service_provider` does not exist in the (valid)
        service provider mapping backend data
    """
    teachers = []
    students = []
    legal_guardians = []
    sp_attribute = get_service_provider_attribute(service_provider)
    sp_id = service_provider_attribute_2_id(sp_attribute)
    t0 = time.time()
    sddb = sddb_client()
    redis_internal_students: List[SddbInternalUser] = sddb.get_users(group.students)
    redis_internal_teachers: List[SddbInternalUser] = sddb.get_users(group.teachers)
    redis_internal_legal_guardians: List[SddbInternalUser] = sddb.get_users(
        group.legal_guardians
    )
    logger.debug(
        "Retrieved %d members in %0.3f sec from Redis.",
        len(redis_internal_students) + len(redis_internal_teachers),
        time.time() - t0,
    )
    for index, redis_internal_student in enumerate(redis_internal_students):
        if not redis_internal_student:
            logger.debug(
                "Member (student) %r not found in SDDB.", group.students[index]
            )
            sddb_rest_client().enqueue(
                service_provider=service_provider,
                pseudonym=group.pseudonym,
                entry_type=APIQueueEntryType.group,
            )
            continue
        students.append(
            Student(
                user_id=redis_internal_student[f"p{sp_id:04d}"],
                username=username_from_user(redis_internal_student),
                firstname=redis_internal_student["data"]["firstname"],
                lastname=redis_internal_student["data"]["lastname"],
                legal_guardians=convert_pks_to_basic_users(
                    group.pseudonym,
                    redis_internal_student["legal_guardians"],
                    service_provider,
                ),
            )
        )
    for index, redis_internal_teacher in enumerate(redis_internal_teachers):
        if not redis_internal_teacher:
            logger.debug(
                "Member (teacher) %r not found in SDDB.", group.teachers[index]
            )
            sddb_rest_client().enqueue(
                service_provider=service_provider,
                pseudonym=group.pseudonym,
                entry_type=APIQueueEntryType.group,
            )
            continue
        teachers.append(
            Teacher(
                user_id=redis_internal_teacher[f"p{sp_id:04d}"],
                username=username_from_user(redis_internal_teacher),
            )
        )
    for index, redis_internal_legal_guardian in enumerate(
        redis_internal_legal_guardians
    ):
        if not redis_internal_legal_guardian:
            logger.debug(
                "Member (legal_guardian) %r not found in SDDB.",
                group.legal_guardians[index],
            )
            sddb_rest_client().enqueue(
                service_provider=service_provider,
                pseudonym=group.pseudonym,
                entry_type=APIQueueEntryType.group,
            )
            continue
        legal_guardians.append(
            LegalGuardian(
                user_id=redis_internal_legal_guardian[f"p{sp_id:04d}"],
                username=username_from_user(redis_internal_legal_guardian),
                firstname=redis_internal_legal_guardian["data"]["firstname"],
                lastname=redis_internal_legal_guardian["data"]["lastname"],
                legal_wards=convert_pks_to_basic_users(
                    group.pseudonym,
                    redis_internal_legal_guardian["legal_wards"],
                    service_provider,
                ),
            )
        )
    return GroupMembers(
        students=students, teachers=teachers, legal_guardians=legal_guardians
    )


def get_user(pseudonym: str, service_provider: str) -> SddbUser:
    """
    Retrieve user.

    :param str pseudonym: a user's pseudonym
    :param str service_provider: the service provider for which to search user with the given pseudonym
    :return: SddbUser object
    :raises HTTPException: 404 if not found
    :raises IdBrokerPseudonymisationDataError: when `service_provider` does not exist in backend or the
        content of the backends data is not valid / unexpected
    :raises IdBrokerPseudonymisationNoMapping: when `service_provider` does not exist in the (valid)
        service provider mapping backend data
    """
    sp_attribute = get_service_provider_attribute(service_provider)
    sp_id = service_provider_attribute_2_id(sp_attribute)
    t0 = time.time()
    try:
        user = sddb_client().get_user(pseudonym=pseudonym, service_provider_id=sp_id)
    except NotFoundError:
        sddb_rest_client().enqueue(
            service_provider=service_provider,
            pseudonym=pseudonym,
            entry_type=APIQueueEntryType.user,
        )
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail=f"No user was found with pseudonym {pseudonym!r} for service provider "
            f"{service_provider!r}.",
        )
    logger.debug("Retrieved user in %0.3f sec from Redis.", time.time() - t0)
    return user


def get_connected_user(
    service_provider: str = Depends(service_provider_from_token),
    pseudonym_of_connecting_user: str = Depends(connecting_user_pseudonym_from_token),
) -> SddbUser:
    logger.debug(
        "service_provider=%r pseudonym_of_connecting_user=%r",
        service_provider,
        pseudonym_of_connecting_user,
    )
    try:
        return get_user(pseudonym_of_connecting_user, service_provider)
    except (
        HTTPException,
        IdBrokerPseudonymisationDataError,
        IdBrokerPseudonymisationNoMapping,
    ) as exc:
        raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED) from exc


def convert_pks_to_basic_users(
    pseudonym: str, pks: list[str], service_provider: str
) -> list[BasicUser]:
    if not pks:
        return []
    sp_attribute = get_service_provider_attribute(service_provider)
    sp_id = service_provider_attribute_2_id(sp_attribute)
    t0 = time.time()
    users: List[SddbInternalUser] = sddb_client().get_users(pks)
    if None in users:
        sddb_rest_client().enqueue(
            service_provider=service_provider,
            pseudonym=pseudonym,
            entry_type=APIQueueEntryType.user,
        )
        logger.warning(
            f"Could not find a connected user of the user with pseudonym {pseudonym!r} from service {service_provider!r}."
        )
    logger.debug("Retrieved users in %0.3f sec from Redis.", time.time() - t0)
    return [
        BasicUser(
            user_id=user[f"p{sp_id:04d}"],
            username=username_from_user(user),
            firstname=user["data"]["firstname"],
            lastname=user["data"]["lastname"],
        )
        for user in users
        if user is not None
    ]


def get_group_by_pseudonym(pseudonym: str, service_provider: str) -> SddbGroup:
    """
    Retrieve school group using a service provider specific pseudonym.

    :param str pseudonym: a school group pseudonym
    :param str service_provider: the service provider for which to search group with the given pseudonym
    :return: SddbGroup object
    :raises NotFoundError: if not found
    :raises IdBrokerPseudonymisationDataError: when `service_provider` does not exist in backend or the
        content of the backends data is not valid / unexpected
    :raises IdBrokerPseudonymisationNoMapping: when `service_provider` does not exist in the (valid)
        service provider mapping backend data
    """
    sp_attribute = get_service_provider_attribute(service_provider)
    sp_id = service_provider_attribute_2_id(sp_attribute)
    t0 = time.time()
    group = sddb_client().get_group_by_pseudonym(
        pseudonym=pseudonym, service_provider_id=sp_id
    )
    logger.debug("Retrieved group in %0.3f sec from Redis.", time.time() - t0)
    return group


def get_group_by_id(group_id: str, service_provider: str) -> SddbGroup:
    """
    Retrieve school group using the groups entryUUID.

    :param str group_id: the school group entryUUID in LDAP
    :param str service_provider: the service provider requesting the group data
    :return: SddbGroup object
    :raises NotFoundError: if not found
    :raises IdBrokerPseudonymisationDataError: when `service_provider` does not exist in backend or the
        content of the backends data is not valid / unexpected
    :raises IdBrokerPseudonymisationNoMapping: when `service_provider` does not exist in the (valid)
        service provider mapping backend data
    """
    sp_attribute = get_service_provider_attribute(service_provider)
    sp_id = service_provider_attribute_2_id(sp_attribute)
    t0 = time.time()
    group = sddb_client().get_group_by_id(group_id=group_id, service_provider_id=sp_id)
    logger.debug("Retrieved group in %0.3f sec from Redis.", time.time() - t0)
    return group


@lru_cache(maxsize=1)
def sddb_client() -> SdDBClient:
    return RedisSdDBClient()


@lru_cache(maxsize=1)
def sddb_rest_client() -> SdDBRestClient:
    username = to_dn(os.environ["LDAP_HOSTDN"], decompose=True)[0][1]
    username = f"{username}$"
    with open("/etc/machine.secret") as fp:
        password = fp.read().strip()
    settings = load_settings(PLUGIN_SETTINGS_FILE)
    host = settings["sddb_rest_host"]
    return SdDBRestClientImpl(username=username, password=password, host=host)


@lru_cache(maxsize=128)
def service_provider_attribute_2_id(sp: str) -> int:
    """'idBrokerPseudonym0002' -> 2"""
    return int(re.match(r"^\w+(\d\d\d\d)$", sp).groups()[0])


def retrieve_service_provider_mapping() -> Dict[str, str]:
    """
    :raises IdBrokerPseudonymisationDataError: when service provider mapping was not found in backend.
    """
    t0 = time.time()
    try:
        spm = sddb_client().get_service_provider_mapping()
    except NotFoundError as exc:
        raise IdBrokerPseudonymisationDataError(
            f"Retrieving service provider mapping through SDDB: {exc!s}"
        ) from exc
    logger.debug(
        "Retrieved service provider mapping in %0.3f sec from Redis.", time.time() - t0
    )
    return spm.mapping


def get_service_provider_attribute(service_provider: str) -> str:
    """
    Get LDAP/UDM attribute name for pseudonyms for service provider.

    Only successful lookups will be cached, so the admin can later add a service provider and the API
    server will not have to be restarted.

    :raises IdBrokerPseudonymisationDataError: when data does not exist in backend or the content of the
        backends data is not valid / unexpected
    :raises IdBrokerPseudonymisationNoMapping: when `service_provider` does not exist in the (valid)
        service provider mapping backend data
    """
    if service_provider not in _service_provider_mapping_cache:
        service_provider_mapping = retrieve_service_provider_mapping()
        _service_provider_mapping_cache.clear()
        _service_provider_mapping_cache.update(service_provider_mapping)
        if service_provider not in service_provider_mapping:
            sddb_rest_client().enqueue(
                service_provider="",
                pseudonym="",
                entry_type=APIQueueEntryType.sp_mapping,
            )
            raise IdBrokerPseudonymisationNoMapping(
                f"No attribute mapping for SP {service_provider!r} found. Known mappings: "
                f"{_service_provider_mapping_cache!r}"
            )
    return _service_provider_mapping_cache[service_provider]
