import zlib

from fastapi import (
    APIRouter,
    BackgroundTasks,
    Depends,
    HTTPException,
    Path,
    Response,
    status,
)
from pydantic import ValidationError

from id_broker_common.kelvin import (
    IDBrokerKelvinSession,
    kelvin_on_primary_session_kwargs,
    kelvin_session,
)
from id_broker_common.ldap_access import ldap_access, wait_for_replication
from provisioning_plugin.dependencies import (
    check_for_authority_admin,
    find_school_name_by_id_in_primary_ldap,
    get_school_by_id,
)
from provisioning_plugin.models import NonEmptyStr, NoStarStr, School
from provisioning_plugin.settings import PLUGIN_SETTINGS_FILE
from provisioning_plugin.utils import (
    get_pseudonyms_for_providers,
    remove_school_objects,
)
from ucsschool.kelvin.client import InvalidRequest, School as KelvinSchool, Session

EXAM_GROUP_DN = "cn=OU{ou}-Klassenarbeit,cn=ucsschool,cn=groups,{ldap_base}"

router = APIRouter(tags=["schools"], dependencies=[Depends(check_for_authority_admin)])


@router.head("/schools/{id}")
async def head(
    school_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    ou_name: str = Depends(
        find_school_name_by_id_in_primary_ldap
    ),  # raises 404 if not found
):
    return Response(status_code=status.HTTP_200_OK)


@router.get("/schools/{id}", response_model=School)
async def get(
    school_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    school: KelvinSchool = Depends(get_school_by_id),  # raises 404 if not found
) -> School:
    try:
        return School.without_prefix(
            School.from_kelvin_school(school.as_dict()), school_authority
        )
    except (ValidationError, ValueError) as exc:
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"The school {school_id!r} in school_authority {school_authority} is malformed.",
        ) from exc


@router.post("/schools", response_model=School, status_code=201)
async def post(
    school_data: School,
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
):
    school_data = school_data.with_prefix(school_data, school_authority)
    # Avoid automatic generation of dc_name by setting it explicitly.
    # The values are not used by the ID-Broker.
    # Schools of the same authority have the same dc_name.
    dc_name = f"dc-{zlib.crc32(school_authority.encode('UTF-8'))}"
    # Open a new session to set a custom timeout, because creating a new school takes time

    async with IDBrokerKelvinSession(
        **kelvin_on_primary_session_kwargs(PLUGIN_SETTINGS_FILE), timeout=120
    ) as session:
        school = KelvinSchool(
            name=school_data.name,
            display_name=school_data.display_name,
            home_share_file_server=dc_name,
            class_share_file_server=dc_name,
            educational_servers=[dc_name],
            udm_properties={
                "ucsschoolRecordUID": school_data.id,
                "ucsschoolSourceUID": school_authority,
            },
            session=session,
        )
        async for udm_prop, pseudonym in get_pseudonyms_for_providers(
            school_data.id, school_authority
        ):
            school.udm_properties[udm_prop] = pseudonym
        try:
            await school.save()
        except InvalidRequest as exc:
            raise HTTPException(status_code=exc.status, detail=exc.reason) from exc
        ldap = ldap_access()
        await wait_for_replication(ldap, school.dn)
        # The exam group (Klassenarbeit) is the last object to be created in
        # ucsschool.lib.models.school.School.call_hooks("create", "post").
        await wait_for_replication(
            ldap,
            EXAM_GROUP_DN.format(ou=school.name, ldap_base=ldap.settings.ldap_base),
        )
        return School.without_prefix(
            School.from_kelvin_school(school.as_dict()), school_authority
        )


@router.delete("/schools/{id}", status_code=status.HTTP_204_NO_CONTENT)
async def delete(
    background_tasks: BackgroundTasks,
    school_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school: KelvinSchool = Depends(get_school_by_id),  # raises 404 if not found
    session: Session = Depends(kelvin_session),
) -> Response:
    """The school will be removed in the background."""
    background_tasks.add_task(remove_school_objects, school.name, session)
    return Response(status_code=status.HTTP_204_NO_CONTENT)
