import logging
import random
from typing import Dict, List, Tuple

from locust import events, task

from performance_tests.common import (
    ProvisioningClient,
    ResponseError,
    TestData,
    get_provisioning_api_admins,
    get_schools,
)

school_authorities: List[str] = []  # ["Traeger1", "Traeger"]
user_credentials: Dict[str, Tuple[str, str]] = {}  # [(username, password), ..]
id_broker_schools: Dict[str, Dict[str, str]] = {}  # {sch_auth: {ou: uuid, ...}}


@events.init.add_listener
def on_init(environment, **kwargs):
    data = TestData()
    school_authorities.extend(data.school_authorities)
    user_credentials.update(get_provisioning_api_admins(data))
    id_broker_schools.update(get_schools(data))


class ProvisioningAPIClient(ProvisioningClient):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.school_authority = random.choice(school_authorities)
        logging.info("Testing with school authority: %r", self.school_authority)
        self.auth_username, self.auth_password = user_credentials[self.school_authority]
        self.schools = list(id_broker_schools[self.school_authority].items())

    @task
    def school_exists(self) -> None:
        name, entry_uuid = random.choice(self.schools)
        try:
            super().school_exists(
                school_authority=self.school_authority, obj_id=entry_uuid
            )
        except ResponseError as exc:
            logging.error(
                "%s: school_authority=%r name=%r entry_uuid=%r -> %r",
                self.greenlet.name,
                self.school_authority,
                name,
                entry_uuid,
                exc,
            )
