# -*- coding: utf-8 -*-
# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.

import copy
import re
from typing import List
from unittest.mock import AsyncMock, Mock, call, mock_open, patch

import pytest
from click.testing import CliRunner
from fastapi import HTTPException

from idbroker.config_utils import CONFIG_PATH
from idbroker.manage_schools_to_sync import (
    ManageSchoolsException,
    ManageSchoolsOnIDBroker,
    add_schools,
    remove_schools,
)
from ucsschool_id_connector.models import (
    SchoolAuthorityConfiguration,
    SchoolAuthorityConfigurationPatchDocument,
)


class FakeOu:
    def __init__(self, name):
        self.name = name
        self.id = name


def fake_get_ous(schools):
    async def _func(ou):
        search_regex = ".*".join([re.escape(part) for part in ou.split("*")])
        return [FakeOu(s) for s in schools if re.match(search_regex, s)]

    return _func


@pytest.fixture(autouse=True)
def patch_env(monkeypatch):
    monkeypatch.setenv("DOCKER_HOST_NAME", "test.hostname")
    monkeypatch.setenv("ldap_base", "dc=ldap,dc=base")
    monkeypatch.setenv("ldap_server_name", "TEST-SERVER-NAME")
    monkeypatch.setenv("ldap_server_port", "789")


def school_auth_config():
    return {
        "name": "Traeger2",
        "active": True,
        "url": "https://provisioning1.broker.test/",
        "plugins": ["id_broker-users", "id_broker-groups"],
        "plugin_configs": {
            "id_broker": {
                "password": "psw",
                "schools": [],
                "username": "provisioning-Traeger2",
                "version": 1,
            }
        },
    }


@patch(
    "idbroker.manage_schools_to_sync.query_service",
    AsyncMock(return_value={"errors": "some-error"}),
)
@pytest.mark.asyncio
async def test_patch_school_authority_config_fails():
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority", logger=Mock(), force=False
    )
    with pytest.raises(ManageSchoolsException):
        await manager.patch_school_authority_config(data={})


@patch(
    "idbroker.manage_schools_to_sync.query_service",
    AsyncMock(side_effect=HTTPException(status_code=500)),
)
@pytest.mark.asyncio
async def test_patch_school_authority_config_fails_due_to_http_error():
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority", logger=Mock(), force=False
    )
    with pytest.raises(ManageSchoolsException):
        await manager.patch_school_authority_config(data={})


@patch(
    "idbroker.manage_schools_to_sync.query_service",
    AsyncMock(return_value={"result": school_auth_config()}),
)
@pytest.mark.asyncio
async def test_patch_school_authority_config():
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority", logger=Mock(), force=False
    )
    await manager.patch_school_authority_config(data=school_auth_config())


@pytest.mark.asyncio
@pytest.mark.parametrize("value", (True, False))
async def test_set_initial_sync_mode(value):
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority", logger=Mock(), force=False
    )
    manager.patch_school_authority_config = AsyncMock()
    await manager.set_initial_sync_mode(value=value)
    expected_data = {"plugin_configs": {"id_broker": {"initial_import_mode": value}}}
    manager.patch_school_authority_config.assert_called_with(data=expected_data)


def test__get_config():
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority", logger=Mock(), force=False
    )
    config = get_school_authority_patch_document(["myschool"])
    with patch(
        "idbroker.manage_schools_to_sync.read_id_broker_config"
    ) as read_id_broker_config_mock:
        read_id_broker_config_mock.configure_mock(return_value=config)
        assert manager.get_config() == config
        read_id_broker_config_mock.assert_called_with(
            config_path=CONFIG_PATH, school_authority_name="test_authority"
        )


@pytest.mark.parametrize("schools", (None, [], ["foo"], ["foo", "BAR"]))
def test__get_schools_from_config(schools):
    config = school_auth_config()
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority", logger=Mock(), force=False
    )
    if schools is None:
        del config["plugin_configs"]["id_broker"]["schools"]
    else:
        config["plugin_configs"]["id_broker"]["schools"] = schools
    manager.get_config = Mock(return_value=SchoolAuthorityConfiguration(**config))
    if schools is None:
        assert manager.get_schools_from_config() == []
    else:
        assert manager.get_schools_from_config() == schools


@pytest.mark.asyncio
async def test__restore_old_config():
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority", logger=Mock(), force=False
    )
    config = get_school_authority_patch_document(["myschool"])
    manager.get_config = Mock(return_value=config)
    manager.patch_school_authority_config = AsyncMock()
    await manager.restore_old_config(config)
    manager.patch_school_authority_config.assert_called()
    manager.patch_school_authority_config = AsyncMock(
        side_effect=ManageSchoolsException
    )
    with patch("builtins.open", mock_open()) as m, patch("json.dump"):
        await manager.restore_old_config(config)
        m.assert_called_with(f"{CONFIG_PATH}/{'test_authority'}.json", "w")


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "force,already_existing",
    [(True, False), (True, True), (False, False), (False, True)],
)
async def test_add_school_to_config(force, already_existing):
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority", logger=Mock(), force=force
    )
    school = "myschool"
    if already_existing:
        school_authority_config = get_school_authority_patch_document(schools=[school])
    else:
        school_authority_config = get_school_authority_patch_document(schools=[])
    manager.get_config = Mock(return_value=school_authority_config)
    manager.patch_school_authority_config = AsyncMock()
    if already_existing is False or force is True:
        await manager.add_school_to_config(
            school=school,
        )
        data = {"plugin_configs": {"id_broker": {"schools": [school]}}}
        manager.patch_school_authority_config.assert_called_with(data=data)
    else:
        with pytest.raises(ManageSchoolsException):
            await manager.add_school_to_config(
                school=school,
            )
        manager.patch_school_authority_config.assert_not_called()


@pytest.mark.parametrize("active", (True, False))
def test_check_config_is_active_for_initial_sync(active):
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority",
        logger=Mock(),
        num_tasks_for_initial_sync=20,
        force=False,
    )
    config = school_auth_config()
    config["plugin_configs"]["id_broker"]["schools"] = ["myschool"]
    config["active"] = active
    manager.get_config = Mock(return_value=SchoolAuthorityConfiguration(**config))
    if not active:
        with pytest.raises(ManageSchoolsException):
            manager.check_config_is_active_for_initial_sync()
    else:
        manager.check_config_is_active_for_initial_sync()


@pytest.mark.parametrize(
    "schools_filter,matching_schools,mismatch_schools",
    (
        (
            ["myschool1", "myschool2"],
            ["myschool1", "myschool2"],
            ["foo", "bar", "world"],
        ),
        (["*"], ["myschool", "foo", "bar"], []),
        (["myschool*"], ["myschool1", "myschool2", "myschool"], ["foo"]),
        (
            ["my*", "*school*", "myschool", "*"],
            ["myschool1", "myschool2", "myschool", "foo"],
            [],
        ),
    ),
)
@pytest.mark.asyncio
async def test_add_schools(schools_filter, matching_schools, mismatch_schools):
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority",
        logger=Mock(),
        num_tasks_for_initial_sync=20,
        force=False,
    )
    config = get_school_authority_patch_document(schools_filter)
    manager.get_config = Mock(return_value=config)
    manager.set_initial_sync_mode = AsyncMock()
    ManageSchoolsOnIDBroker.restore_old_config = AsyncMock()
    manager.add_school_to_config = AsyncMock()
    manager.wait_for_empty_queue = AsyncMock()
    manager.school_scheduler = AsyncMock()
    manager.check_config_is_active_for_initial_sync = Mock()
    with (
        patch("idbroker.manage_schools_to_sync.setup") as setup_mock,
        patch("idbroker.manage_schools_to_sync.sync_complete") as sync_complete_mock,
        patch(
            "idbroker.manage_schools_to_sync.get_ous",
            fake_get_ous(matching_schools + mismatch_schools),
        ),
    ):
        await manager.add_schools(schools=schools_filter)
    assert manager.num_tasks == 20
    assert manager.add_school_to_config.call_count == len(schools_filter)
    assert manager.add_school_to_config.call_args_list == [
        call(**{"school": school}) for school in schools_filter
    ]
    assert manager.set_initial_sync_mode.call_count == len(schools_filter) * 2
    assert manager.set_initial_sync_mode.call_args_list == [
        call(
            **{
                "value": True,
            }
        ),
        call(
            **{
                "value": False,
            }
        ),
    ] * len(schools_filter)
    setup_mock.assert_called_with(config_path=CONFIG_PATH)
    assert sync_complete_mock.call_count == len(matching_schools) * 2
    call_args_list = []
    for school in matching_schools:
        call_args_list.append(
            call(
                **{
                    "school": school,
                    "dry_run": False,
                    "max_workers": 20,
                    "sync_members": False,
                }
            )
        )
        call_args_list.append(
            call(
                **{
                    "school": school,
                    "dry_run": False,
                    "max_workers": 20,
                    "sync_members": True,
                }
            )
        )
    assert sync_complete_mock.call_args_list == call_args_list
    manager.wait_for_empty_queue.assert_called()


@pytest.mark.asyncio
async def test_add_school_school_does_not_exist():
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority",
        logger=Mock(),
        num_tasks_for_initial_sync=20,
        force=False,
    )
    config = get_school_authority_patch_document(["myschool"])
    manager.get_config = Mock(return_value=config)
    manager.set_initial_sync_mode = AsyncMock()
    ManageSchoolsOnIDBroker.restore_old_config = AsyncMock()
    manager.add_school_to_config = AsyncMock()
    manager.wait_for_empty_queue = AsyncMock()
    manager.school_scheduler = AsyncMock()
    manager.check_config_is_active_for_initial_sync = Mock(return_value=True)
    school_name = "nonexistingschool"
    with (
        patch("idbroker.manage_schools_to_sync.setup"),
        patch("idbroker.manage_schools_to_sync.sync_complete"),
        patch("idbroker.manage_schools_to_sync.get_ous", return_value=[]),
    ):
        with pytest.raises(ManageSchoolsException):
            await manager.add_schools(schools=[school_name])


class MySpecialException(Exception): ...


def test_add_schools_script():
    runner = CliRunner()
    with (
        patch("idbroker.manage_schools_to_sync.setup_logging"),
        patch.object(ManageSchoolsOnIDBroker, "add_schools") as add_school_mock,
    ):
        result = runner.invoke(
            add_schools,
            [
                "--num_tasks",
                2,
                "--school_authority",
                "test-authority",
                "school1",
                "school2",
            ],
        )
        add_school_mock.assert_called_with(schools=("school1", "school2"))
    assert not result.exception


@pytest.mark.asyncio
@pytest.mark.parametrize("force", (True, False), ids=lambda x: f"force-{x}")
@pytest.mark.parametrize("existing", (True, False), ids=lambda x: f"existing-{x}")
async def test_remove_school_from_config(force, existing):
    schools = []
    if existing:
        schools = ["myschool"]
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority", logger=Mock(), force=force
    )
    manager.get_config = Mock(return_value=get_school_authority_patch_document(schools))
    manager.patch_school_authority_config = AsyncMock()
    manager.safe_current_config = Mock()
    if not force and not existing:
        with pytest.raises(
            ManageSchoolsException, match=r"School myschool was already removed"
        ):
            await manager.remove_school_from_config(school="myschool")
    else:
        await manager.remove_school_from_config(school="myschool")
        if existing:
            args = manager.patch_school_authority_config.call_args
            assert args[1]["data"] == {"plugin_configs": {"id_broker": {"schools": []}}}
        else:
            manager.patch_school_authority_config.assert_not_called()


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "schools_filter",
    (
        ["myschool1", "myschool2"],
        ["*"],
        ["myschool*"],
        ["my*", "*school*", "myschool", "*"],
    ),
)
@pytest.mark.parametrize("force", (True, False), ids=lambda x: f"force-{x}")
@pytest.mark.parametrize(
    "delete_schools", (True, False), ids=lambda x: f"delete_schools-{x}"
)
async def test_remove_school(schools_filter, force, delete_schools):
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority", logger=Mock(), force=force
    )
    if force:
        config = get_school_authority_patch_document([])
    else:
        config = get_school_authority_patch_document(schools_filter)
    manager.get_config = Mock(return_value=config)
    manager.remove_school_from_config = AsyncMock()
    manager.remove_schools_on_id_broker = AsyncMock()
    with patch(
        "idbroker.manage_schools_to_sync.get_ous", return_value=[FakeOu("myschool")]
    ):
        await manager.remove_schools(
            schools=schools_filter, delete_schools=delete_schools
        )
    if delete_schools:
        assert manager.remove_schools_on_id_broker.call_args_list == [
            call(school_filter=s) for s in schools_filter
        ]
    else:
        manager.remove_schools_on_id_broker.assert_not_called()


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "delete_schools", (True, False), ids=lambda x: f"delete_schools-{x}"
)
async def test_remove_school_wildcard_error(delete_schools):
    # We don't support wildcards for matching config keys
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority", logger=Mock(), force=True
    )
    config = get_school_authority_patch_document(["myschool"])
    manager.get_config = Mock(return_value=config)
    manager.remove_schools_on_id_broker = AsyncMock()
    with patch(
        "idbroker.manage_schools_to_sync.get_ous", return_value=[FakeOu("myschool")]
    ):
        with pytest.raises(ManageSchoolsException) as excinfo:
            await manager.remove_schools(schools=["my*"], delete_schools=delete_schools)
        assert "Wildcards are not supported" in str(excinfo)


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "exception",
    (MySpecialException(), ManageSchoolsException(), KeyboardInterrupt()),
    ids=lambda x: x.__class__,
)
@pytest.mark.asyncio
async def test_add_schools_exception_during_execution(exception):
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority", logger=Mock(), force=False
    )
    config = get_school_authority_patch_document(["myschool"])
    manager.get_config = Mock(return_value=config)
    manager.restore_old_config = AsyncMock()
    with patch("idbroker.manage_schools_to_sync.get_ous", side_effect=exception):
        with pytest.raises(exception.__class__):
            await manager.add_schools(["school1", "school2"])
    manager.restore_old_config.assert_called_with(config)


@pytest.mark.parametrize("all_schools", (True, False), ids=lambda x: f"all_schools-{x}")
@pytest.mark.parametrize(
    "exception",
    (MySpecialException, ManageSchoolsException, KeyboardInterrupt),
    ids=lambda x: x.__class__,
)
def test_add_schools_script_exception_during_execution(all_schools, exception):
    runner = CliRunner()
    fake_ous = [FakeOu(school) for school in get_fake_schools()]
    with (
        patch("idbroker.manage_schools_to_sync.setup_logging"),
        patch.object(ManageSchoolsOnIDBroker, "add_schools", side_effect=exception),
        patch("idbroker.manage_schools_to_sync.get_ous", return_value=fake_ous),
    ):
        if all_schools:
            args = [
                "--num_tasks",
                2,
                "--school_authority",
                "test-authority",
                "--all_schools",
            ]
        else:
            args = [
                "--num_tasks",
                2,
                "--school_authority",
                "test-authority",
                "school1",
                "school2",
            ]

        if exception is MySpecialException:
            result = runner.invoke(
                add_schools,
                args,
            )
            assert isinstance(result.exception, MySpecialException)
        else:
            result = runner.invoke(
                add_schools,
                args,
            )
            assert isinstance(result.exception, SystemExit)


@pytest.mark.parametrize(
    "exception",
    (MySpecialException(), ManageSchoolsException(), KeyboardInterrupt()),
    ids=lambda x: x.__class__,
)
@pytest.mark.asyncio
async def test_remove_school_exception_during_execution(exception):
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority", logger=Mock(), force=False
    )
    manager.remove_school_from_config = AsyncMock(side_effect=exception)
    config = get_school_authority_patch_document()
    manager.get_config = Mock(return_value=config)
    manager.restore_old_config = AsyncMock()
    with pytest.raises(exception.__class__):
        await manager.remove_schools(["school1", "school2"], delete_schools=True)
    manager.restore_old_config.assert_called_with(config)


@pytest.mark.parametrize("all_schools", (True, False), ids=lambda x: f"all_schools-{x}")
@pytest.mark.parametrize(
    "exception",
    (MySpecialException, ManageSchoolsException, KeyboardInterrupt),
    ids=lambda x: x,
)
def test_remove_schools_script_exception_during_execution(all_schools, exception):
    runner = CliRunner()
    with (
        patch("idbroker.manage_schools_to_sync.setup_logging"),
        patch.object(ManageSchoolsOnIDBroker, "remove_schools", side_effect=exception),
        patch.object(
            ManageSchoolsOnIDBroker, "get_schools_from_config", return_value=["school1"]
        ),
    ):
        if all_schools:
            args = ["--school_authority", "test-authority", "--all_schools"]
        else:
            args = ["--school_authority", "test-authority", "school1", "school2"]
        if exception is MySpecialException:
            res = runner.invoke(
                remove_schools,
                args,
            )
            assert isinstance(res.exception, MySpecialException)
        else:
            result = runner.invoke(
                remove_schools,
                args,
            )
            assert isinstance(result.exception, SystemExit)


@pytest.mark.parametrize(
    "exception",
    (MySpecialException(), ManageSchoolsException(), KeyboardInterrupt()),
    ids=lambda x: x.__class__,
)
@pytest.mark.asyncio
async def test_configure_schools_exception_during_execution(exception):
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority", logger=Mock(), force=False
    )
    manager.add_school_to_config = AsyncMock(side_effect=exception)
    config = get_school_authority_patch_document(["myschool"])
    manager.get_config = Mock(return_value=config)
    manager.restore_old_config = AsyncMock()
    with pytest.raises(exception.__class__):
        await manager.configure_schools(["school1", "school2"])
    manager.restore_old_config.assert_called_with(config)


@pytest.mark.parametrize("all_schools", (True, False), ids=lambda x: f"all_schools-{x}")
@pytest.mark.parametrize(
    "exception",
    (MySpecialException, ManageSchoolsException, KeyboardInterrupt),
    ids=lambda x: x.__class__,
)
def test_configure_schools_script_exception_during_execution(all_schools, exception):
    runner = CliRunner()
    fake_ous = [FakeOu(school) for school in get_fake_schools()]
    with (
        patch("idbroker.manage_schools_to_sync.setup_logging"),
        patch("idbroker.manage_schools_to_sync.get_ous", return_value=fake_ous),
        patch("idbroker.manage_schools_to_sync.setup_logging"),
        patch.object(
            ManageSchoolsOnIDBroker, "configure_schools", side_effect=exception
        ),
    ):
        if all_schools:
            args = [
                "--school_authority",
                "test-authority",
                "--initial_sync",
                "false",
                "--all_schools",
            ]
        else:
            args = [
                "--school_authority",
                "test-authority",
                "--initial_sync",
                "false",
                "school1",
            ]
        if exception is MySpecialException:
            res = runner.invoke(
                add_schools,
                args,
            )
            assert isinstance(res.exception, MySpecialException)
        else:
            result = runner.invoke(
                add_schools,
                args,
            )
            assert isinstance(result.exception, SystemExit)


@pytest.mark.asyncio
async def test_wait_for_empty_queue():
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority",
        logger=Mock(),
        num_tasks_for_initial_sync=20,
        force=False,
    )
    ManageSchoolsOnIDBroker.queue_check_timeout = 0.1
    ManageSchoolsOnIDBroker.queue_check_interval = 0.01
    with patch(
        "idbroker.manage_schools_to_sync.query_service",
        AsyncMock(return_value={"result": {"length": 0}}),
    ):
        await manager.wait_for_empty_queue()


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "successful,length",
    [(False, 1), (True, 0)],
    ids=["existing-after-timeout", "not-existing-after-timeout"],
)
async def test_wait_for_empty_queue_errors_during_query(successful, length):
    manager = ManageSchoolsOnIDBroker(
        school_authority="test_authority",
        logger=Mock(),
        num_tasks_for_initial_sync=20,
        force=False,
    )
    ManageSchoolsOnIDBroker.queue_check_timeout = 0.1
    ManageSchoolsOnIDBroker.queue_check_interval = 0.01
    with patch(
        "idbroker.manage_schools_to_sync.query_service",
        AsyncMock(return_value={"errors": ["some error"]}),
    ):
        with pytest.raises(
            ManageSchoolsException,
            match="Error while trying to get the length of queue",
        ):
            await manager.wait_for_empty_queue()


@pytest.mark.parametrize(
    "delete_schools", (True, False), ids=lambda x: f"delete_schools-{x}"
)
@pytest.mark.parametrize("force", (True, False), ids=lambda x: f"force-{x}")
def test_remove_schools_script(delete_schools, force):
    runner = CliRunner()
    args = [
        "--school_authority",
        "test-authority",
        "school1",
        "school2",
    ]
    if delete_schools is False:
        args += ["--delete_schools", "false"]
    if force:
        args += ["--force"]
    with patch.object(ManageSchoolsOnIDBroker, "remove_schools") as mock:
        result = runner.invoke(
            remove_schools,
            args,
        )
        mock.assert_called_with(
            schools=("school1", "school2"), delete_schools=delete_schools
        )
    assert not result.exception


@pytest.mark.parametrize("all_schools", (True, False), ids=lambda x: f"all_schools-{x}")
@pytest.mark.parametrize(
    "passed_schools", (True, False), ids=lambda x: f"passed_schools-{x}"
)
def test_remove_schools_script_all_schools(all_schools, passed_schools):
    runner = CliRunner()
    args = [
        "--school_authority",
        "test-authority",
    ]
    if all_schools:
        args += ["--all_schools"]
    if passed_schools:
        args += ["school1", "school2"]
    with (
        patch.object(ManageSchoolsOnIDBroker, "remove_schools") as remove_schools_mock,
        patch.object(
            ManageSchoolsOnIDBroker,
            "get_schools_from_config",
            return_value=["testschool"],
        ) as get_schools_from_config_mock,
    ):
        result = runner.invoke(
            remove_schools,
            args,
        )
        if all_schools and passed_schools:
            remove_schools_mock.assert_not_called()
            get_schools_from_config_mock.assert_not_called()
            assert isinstance(result.exception, SystemExit)
        elif passed_schools:
            remove_schools_mock.assert_called_with(
                schools=("school1", "school2"), delete_schools=True
            )
            get_schools_from_config_mock.assert_not_called()
            assert not result.exception
        elif all_schools and not passed_schools:
            remove_schools_mock.assert_called_with(
                schools=["testschool"], delete_schools=True
            )
            get_schools_from_config_mock.assert_called()
            assert not result.exception


def test_remove_schools_script_all_schools_no_schools_and_no_all_school_option_raises():
    runner = CliRunner()
    args = [
        "--school_authority",
        "test-authority",
    ]
    with patch.object(ManageSchoolsOnIDBroker, "remove_schools") as remove_schools_mock:
        result = runner.invoke(
            remove_schools,
            args,
        )
        remove_schools_mock.assert_not_called()
        assert isinstance(result.exception, SystemExit)


def test_remove_schools_script_all_schools_no_schools_left():
    runner = CliRunner()
    args = ["--school_authority", "test-authority", "--all_schools"]
    with (
        patch.object(ManageSchoolsOnIDBroker, "remove_schools") as remove_schools_mock,
        patch.object(
            ManageSchoolsOnIDBroker, "get_schools_from_config", return_value=[]
        ),
    ):
        result = runner.invoke(
            remove_schools,
            args,
        )
        remove_schools_mock.assert_not_called()
        assert isinstance(result.exception, SystemExit)


def test_add_schools_script_all_schools_no_schools_and_no_all_school_option_raises():
    runner = CliRunner()
    args = [
        "--school_authority",
        "test-authority",
    ]
    with patch.object(ManageSchoolsOnIDBroker, "add_schools") as mock:
        result = runner.invoke(
            add_schools,
            args,
        )
        mock.assert_not_called()
        assert isinstance(result.exception, SystemExit)


@pytest.mark.parametrize("all_schools", (True, False), ids=lambda x: f"all_schools-{x}")
@pytest.mark.parametrize(
    "passed_schools", (True, False), ids=lambda x: f"passed_schools-{x}"
)
def test_add_schools_script_all_schools(all_schools, passed_schools):
    runner = CliRunner()
    args = [
        "--school_authority",
        "test-authority",
    ]
    if all_schools:
        args += ["--all_schools"]
    if passed_schools:
        args += ["school1", "school2"]
    with (
        patch.object(ManageSchoolsOnIDBroker, "add_schools") as mock,
        patch(
            "idbroker.manage_schools_to_sync.get_ous",
            return_value=[FakeOu("testschool")],
        ),
    ):
        result = runner.invoke(
            add_schools,
            args,
        )
        if all_schools and passed_schools:
            mock.assert_not_called()
            assert isinstance(result.exception, SystemExit)
        elif passed_schools:
            mock.assert_called_with(schools=("school1", "school2"))
            assert not result.exception
        elif all_schools and not passed_schools:
            mock.assert_called_with(schools=["testschool"])
            assert not result.exception


def get_fake_schools():
    return ["school1", "school2", "school3"]


@pytest.mark.asyncio
async def test_configure_all_schools():
    manager = ManageSchoolsOnIDBroker(
        school_authority="test-auth", logger=Mock(), force=False
    )
    manager.add_school_to_config = AsyncMock()
    manager.safe_current_config = Mock()
    manager.restore_old_config = AsyncMock()
    config = school_auth_config()
    config["plugin_configs"]["id_broker"]["schools"] = ["school1", "school2", "school3"]
    school_authority_config = SchoolAuthorityConfigurationPatchDocument(**config)
    with (
        patch(
            "idbroker.manage_schools_to_sync.get_ous", return_value=get_fake_schools()
        ),
        patch(
            "idbroker.manage_schools_to_sync.read_id_broker_config",
            return_value=school_authority_config,
        ),
    ):
        await manager.configure_schools(schools=["school1", "school2", "school3"])
    assert manager.add_school_to_config.call_count == 3
    assert manager.add_school_to_config.call_args_list == [
        call(school=f"school{i}") for i in range(1, 4)
    ]


@pytest.mark.asyncio
async def test_configure_schools_by_name():
    manager = ManageSchoolsOnIDBroker(
        school_authority="test-auth", logger=Mock(), force=False
    )
    manager.add_school_to_config = AsyncMock()
    config = get_school_authority_patch_document(["myschool"])
    manager.get_config = Mock(return_value=config)
    await manager.configure_schools(schools=["school1", "school2"])
    assert manager.add_school_to_config.call_count == 2
    assert manager.add_school_to_config.call_args_list == [
        call(school=f"school{i}") for i in range(1, 3)
    ]


def test_all_schools_without_initial_sync_script():
    runner = CliRunner()
    with (
        patch.object(ManageSchoolsOnIDBroker, "configure_schools") as mock,
        patch(
            "idbroker.manage_schools_to_sync.get_ous",
            return_value=[FakeOu(f"school{i}") for i in range(10)],
        ),
    ):
        result = runner.invoke(
            add_schools,
            [
                "--school_authority",
                "test-authority",
                "--initial_sync",
                "false",
                "--all_schools",
            ],
        )
        mock.assert_called_once()
        assert not result.exception


class IDBrokerSchoolMock:
    def __init__(
        self, school_authority, plugin_name, school_exists_after_first_attempt
    ):
        self.school_exists_after_first_attempt = school_exists_after_first_attempt
        self._delete_called_with = []
        self._exists_called_with = []

    async def delete(self, school_id):
        self._delete_called_with.append(school_id)

    async def exists(self, school_id):
        if (
            self.school_exists_after_first_attempt
            and school_id not in self._exists_called_with
        ):
            self._exists_called_with.append(school_id)
            return True
        self._exists_called_with.append(school_id)
        return False


def get_school_authority_patch_document(schools: List[str] = None):
    if schools is None:
        schools = []
    config = school_auth_config()
    config["plugin_configs"]["id_broker"]["schools"] = schools
    return SchoolAuthorityConfigurationPatchDocument(**config)


@pytest.mark.parametrize(
    "schools_filter,matching_schools,mismatch_schools",
    (
        (
            ["myschool1", "myschool2"],
            ["myschool1", "myschool2"],
            ["foo", "bar", "world"],
        ),
        (["*"], ["myschool", "foo", "bar"], []),
        (["myschool*"], ["myschool1", "myschool2", "myschool"], ["foo"]),
        (
            ["my*", "*school*", "myschool", "*"],
            ["myschool1", "myschool2", "myschool", "foo"],
            [],
        ),
    ),
)
@pytest.mark.asyncio
async def test_remove_schools_on_id_broker(
    schools_filter, matching_schools, mismatch_schools, school_authority_conf
):
    school_authority_conf.plugin_configs["id_broker"]["schools"] = copy.copy(
        schools_filter
    )
    manager = ManageSchoolsOnIDBroker(
        school_authority="test-auth", logger=Mock(), force=False
    )
    mock_school_client = IDBrokerSchoolMock(
        school_authority=school_authority_conf,
        plugin_name="id_broker",
        school_exists_after_first_attempt=True,
    )
    manager.get_config = lambda: school_authority_conf
    manager.wait_school_deletion_interval = 0.001
    with (
        patch(
            "idbroker.manage_schools_to_sync.IDBrokerSchool",
            return_value=mock_school_client,
        ),
        patch(
            "idbroker.manage_schools_to_sync.get_ous",
            fake_get_ous(matching_schools + mismatch_schools),
        ),
    ):
        for school_name in schools_filter:
            school_authority_conf.plugin_configs["id_broker"]["schools"].remove(
                school_name
            )
            await manager.remove_schools_on_id_broker(school_filter=school_name)
        assert mock_school_client._delete_called_with == matching_schools
        for mismatch in mismatch_schools:
            assert mismatch not in mock_school_client._delete_called_with
        assert mock_school_client._exists_called_with == [
            i for school in matching_schools for i in [school] * 2
        ]


@pytest.mark.asyncio
async def test_remove_schools_on_id_broker_skip_configured_school(
    school_authority_conf,
):
    school_authority_conf.plugin_configs["id_broker"]["schools"] = [
        "myschool11",
        "myschool*",
    ]
    manager = ManageSchoolsOnIDBroker(
        school_authority="test-auth", logger=Mock(), force=False
    )
    mock_school_client = IDBrokerSchoolMock(
        school_authority=school_authority_conf,
        plugin_name="id_broker",
        school_exists_after_first_attempt=True,
    )
    manager.get_config = lambda: school_authority_conf
    manager.wait_school_deletion_interval = 0.001
    with (
        patch(
            "idbroker.manage_schools_to_sync.IDBrokerSchool",
            return_value=mock_school_client,
        ),
        patch(
            "idbroker.manage_schools_to_sync.get_ous",
            return_value=[FakeOu("myschool11"), FakeOu("myschool22")],
        ),
    ):
        school_authority_conf.plugin_configs["id_broker"]["schools"].remove("myschool*")
        await manager.remove_schools_on_id_broker(school_filter="myschool*")
        assert mock_school_client._delete_called_with == ["myschool22"]
        assert "myschool11" not in mock_school_client._delete_called_with


@pytest.mark.asyncio
async def test_remove_schools_on_id_broker_school_not_existing(school_authority_conf):
    manager = ManageSchoolsOnIDBroker(
        school_authority="test-auth", logger=Mock(), force=False
    )
    mock_school_client = IDBrokerSchoolMock(
        school_authority=school_authority_conf,
        plugin_name="id_broker",
        school_exists_after_first_attempt=False,
    )
    manager.get_config = lambda: school_authority_conf
    with (
        patch(
            "idbroker.manage_schools_to_sync.IDBrokerSchool",
            return_value=mock_school_client,
        ),
        patch("idbroker.manage_schools_to_sync.get_ous", return_value=[]),
    ):
        with pytest.raises(SystemExit) as exc:
            await manager.remove_schools_on_id_broker(school_filter="myschool")
            assert exc.code == 1


@pytest.mark.asyncio
async def test_remove_schools_on_id_broker_school_existing_after_first_head(
    school_authority_conf,
):
    manager = ManageSchoolsOnIDBroker(
        school_authority="test-auth", logger=Mock(), force=False
    )
    fake_schools = [FakeOu("myschool")]
    mock_school_client = IDBrokerSchoolMock(
        school_authority=school_authority_conf,
        plugin_name="id_broker",
        school_exists_after_first_attempt=True,
    )
    manager.wait_school_deletion_timeout = 1
    manager.wait_school_deletion_interval = 0.01
    manager.get_config = lambda: school_authority_conf
    with (
        patch(
            "idbroker.manage_schools_to_sync.IDBrokerSchool",
            return_value=mock_school_client,
        ),
        patch("idbroker.manage_schools_to_sync.get_ous", return_value=fake_schools),
    ):
        await manager.remove_schools_on_id_broker(school_filter="myschool")
        assert mock_school_client._delete_called_with == ["myschool"]
        assert mock_school_client._exists_called_with == ["myschool", "myschool"]
