import copy
from unittest.mock import AsyncMock, patch

import pytest
from id_broker_handlers import IDBrokerUserDispatcher, create_work_group_if_missing

from idbroker.id_broker_client import (
    IDBrokerNotFoundError,
    IDBrokerWorkGroup as WorkgroupClient,
)
from ucsschool_id_connector.models import ListenerUserOldDataEntry, SchoolUserRole
from ucsschool_id_connector.utils import school_class_dn_regex, workgroup_dn_regex


@pytest.mark.asyncio
async def test_handle_attr_context(
    per_sa_user_dispatcher,
    get_listener_user_add_modify_object,
    user_dn,
    workgroup_dn,
    school_class_dn,
    monkeypatch,
):
    listener_obj = {
        "school": ["DEMOSCHOOL"],
        "schools": ["DEMOSCHOOL", "DEMOSCHOOL-2"],
        "groups": [workgroup_dn, school_class_dn],
    }
    obj = get_listener_user_add_modify_object(dn=user_dn, object=listener_obj)
    monkeypatch.setitem(
        per_sa_user_dispatcher.school_authority.plugin_configs["id_broker"],
        "schools",
        ["DEMOSCHOOL"],
    )
    context = await per_sa_user_dispatcher._handle_attr_context(obj=obj)
    assert context["DEMOSCHOOL"]["workgroups"] == {"wg1"}
    assert context["DEMOSCHOOL"]["classes"] == {"1a"}
    assert context["DEMOSCHOOL"]["roles"] == {SchoolUserRole.teacher}


@pytest.mark.asyncio
async def test__empty_groups(
    per_sa_user_dispatcher,
):
    context_text = {
        "DEMOSCHOOL": {
            "classes": ["my-test-class"],
            "workgroups": ["my-test-workgroup"],
        }
    }
    context = per_sa_user_dispatcher._empty_groups(context_text)
    assert context["DEMOSCHOOL"]["classes"] == []
    assert context["DEMOSCHOOL"]["workgroups"] == []


class MockValue(object):
    def __init__(self, value):
        self.value = value


class MockObject(object):
    def __init__(self, entry_uuid, description):
        self.entryUUID = MockValue(entry_uuid)
        self.description = MockValue(description)


class MockLDAPAccess:
    called_with_filter_s = None
    called_with_attributes = None
    entries_in_back_end = None

    async def search(self, filter_s, attributes):
        MockLDAPAccess.called_with_filter_s = filter_s
        MockLDAPAccess.called_with_attributes = attributes
        for dn, attrs in MockLDAPAccess.entries_in_back_end:
            if filter_s == dn:
                return [attrs]
        return []


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "configured_schools",
    [(["Configured1", "Configured2"],), (["*"],), (["Configured*"],)],
)
@pytest.mark.parametrize(
    "new_schools,old_schools,expected",
    [
        (
            ["not_configured"],
            ["Configured1", "Configured2", "not_configured"],
            "remove",
        ),
        (None, ["Configured1", "Configured2", "not_configured"], "remove"),
        (
            ["Configured1", "Configured2"],
            ["Configured1", "Configured2"],
            "add_or_modify",
        ),
        (
            ["Configured1", "Configured2"],
            ["Configured1", "Configured2", "not_configured"],
            "add_or_modify",
        ),
        (["Configured1", "Configured2"], ["not_configured"], "add_or_modify"),
        (["Configured1", "Configured2"], None, "add_or_modify"),
        (
            ["Configured1", "Configured2", "not_configured"],
            ["not_configured"],
            "add_or_modify_filtered",
        ),
        (
            ["Configured1", "Configured2", "not_configured"],
            ["Configured1", "Configured2", "not_configured"],
            "add_or_modify_filtered",
        ),
    ],
)
async def test_handle_listener_object(
    new_schools,
    old_schools,
    expected,
    configured_schools,
    school_authority_conf,
    get_listener_user_add_modify_object,
    get_listener_user_remove_object,
    user_dn,
    monkeypatch,
    workgroup_dn,
    school_class_dn,
):
    if "*" in configured_schools and (
        "not_configured" in new_schools or "not_configured" in old_schools
    ):
        # "*" means every school is configured and would match "not_configured"
        return True
    workgroup_dn_regex.cache_clear()
    school_class_dn_regex.cache_clear()
    monkeypatch.setenv("ldap_base", "dc=ldap,dc=base")
    monkeypatch.setenv("ldap_server_name", "TEST-SERVER-NAME")
    monkeypatch.setenv("ldap_server_port", "789")
    dispatcher = IDBrokerUserDispatcher()
    if new_schools is not None:
        listener_obj = get_listener_user_add_modify_object(
            dn=user_dn,
            object={
                "school": new_schools,
                "ucsschoolRole": [f"student:school:{school}" for school in new_schools],
                "groups": [
                    workgroup_dn.replace("DEMOSCHOOL", school) for school in new_schools
                ]
                + [
                    school_class_dn.replace("DEMOSCHOOL", school)
                    for school in new_schools
                ],
            },
        )
    else:
        listener_obj = get_listener_user_remove_object(dn=user_dn)
    if old_schools is not None:
        listener_obj.old_data = ListenerUserOldDataEntry(schools=old_schools)
    listener_obj_orig = copy.deepcopy(listener_obj)
    school_authority_conf.plugin_configs["id_broker"]["schools"] = [
        "Configured1",
        "Configured2",
    ]
    dispatcher.handler(
        school_authority_conf, dispatcher.plugin_name
    ).handle_remove = AsyncMock()
    dispatcher.handler(
        school_authority_conf, dispatcher.plugin_name
    ).handle_create_or_update = AsyncMock()
    if expected == "add_or_modify_filtered":
        # make sure the test object has a school that needs to be filtered
        assert "not_configured" in listener_obj.schools
        assert any(
            ["not_configured" in role for role in listener_obj.object["ucsschoolRole"]]
        )
        assert any(
            ["not_configured" in group for group in listener_obj.object["groups"]]
        )
    await dispatcher.handle_listener_object(school_authority_conf, listener_obj)
    if expected == "remove":
        dispatcher.handler(
            school_authority_conf, dispatcher.plugin_name
        ).handle_create_or_update.assert_not_called()
        dispatcher.handler(
            school_authority_conf, dispatcher.plugin_name
        ).handle_remove.assert_called_with(listener_obj)
        assert listener_obj == listener_obj_orig
    elif expected == "add_or_modify":
        dispatcher.handler(
            school_authority_conf, dispatcher.plugin_name
        ).handle_remove.assert_not_called()
        dispatcher.handler(
            school_authority_conf, dispatcher.plugin_name
        ).handle_create_or_update.assert_called_with(listener_obj)
        assert listener_obj == listener_obj_orig
    else:
        dispatcher.handler(
            school_authority_conf, dispatcher.plugin_name
        ).handle_remove.assert_not_called()
        dispatcher.handler(
            school_authority_conf, dispatcher.plugin_name
        ).handle_create_or_update.assert_called_with(listener_obj)
        assert listener_obj != listener_obj_orig
        assert "not_configured" not in listener_obj.schools
        assert listener_obj.schools == ["Configured1", "Configured2"]
        assert not any(
            ["not_configured" in role for role in listener_obj.object["ucsschoolRole"]]
        )
        assert listener_obj.object["ucsschoolRole"] == [
            "student:school:Configured1",
            "student:school:Configured2",
        ]
        assert not any(
            ["not_configured" in group for group in listener_obj.object["groups"]]
        )
        assert sorted(listener_obj.object["groups"]) == sorted(
            [
                workgroup_dn.replace("DEMOSCHOOL", "Configured1"),
                school_class_dn.replace("DEMOSCHOOL", "Configured1"),
                workgroup_dn.replace("DEMOSCHOOL", "Configured2"),
                school_class_dn.replace("DEMOSCHOOL", "Configured2"),
            ]
        )


@pytest.mark.asyncio
async def test_create_workgroup_if_missing(
    school_authority_conf, per_sa_user_dispatcher
):
    wg_name = "wg1"
    mock_ldap_access = MockLDAPAccess()
    MockLDAPAccess.entries_in_back_end = [("", "")]
    logger = per_sa_user_dispatcher.logger
    client = WorkgroupClient(
        school_authority=school_authority_conf, plugin_name="id_broker"
    )
    with pytest.raises(IDBrokerNotFoundError):
        await create_work_group_if_missing(
            work_group=wg_name,
            ou="TEST",
            id_broker_work_group=client,
            ldap_access=mock_ldap_access,
            logger=logger,
        )

    MockLDAPAccess.entries_in_back_end = [
        (
            "(&(objectClass=univentionGroup)(cn=TEST-wg1))",
            MockObject(entry_uuid="test-entry-uuid", description="description"),
        )
    ]

    async def _exists(self, **kwargs):
        return True

    called_create = False

    async def _create(self, **kwargs):
        nonlocal called_create
        called_create = True

    with patch("idbroker.id_broker_client.ProvisioningAPIClient._exists", _exists):
        await create_work_group_if_missing(
            work_group=wg_name,
            ou="TEST",
            id_broker_work_group=client,
            ldap_access=mock_ldap_access,
            logger=logger,
        )
        assert called_create is False

    async def _exists(self, **kwargs):
        return False

    with patch("idbroker.id_broker_client.ProvisioningAPIClient._exists", _exists):
        with patch("idbroker.id_broker_client.ProvisioningAPIClient._create", _create):
            await create_work_group_if_missing(
                work_group=wg_name,
                ou="TEST",
                id_broker_work_group=client,
                ldap_access=mock_ldap_access,
                logger=logger,
            )
        assert called_create is True
