# -*- coding: utf-8 -*-
# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.

from unittest.mock import AsyncMock

import id_broker_handlers
import pytest
from id_broker_handlers import (
    IDBrokerPerSAGroupDispatcher,
    has_workgroup_role,
    role_workgroup,
)
from ucsschool_id_connector_defaults.group_handler_base import GroupNotFoundError  # noqa: E402

from idbroker.id_broker_client import (
    IDBrokerError,
    IDBrokerNotFoundError,
    IDBrokerSchoolClass,
    IDBrokerWorkGroup,
    SchoolClass,
    WorkGroup,
)
from ucsschool_id_connector.models import ListenerGroupRemoveObject
from ucsschool_id_connector.utils import workgroup_dn_regex


def workgroup_request_body():
    return {
        "id": "my-id",
        "name": "DEMOSCHOOL-wg1",
        "school": "DEMOSCHOOL",
        "ucsschoolRole": [f"{role_workgroup}:school:DEMOSCHOOL"],
    }


def school_class_request_body():
    return {
        "id": "my-id",
        "name": "DEMOSCHOOL-1a",
        "school": "DEMOSCHOOL",
        "ucsschoolRole": ["school_class:school:DEMOSCHOOL"],
    }


def test_has_workgroup_role():
    assert has_workgroup_role(
        roles=[f"{role_workgroup}:school:testschool"], school="testschool"
    )
    assert (
        has_workgroup_role(
            roles=[f"{role_workgroup}:school:testschool"], school="falseschool"
        )
        is False
    )
    assert has_workgroup_role(roles=[], school="testschool") is False


def test_id_broker_per_sa_group_dispatcher_init(per_sa_group_dispatcher):
    assert isinstance(per_sa_group_dispatcher.id_broker_work_group, IDBrokerWorkGroup)
    assert isinstance(
        per_sa_group_dispatcher.id_broker_school_class, IDBrokerSchoolClass
    )
    assert per_sa_group_dispatcher.work_group_dn_regex == workgroup_dn_regex()


@pytest.mark.asyncio
async def test_is_work_group(
    per_sa_group_dispatcher, get_listener_group_add_modify_object, workgroup_dn
):
    obj = get_listener_group_add_modify_object(dn=workgroup_dn)
    assert await per_sa_group_dispatcher.is_work_group(obj)
    obj = get_listener_group_add_modify_object(
        dn="cn=wg1,cn=schueler,cn=groups,ou=DEMOSCHOOL,dc=ldap,dc=base"
    )
    assert await per_sa_group_dispatcher.is_work_group(obj) is False
    obj = get_listener_group_add_modify_object(
        dn="cn=wg1,cn=other,cn=groups,dc=ldap,dc=base"
    )
    assert await per_sa_group_dispatcher.is_work_group(obj) is False


@pytest.mark.asyncio
async def test_create_or_update_preconditions_met(
    per_sa_group_dispatcher,
    school_authority_conf,
    get_listener_group_add_modify_object,
    workgroup_dn,
):
    obj = get_listener_group_add_modify_object(dn=workgroup_dn)
    assert (
        await per_sa_group_dispatcher.create_or_update_preconditions_met(obj=obj)
        is True
    )
    # Should not be handled in initial import mode.
    school_authority_conf.plugin_configs["id_broker"]["initial_import_mode"] = True
    assert (
        await per_sa_group_dispatcher.create_or_update_preconditions_met(obj=obj)
        is False
    )


@pytest.mark.asyncio
async def test_remove_preconditions_met(
    per_sa_group_dispatcher,
    school_authority_conf,
    get_listener_group_add_modify_object,
    workgroup_dn,
):
    obj = get_listener_group_add_modify_object(dn=workgroup_dn)
    assert await per_sa_group_dispatcher.remove_preconditions_met(obj=obj) is True
    # Should not be handled in initial import mode.
    school_authority_conf.plugin_configs["id_broker"]["initial_import_mode"] = True
    assert await per_sa_group_dispatcher.remove_preconditions_met(obj=obj) is False


@pytest.mark.asyncio
async def test_fetch_obj_workgroup_not_found(per_sa_group_dispatcher, workgroup_dn):
    async def _mock_get(**kwargs):
        raise IDBrokerNotFoundError(status=500)

    per_sa_group_dispatcher.id_broker_work_group.get = _mock_get
    with pytest.raises(
        GroupNotFoundError,
        match=f"No workgroup found with search params: {{'id': 'my_id', 'dn': '{workgroup_dn}'}}",
    ):
        await per_sa_group_dispatcher.fetch_obj({"id": "my_id", "dn": workgroup_dn})


@pytest.mark.asyncio
async def test_fetch_obj_school_class_not_found(
    per_sa_group_dispatcher, school_class_dn
):
    async def _mock_get(**kwargs):
        raise IDBrokerNotFoundError(status=500)

    per_sa_group_dispatcher.id_broker_school_class.get = _mock_get
    with pytest.raises(
        GroupNotFoundError,
        match=f"No school class found with search params: {{'id': 'my_id', 'dn': '{school_class_dn}'}}",
    ):
        await per_sa_group_dispatcher.fetch_obj({"id": "my_id", "dn": school_class_dn})


@pytest.mark.asyncio
async def test_handle_attr_school(
    per_sa_group_dispatcher, get_listener_group_add_modify_object, workgroup_dn
):
    obj = get_listener_group_add_modify_object(dn=workgroup_dn)
    assert await per_sa_group_dispatcher._handle_attr_school(obj=obj) == "DEMOSCHOOL"


@pytest.mark.asyncio
async def test_handle_attr_name(
    per_sa_group_dispatcher, get_listener_group_add_modify_object, workgroup_dn
):
    obj = get_listener_group_add_modify_object(
        dn=workgroup_dn, object={"name": "DEMOSCHOOL-wg1"}
    )
    assert await per_sa_group_dispatcher._handle_attr_name(obj=obj) == "wg1"


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "request_body,group_cls",
    [(workgroup_request_body(), WorkGroup), (school_class_request_body(), SchoolClass)],
    ids=["workgroup", "school_class"],
)
async def test_do_create_missing_members_triggers_fix_school_group_members(
    per_sa_group_dispatcher,
    request_body,
    group_cls,  # type: WorkGroup | SchoolClass
):
    called_with_args = None

    async def _fix_school_group_members(g):
        nonlocal called_with_args
        called_with_args = g

    async def create_school_if_missing(*a, **kwargs): ...

    async def create_member_missing(*a, **kwargs):
        raise IDBrokerError(422, "member")

    async def create_internal_error(*a, **kwargs):
        raise IDBrokerError(500, "internal")

    id_broker_handlers.create_school_if_missing = create_school_if_missing
    per_sa_group_dispatcher.id_broker_work_group.create = create_member_missing
    per_sa_group_dispatcher.id_broker_school_class.create = create_member_missing
    per_sa_group_dispatcher._fix_school_group_members = _fix_school_group_members
    await per_sa_group_dispatcher.do_create(request_body)
    assert called_with_args == group_cls(**request_body)
    per_sa_group_dispatcher.id_broker_work_group.create = create_internal_error
    per_sa_group_dispatcher.id_broker_school_class.create = create_internal_error
    with pytest.raises(IDBrokerError):
        await per_sa_group_dispatcher.do_create(request_body)


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "request_body,group_cls",
    [(workgroup_request_body(), WorkGroup), (school_class_request_body(), SchoolClass)],
    ids=["workgroup", "school_class"],
)
@pytest.mark.parametrize("method", ["do_create", "do_modify"])
async def test_skipped_in_initial_import_mode(
    school_authority_conf,
    request_body,
    monkeypatch,
    method,
    group_cls,  # type: WorkGroup | SchoolClass
):
    monkeypatch.setenv("ldap_base", "dc=ldap,dc=base")
    monkeypatch.setenv("ldap_server_name", "TEST-SERVER-NAME")
    monkeypatch.setenv("ldap_server_port", "789")
    school_authority_conf.plugin_configs["id_broker"]["initial_import_mode"] = True
    per_sa_group_dispatcher = IDBrokerPerSAGroupDispatcher(
        school_authority=school_authority_conf, plugin_name="id_broker-groups"
    )
    group = group_cls(**request_body)

    if method == "do_create":
        assert await per_sa_group_dispatcher.do_create(request_body) is None
    elif method == "do_modify":
        assert (
            await per_sa_group_dispatcher.do_modify(
                request_body, api_school_group_data=group
            )
            is None
        )
    assert per_sa_group_dispatcher.initial_import_mode is True


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "request_body,group_cls",
    [(workgroup_request_body(), WorkGroup), (school_class_request_body(), SchoolClass)],
    ids=["workgroup", "school_class"],
)
async def test_do_modify_missing_members_triggers_fix_school_group_members(
    per_sa_group_dispatcher,
    request_body,
    group_cls,  # type: WorkGroup | SchoolClass
):
    called_with_arg = None

    async def _fix_school_group_members(group):
        nonlocal called_with_arg
        called_with_arg = group

    async def create_school_if_missing(*a, **kwargs): ...

    async def update_member_missing(*a, **kwargs):
        raise IDBrokerError(422, "member")

    async def update_internal_error(*a, **kwargs):
        raise IDBrokerError(500, "internal")

    id_broker_handlers.create_school_if_missing = create_school_if_missing
    per_sa_group_dispatcher.id_broker_work_group.update = update_member_missing
    per_sa_group_dispatcher.id_broker_school_class.update = update_member_missing
    per_sa_group_dispatcher._fix_school_group_members = _fix_school_group_members
    group = group_cls(**request_body)
    await per_sa_group_dispatcher.do_modify(
        request_body=request_body, api_school_group_data=group
    )
    assert called_with_arg == group
    per_sa_group_dispatcher.id_broker_work_group.update = update_internal_error
    per_sa_group_dispatcher.id_broker_school_class.update = update_internal_error
    with pytest.raises(IDBrokerError):
        await per_sa_group_dispatcher.do_modify(
            request_body=request_body, api_school_group_data=group
        )


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "request_body,group_cls",
    [(workgroup_request_body(), WorkGroup), (school_class_request_body(), SchoolClass)],
    ids=["workgroup", "school_class"],
)
async def test_do_remove(
    per_sa_group_dispatcher,
    request_body,
    group_cls,  # type: WorkGroup | SchoolClass
    workgroup_dn,
    school_class_dn,
):
    per_sa_group_dispatcher.id_broker_work_group.delete = AsyncMock()
    per_sa_group_dispatcher.id_broker_school_class.delete = AsyncMock()
    group = group_cls(**request_body)
    if isinstance(group_cls, WorkGroup):
        dn = workgroup_dn
    else:
        dn = school_class_dn
    obj = ListenerGroupRemoveObject(
        dn=dn,
        id="my-id",
        udm_object_type="groups/group",
        options=[],
        object={"name": "DEMOSCHOOL-wg1"},
    )
    await per_sa_group_dispatcher.do_remove(obj=obj, api_school_group_data=group)
    if isinstance(group_cls, WorkGroup):
        per_sa_group_dispatcher.id_broker_work_group.delete.assert_called_with(group.id)
        per_sa_group_dispatcher.id_broker_school_class.delete.assert_not_awaited()
    else:
        per_sa_group_dispatcher.id_broker_school_class.delete.assert_called_with(
            group.id
        )
        per_sa_group_dispatcher.id_broker_work_group.delete.assert_not_awaited()


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "request_body,group_cls",
    [(workgroup_request_body(), WorkGroup), (school_class_request_body(), SchoolClass)],
    ids=["workgroup", "school_class"],
)
async def test_fix_school_group_members(
    per_sa_group_dispatcher,
    request_body,
    group_cls,  # type: WorkGroup | SchoolClass
):
    async def create(group):
        return group

    async def get(*args, **kwargs):
        raise IDBrokerNotFoundError(status=404)

    group = group_cls(**request_body)
    per_sa_group_dispatcher.id_broker_work_group.create = create
    per_sa_group_dispatcher.id_broker_school_class.create = create
    per_sa_group_dispatcher.id_broker_work_group.get = get
    per_sa_group_dispatcher.id_broker_school_class.get = get
    await per_sa_group_dispatcher._fix_school_group_members(group)


@pytest.mark.asyncio
async def test_search_params(
    per_sa_group_dispatcher, get_listener_group_add_modify_object, workgroup_dn
):
    obj = get_listener_group_add_modify_object(dn=workgroup_dn, id="MY-ID")
    assert await per_sa_group_dispatcher.search_params(obj=obj) == {
        "id": "MY-ID",
        "school_authority": "TEST",
        "dn": workgroup_dn,
    }
