from unittest.mock import MagicMock

import id_broker_distribution
import pytest

import idbroker.utils
from ucsschool_id_connector.models import (
    ListenerGroupAddModifyObject,
    ListenerGroupRemoveObject,
    ListenerUserAddModifyObject,
    ListenerUserRemoveObject,
)
from ucsschool_id_connector.queues import InQueue


@pytest.mark.parametrize(
    "obj",
    [
        ListenerUserAddModifyObject(
            dn="foobar",
            id="my-id",
            udm_object_type="users/user",
            options=["default", "ucsschoolTeacher"],
            object={"school": "my-school"},
        ),
        ListenerUserRemoveObject(dn="foobar", id="my-id", udm_object_type="users/user"),
        ListenerGroupAddModifyObject(
            dn="foobar",
            id="my-id",
            udm_object_type="groups/group",
            options=[],
            object={},
        ),
        ListenerGroupRemoveObject(
            dn="foobar", id="my-id", udm_object_type="groups/group"
        ),
    ],
)
def test_obj_types(obj, school_authority_conf, monkeypatch):
    monkeypatch.setattr(
        idbroker.utils,
        "group_event_on_configured_schools",
        MagicMock(return_value=True),
    )
    monkeypatch.setattr(
        idbroker.utils, "user_event_on_configured_schools", MagicMock(return_value=True)
    )
    distribution = id_broker_distribution.IDBrokerDistributionImpl()
    school_authority_conf.active = True
    school_authority_conf.plugin_configs["id_broker"]["schools"] = ["some-school"]
    assert distribution.obj_should_be_distributed(obj, school_authority_conf)
    if isinstance(obj, ListenerGroupAddModifyObject) or isinstance(
        obj, ListenerGroupRemoveObject
    ):
        idbroker.utils.group_event_on_configured_schools.assert_called()
        idbroker.utils.user_event_on_configured_schools.assert_not_called()
    else:
        idbroker.utils.group_event_on_configured_schools.assert_not_called()
        idbroker.utils.user_event_on_configured_schools.assert_called()


@pytest.mark.parametrize(
    "exists,active,schools,should_be_distributed,expected",
    [
        (False, True, ["test_school"], True, False),
        (True, False, ["test_school"], True, False),
        (True, False, [], True, False),
        (True, True, [], True, False),
        (True, True, None, True, False),
        (True, True, ["test_school"], False, False),
        (True, True, ["DEMOSCHOOL"], True, True),
    ],
)
def test_obj_should_be_distributed(
    exists,
    active,
    schools,
    should_be_distributed,
    expected,
    school_authority_conf,
    get_listener_user_add_modify_object,
    user_dn,
    monkeypatch,
):
    listener_obj = get_listener_user_add_modify_object(
        dn=user_dn,
        object={
            "school": ["DEMOSCHOOL"],
        },
    )
    distribution = id_broker_distribution.IDBrokerDistributionImpl()
    school_authority_conf.active = active
    if schools is not None:
        school_authority_conf.plugin_configs["id_broker"]["schools"] = schools
    if not exists:
        school_authority_conf = None
    assert (
        distribution.obj_should_be_distributed(listener_obj, school_authority_conf)
        is expected
    )


@pytest.mark.asyncio
@pytest.mark.parametrize("should_be_distributed", [True, False])
async def test_school_authorities_to_distribute_to(
    monkeypatch,
    school_authority_conf,
    should_be_distributed,
    get_listener_user_add_modify_object,
    user_dn,
):
    monkeypatch.setattr(
        id_broker_distribution.IDBrokerDistributionImpl,
        "_get_id_broker_school_authorities",
        lambda *args: [school_authority_conf],
    )
    monkeypatch.setattr(
        id_broker_distribution.IDBrokerDistributionImpl,
        "obj_should_be_distributed",
        lambda *args: should_be_distributed,
    )
    distribution = id_broker_distribution.IDBrokerDistributionImpl()
    listener_obj = get_listener_user_add_modify_object(
        dn=user_dn,
        object={
            "school": ["DEMOSCHOOL"],
        },
    )
    if should_be_distributed:
        assert await distribution.school_authorities_to_distribute_to(
            listener_obj, InQueue()
        ) == [school_authority_conf.name]
    else:
        assert (
            await distribution.school_authorities_to_distribute_to(
                listener_obj, InQueue()
            )
            == []
        )
