# -*- coding: utf-8 -*-

# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.

import asyncio
from pathlib import Path

import faker
import pytest
import pytest_asyncio
from tenacity import (
    AsyncRetrying,
    retry_if_exception_type,
    stop_after_attempt,
    wait_fixed,
)

import ucsschool_id_connector.plugin_loader
from ucsschool.kelvin.client import User as KelvinUser
from ucsschool.kelvin.client.exceptions import NoObject

# load ID Broker plugin
ucsschool_id_connector.plugin_loader.load_plugins()
id_broker = pytest.importorskip("idbroker")
pytestmark = pytest.mark.id_broker
fake = faker.Faker()


@pytest_asyncio.fixture()
def new_id_broker_school_auth_conf(
    school_auth_config_id_broker, kelvin_session, id_broker_fqdn, create_school
) -> callable:
    async def _func(school):
        s_a_name = "".join(fake.street_name().split())
        print(f"*** Creating school authority {s_a_name!r}...")
        password = fake.password(length=15)
        await create_school(host=id_broker_fqdn, ou_name=school)

        username = f"provisioning-{s_a_name}"
        kelvin_user = KelvinUser(
            name=username,
            school=school,
            firstname=fake.first_name(),
            lastname=fake.last_name(),
            disabled=False,
            password=password,
            record_uid=fake.uuid4(),
            roles=["teacher"],
            schools=[school],
            session=kelvin_session(id_broker_fqdn),
        )
        async for attempt in AsyncRetrying(
            reraise=True,
            stop=stop_after_attempt(6),
            retry=retry_if_exception_type(NoObject),
            wait=wait_fixed(2),
        ):
            with attempt:
                await kelvin_user.save()
        print(f"Created admin user {kelvin_user.name!r}.")
        print(
            f"Waiting 10 sec for user {username!r} to be replicated to LDAP of {id_broker_fqdn!r}..."
        )
        await asyncio.sleep(10.0)

        return school_auth_config_id_broker(s_a_name, password)

    return _func


@pytest_asyncio.fixture()
def set_school_authority_on_id_connector_special(
    make_school_authority, new_id_broker_school_auth_conf
):
    """
    Difference to set_school_authority_on_id_connector: instead of using Traeger*
    a custom configuration can be used.
    """

    async def _func(school):
        config = await new_id_broker_school_auth_conf(school)
        config["plugin_configs"]["id_broker"]["schools"] = [school]
        school_authority = await make_school_authority(
            plugin_name="id_broker", **config
        )

        return school_authority

    return _func


@pytest.mark.parametrize("role", ["student", "teacher", "legal_guardian"])
@pytest.mark.asyncio
async def test_sync_user_multiple_config_support(
    kelvin_schools_on_sender,
    wait_for_kelvin_user_on_id_broker,
    new_id_broker_school_auth_conf,
    make_sender_user_special,
    role,
    schedule_schools_delete,
    set_school_authority_on_id_connector_special,
    temp_clear_dir,
):
    """
    Using multiple school authorities
    if there is no intersection of the sets of the configured schools
    Comment: This should also work if two id broker targets are used.
    """
    # save existing school authority configurations
    temp_clear_dir(
        Path(
            "/var/lib/univention-appcenter/apps/ucsschool-id-connector/conf/school_authorities/"
        )
    )
    (
        sender_school__authority_1,
        sender_school__authority_2,
    ) = await kelvin_schools_on_sender(2)
    conf_authority_1 = await set_school_authority_on_id_connector_special(
        school=sender_school__authority_1.name,
    )
    conf_authority_2 = await set_school_authority_on_id_connector_special(
        school=sender_school__authority_2.name,
    )
    await schedule_schools_delete(
        school_authority=conf_authority_1, schools=[sender_school__authority_1.name]
    )
    await schedule_schools_delete(
        school_authority=conf_authority_2, schools=[sender_school__authority_2.name]
    )
    sender_user__school_authority_1 = await make_sender_user_special(
        ous=[sender_school__authority_1.name],
        s_a_name=conf_authority_1.name,
        roles=(role,),
    )
    sender_user__school_authority_2 = await make_sender_user_special(
        ous=[sender_school__authority_2.name],
        s_a_name=conf_authority_2.name,
        roles=(role,),
    )
    for school, config, user in [
        (sender_school__authority_1, conf_authority_1, sender_user__school_authority_1),
        (sender_school__authority_2, conf_authority_2, sender_user__school_authority_2),
    ]:
        async for attempt in AsyncRetrying(
            reraise=True,
            stop=stop_after_attempt(6),
            retry=retry_if_exception_type(AssertionError),
            wait=wait_fixed(2),
        ):
            with attempt:
                remote_user: KelvinUser = await wait_for_kelvin_user_on_id_broker(
                    s_a_name=config.name, school=school.name, name=user["name"]
                )
                for key, value in user["school_classes"].items():
                    assert f"{config.name}-{key}" in remote_user.school_classes
                    assert set(value) == set(
                        remote_user.school_classes[f"{config.name}-{key}"]
                    )
