# -*- coding: utf-8 -*-
# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.

import re
from typing import Set, Tuple, Union

from ucsschool_id_connector.models import (
    ListenerGroupAddModifyObject,
    ListenerGroupRemoveObject,
    ListenerObject,
    ListenerUserAddModifyObject,
    ListenerUserRemoveObject,
    SchoolAuthorityConfiguration,
)
from ucsschool_id_connector.utils import school_class_dn_regex, workgroup_dn_regex

ListenerGroupType = Union[ListenerGroupAddModifyObject, ListenerGroupRemoveObject]


ListenerUserType = Union[ListenerUserAddModifyObject, ListenerUserRemoveObject]


def school_in_configured_schools(school: str, configured_schools: Set[str]):
    for configured_school in configured_schools:
        search_regex = ".*".join(
            [re.escape(part) for part in configured_school.split("*")]
        )
        if re.match(search_regex, school):
            return True
    return False


def group_event_on_configured_schools(
    obj: ListenerGroupType, configured_schools: Set[str]
) -> bool:
    return group_dn_has_configured_school(obj.dn, configured_schools)


def user_event_on_configured_schools(
    obj: ListenerUserType, configured_schools: Set[str]
) -> bool:
    old_schools = get_old_schools(obj)
    new_schools = get_new_schools(obj)
    for school in new_schools.union(old_schools):
        if school_in_configured_schools(school, configured_schools):
            return True
    return False


def user_has_left_configured_schools(
    obj: ListenerUserType, configured_schools: Set[str]
) -> bool:
    old_schools = get_old_schools(obj)
    new_schools = get_new_schools(obj)
    old_has_configured_schools = any(
        school_in_configured_schools(school, configured_schools)
        for school in old_schools
    )
    new_has_configured_schools = any(
        school_in_configured_schools(school, configured_schools)
        for school in new_schools
    )
    return old_has_configured_schools and not new_has_configured_schools


def user_has_unconfigured_schools(
    obj: ListenerUserType, configured_schools: Set[str]
) -> bool:
    new_schools = get_new_schools(obj)
    new_has_unconfigured_schools = any(
        not school_in_configured_schools(school, configured_schools)
        for school in new_schools
    )
    return new_has_unconfigured_schools


def get_configured_schools(sac: SchoolAuthorityConfiguration) -> set:
    return {school.lower() for school in sac.plugin_configs["id_broker"]["schools"]}


def get_new_schools(obj: ListenerUserType) -> set:
    return {school.lower() for school in getattr(obj, "schools", [])}


def get_old_schools(obj: ListenerUserType) -> set:
    return {school.lower() for school in getattr(obj.old_data, "schools", [])}


def obj_event_on_configured_schools(
    obj: ListenerObject, sac: SchoolAuthorityConfiguration
) -> bool:
    configured_schools = get_configured_schools(sac)
    if isinstance(obj, ListenerGroupAddModifyObject) or isinstance(
        obj, ListenerGroupRemoveObject
    ):
        return group_event_on_configured_schools(obj, configured_schools)
    return user_event_on_configured_schools(obj, configured_schools)


def role_has_configured_school(role_string: str, configured_schools: Set[str]) -> bool:
    role, context_type, context = get_role_info(role_string)
    return context_type == "school" and school_in_configured_schools(
        context.lower(), configured_schools
    )


def group_dn_has_configured_school(dn: str, configured_schools: Set[str]) -> bool:
    m = school_class_dn_regex().match(dn) or workgroup_dn_regex().match(dn)
    if m:
        school = m.groupdict()["ou"].lower()
        return school_in_configured_schools(school, configured_schools)
    return False


def get_role_info(ucsschool_role_string: str) -> Tuple[str, str, str]:
    """

    This function separates the individual elements of an ucsschool role string.
    Adapted from ucsschool/ucs-school-lib/modules/ucsschool/lib/roles.py
    :param ucsschool_role_string: The role string to separate
    :return: (role, context_type, context)
    """
    role, context_type, context = ucsschool_role_string.split(":")
    return role, context_type, context
