import copy
import json
from functools import lru_cache
from pathlib import Path
from typing import Any, Dict

from ucsschool.apis.utils import LDAPCredentials, LDAPSettings


def remove_prefix(value: str, prefix: str) -> str:
    """
    Removes the given prefix from the given string.

    >>> remove_prefix("prefix-some_string", "prefix-")
    'some_string'

    :param value: The string to remove the prefix from
    :param prefix: The prefix to remove from the string
    :return: The string without the prefix
    :raises ValueError: If the given string does not start with the given prefix
    """
    if not value.startswith(prefix):
        raise ValueError(f"The value {value} must begin with the prefix: {prefix}")
    return value.removeprefix(prefix)


@lru_cache(maxsize=1)
def ldap_settings() -> LDAPSettings:
    """Cached LDAP settings object"""
    return LDAPSettings()


@lru_cache(maxsize=1)
def ldap_credentials() -> LDAPCredentials:
    """Cached LDAP credentials object"""
    return LDAPCredentials(ldap_settings())


def load_settings(path: Path) -> Dict[str, Any]:
    try:
        with path.open() as settings_file:
            return json.load(settings_file)
    except (EnvironmentError, ValueError) as exc:
        raise ValueError(
            f"Missing or malformed settings file '{path!s}': {exc!s}"
        ) from exc


def remove_passwords(settings: Dict[str, Any]) -> Dict[str, Any]:
    res = copy.deepcopy(settings)
    for k in res:
        if "password" in k.lower() or "token" in k.lower():
            res[k] = "********"
    return res
