import base64
import bz2
import json
import time
from functools import lru_cache
from typing import Dict

from async_lru import alru_cache
from ldap3.utils.dn import to_dn

from ucsschool.apis.constants import CN_ADMIN_PASSWORD_FILE
from ucsschool.apis.utils import LDAPCredentials, get_correlation_id, get_logger
from udm_rest_client import UDM, NoObject

from .exceptions import (
    IdBrokerPseudonymisationDataError,
    IdBrokerPseudonymisationNoSecret,
)
from .utils import ldap_settings

SP_MAPPINGS_DN = (
    "cn=IDBrokerServiceProviderMappings,cn=IDBrokerSettings,cn=univention,{ldap_base}"
)
SP_SECRETS_DN = (
    "cn=IDBrokerServiceProviderSecrets,cn=IDBrokerSettings,cn=univention,{ldap_base}"
)

logger = get_logger()


@lru_cache(maxsize=1)
def udm_kwargs() -> Dict[str, str]:
    """Cached UDM REST API client credentials using the Docker apps host DN."""
    _ldap_settings = ldap_settings()
    _ldap_credentials = LDAPCredentials(_ldap_settings)
    username = to_dn(_ldap_credentials.host_dn, decompose=True)[0][1]
    username = f"{username}$"
    password = _ldap_credentials.machine_password
    return {
        "username": username,
        "password": password,
        "url": f"https://{_ldap_settings.host_fqdn}/univention/udm/",
        "ssl_ca_cert": "/etc/ssl/certs/ca-certificates.crt",
    }


@lru_cache(maxsize=1)
def get_admin_connection_kwargs():
    ldap_setting = ldap_settings()
    with open(CN_ADMIN_PASSWORD_FILE, "r") as fp:
        bindpw = fp.read().strip()
    return {
        "username": "cn=admin",
        "password": bindpw,
        "url": f"https://{ldap_setting.master_fqdn}/univention/udm/",
        "ssl_ca_cert": "/etc/ssl/certs/ca-certificates.crt",
    }


def decode_settings_data(value: bytes) -> bytes:
    return bz2.decompress(base64.b64decode(value))


async def get_settings_data_content(dn: str) -> Dict[str, str]:
    """
    Get `data` attribute of a `settings/data` object.

    :raises IdBrokerPseudonymisationDataError: when `settings/data` UDM object does not exist or content
        is not valid JSON
    """
    _udm_kwargs = udm_kwargs()
    _udm_kwargs["request_id"] = get_correlation_id()
    async with UDM(**_udm_kwargs) as udm:
        mod = udm.get("settings/data")
        try:
            obj = await mod.get(dn)
        except NoObject as exc:
            raise IdBrokerPseudonymisationDataError(f"Loading {dn!r}: {exc!s}") from exc
        try:
            raw_data = decode_settings_data(obj.props.data)
            data = json.loads(raw_data)
            if not isinstance(data, dict):
                raise ValueError(f"Expected a dict, found: {type(data)!r}")
        except ValueError as exc:
            raise IdBrokerPseudonymisationDataError(
                f"Invalid data in {dn!r}: {exc!s}"
            ) from exc
        return data


@alru_cache(maxsize=30)
async def _cached_get_service_provider_secret(s_p: str) -> str:
    """Attention: Cache will also contain exceptions for missing service providers."""
    ldap_base = ldap_settings().ldap_base
    dn = SP_SECRETS_DN.format(ldap_base=ldap_base)
    t0 = time.time()
    data = await get_settings_data_content(dn)
    logger.debug(
        "Retrieved service provider secret for %r from UDM REST API in %0.3f sec.",
        s_p,
        time.time() - t0,
    )
    try:
        return data[s_p]
    except KeyError as exc:
        raise IdBrokerPseudonymisationNoSecret(
            f"No secret for SP {s_p!r} found. Known SPs: {sorted(data.keys())!r}"
        ) from exc


async def get_service_provider_secret(service_provider: str) -> str:
    """
    Get secret for service provider.

    Only successful lookups will be cached, so the admin can later add a service provider and the API
    server will not have to be restarted.

    :raises IdBrokerPseudonymisationDataError: when `settings/data` UDM object does not exist or content
        is not valid JSON
    :raises IdBrokerPseudonymisationNoSecret: when `service_provider` does not exist in `settings/data`
        UDM object
    """
    try:
        return await _cached_get_service_provider_secret(service_provider)
    except IdBrokerPseudonymisationNoSecret:
        # Remove exception from cache. We only cache successful lookups.
        _cached_get_service_provider_secret.invalidate(service_provider)
        raise
