import json
from functools import lru_cache
from typing import Dict, Optional

from univention.config_registry import ConfigRegistry
from univention.udm import UDM, NoObject


class IdBrokerPseudonymisationError(Exception): ...


class IdBrokerPseudonymisationNoMapping(IdBrokerPseudonymisationError): ...


class IdBrokerPseudonymisationNoSecret(IdBrokerPseudonymisationError): ...


class IdBrokerPseudonymisationDataError(IdBrokerPseudonymisationError): ...


@lru_cache(maxsize=1)
def ldap_base() -> str:
    return ConfigRegistry().load().get("ldap/base")


def get_service_provider_attribute_dn() -> str:
    return f"cn=IDBrokerServiceProviderMappings,cn=IDBrokerSettings,cn=univention,{ldap_base()}"


def get_service_provider_secret_dn() -> str:
    return f"cn=IDBrokerServiceProviderSecrets,cn=IDBrokerSettings,cn=univention,{ldap_base()}"


def get_settings_data_content(dn: str, udm: Optional[UDM] = None) -> Dict[str, str]:
    """
    Get `data` attribute of a `settings/data` object.

    :raises IdBrokerPseudonymisationDataError: when `settings/data` UDM object does not exist or content
        is not valid JSON
    """
    if udm is None:
        udm = UDM.machine().version(2)
    try:
        obj = udm.get("settings/data").get(dn)
    except NoObject as exc:
        raise IdBrokerPseudonymisationDataError(f"Loading {dn!r}: {exc!s}") from exc
    try:
        data = json.loads(obj.props.data.raw)
        if not isinstance(data, dict):
            raise ValueError(f"Expected a dict, found: {type(data)!r}")
    except ValueError as exc:
        raise IdBrokerPseudonymisationDataError(
            f"Invalid data in {dn!r}: {exc!s}"
        ) from exc
    return data


def get_service_provider_attribute(
    service_provider: str, udm: Optional[UDM] = None
) -> str:
    """
    Get LDAP/UDM attribute name for pseudonyms for service provider.

    :raises IdBrokerPseudonymisationDataError: when `settings/data` UDM object does not exist or content
        is not valid JSON
    :raises IdBrokerPseudonymisationNoMapping: when `service_provider` does not exist in `settings/data`
        UDM object
    """
    data = get_settings_data_content(get_service_provider_attribute_dn(), udm=udm)
    try:
        return data[service_provider]
    except KeyError as exc:
        raise IdBrokerPseudonymisationNoMapping(
            f"No attribute mapping for SP {service_provider!r} found. Known mappings: "
            f"{sorted(data.keys())!r}"
        ) from exc


def get_service_provider_secret(
    service_provider: str, udm: Optional[UDM] = None
) -> str:
    """
    Get secret for service provider.

    :raises IdBrokerPseudonymisationDataError: when `settings/data` UDM object does not exist or content
        is not valid JSON
    :raises IdBrokerPseudonymisationNoSecret: when `service_provider` does not exist in `settings/data`
        UDM object
    """
    data = get_settings_data_content(get_service_provider_secret_dn(), udm=udm)
    try:
        return data[service_provider]
    except KeyError as exc:
        raise IdBrokerPseudonymisationNoSecret(
            f"No secret for SP {service_provider!r} found. Known SPs: {sorted(data.keys())!r}"
        ) from exc
