import base64
import bz2
import json

import pytest

from id_broker_common.exceptions import IdBrokerPseudonymisationNoSecret
from id_broker_common.pseudonyms_udm_rest import (
    get_service_provider_secret,
    get_settings_data_content,
    udm_kwargs,
)
from ucsschool.apis.utils import LDAPSettings
from udm_rest_client import UDM
from udm_rest_client.exceptions import APICommunicationError

from ..utils import must_run_in_docker

pytestmark = must_run_in_docker
HOST_ACCOUNT_FILE = "/var/lib/univention-appcenter/apps/ucsschool-apis/data/hostaccount.txt"
HOST_PASSWORD_FILE = "/var/lib/univention-appcenter/apps/ucsschool-apis/data/hostpassword.txt"


@pytest.fixture()
def ldap_settings():
    return LDAPSettings()


@pytest.fixture()
def mappings_settings_data_dn(ldap_settings):
    ldap_base = ldap_settings.ldap_base
    return f"cn=IDBrokerServiceProviderMappings,cn=IDBrokerSettings,cn=univention,{ldap_base}"


@pytest.fixture()
def secrets_settings_data_dn(ldap_settings):
    ldap_base = ldap_settings.ldap_base
    return f"cn=IDBrokerServiceProviderSecrets,cn=IDBrokerSettings,cn=univention,{ldap_base}"


# these are the credentials of the docker host,
# copied into te container before the test starts
@pytest.fixture()
def host_credentials(ldap_settings):
    return {
        "username": open(HOST_ACCOUNT_FILE).read().strip(),
        "password": open(HOST_PASSWORD_FILE).read().strip(),
        "url": f"https://{ldap_settings.master_fqdn}/univention/udm/",
        "ssl_ca_cert": "/etc/ssl/certs/ca-certificates.crt",
    }


# these are the credentials of the app account
@pytest.fixture()
def app_credentials():
    return udm_kwargs()


@pytest.fixture()
@pytest.mark.asyncio
async def modify_settings():
    settings = []

    async def _modify_settings(credentials, dn, setting):
        async with UDM(**credentials) as udm:
            mod = udm.get("settings/data")
            obj = await mod.get(dn)
            settings.append((credentials, dn, obj.props.data))
            obj.props.data = base64.b64encode(bz2.compress(json.dumps(setting).encode("utf-8"))).decode(
                "utf-8"
            )
            await obj.save()

    yield _modify_settings
    for credentials, dn, data in settings:
        async with UDM(**credentials) as udm:
            mod = udm.get("settings/data")
            obj = await mod.get(dn)
            obj.props.data = data
            await obj.save()


@pytest.mark.asyncio
async def test_get_service_provider_secret(host_credentials, secrets_settings_data_dn, modify_settings):
    setting = {"idp": "secret"}
    await modify_settings(host_credentials, secrets_settings_data_dn, setting)
    data = await get_settings_data_content(secrets_settings_data_dn)
    assert data == setting
    secret = await get_service_provider_secret("idp")
    assert secret == "secret"
    with pytest.raises(IdBrokerPseudonymisationNoSecret):
        await get_service_provider_secret("does_not_exists")


@pytest.mark.parametrize(
    "settings_dn",
    [
        "mappings_settings_data_dn",
        "secrets_settings_data_dn",
    ],
)
@pytest.mark.asyncio
async def test_app_account_not_allowed_to_write(app_credentials, settings_dn, modify_settings, request):
    setting = {"idp": "mapping"}
    settings_dn = request.getfixturevalue(settings_dn)
    with pytest.raises(APICommunicationError):
        await modify_settings(app_credentials, settings_dn, setting)


@pytest.mark.parametrize(
    "settings_dn",
    [
        "mappings_settings_data_dn",
        "secrets_settings_data_dn",
    ],
)
@pytest.mark.asyncio
async def test_get_settings_data_content_is_json(settings_dn, request):
    settings_dn = request.getfixturevalue(settings_dn)
    data = await get_settings_data_content(settings_dn)
    json.dumps(data)
