import abc
import datetime
from enum import Enum
from typing import Any, Dict, List, Literal, NamedTuple, Optional, TypedDict

GroupType = Literal["school_class", "workgroup"]


class NotFoundError(Exception):
    ...


class SddbUser(NamedTuple):
    pseudonym: str  # value of idBrokerPseudonym00..
    api_version: int  # version of format of object in SDDB
    name: str  # UID
    school: str  # OU
    schools: List[str]  # OU1, OU2, ...
    school_authority: str
    school_classes: List[str]  # list of IDs (entryUUID)
    workgroups: List[str]  # list of IDs (entryUUID)
    modifyTimestamp: datetime.datetime
    data: Dict[str, Any]  # complete DataSource object (Kelvin User) without pseudonyms

    def __repr__(self):
        return (
            f"{self.__class__.__name__}(pseudonym={self.pseudonym!r}, name: {self.name!r} "
            f"school={self.school!r} school_authority={self.school_authority!r} "
            f"school_classes: {len(self.school_classes)}, workgroups: {len(self.workgroups)})"
        )


class SddbGroup(NamedTuple):
    pseudonym: str  # value of idBrokerPseudonym00..
    api_version: int  # version of format of object in SDDB
    type: GroupType
    name: str  # CN
    school: str  # OU
    school_authority: str
    modifyTimestamp: datetime.datetime
    students: List[str]  # SDDB IDs, use SdDBClient.get_users() to retrieve referenced users
    teachers: List[str]  # SDDB IDs, use SdDBClient.get_users() to retrieve referenced users
    data: Dict[str, Any]  # complete DataSource obj (Kelvin SchoolClass / WorkGroup) without pseudonyms

    def __repr__(self):
        return (
            f"{self.__class__.__name__}(pseudonym={self.pseudonym!r}, type={self.type!r}, "
            f"name: {self.name!r} school={self.school!r} school_authority={self.school_authority!r} "
            f"students: {len(self.students)}, teachers: {len(self.teachers)})"
        )


class SddbServiceProviderMapping(NamedTuple):
    api_version: int  # version of format of object in SDDB
    name: str  # CN
    mapping: Dict[str, str]  # SP to LDAP attribute name mapping

    def __repr__(self):
        return f"{self.__class__.__name__}(name: {self.name!r} mapping={self.mapping!r})"


class SddbInternalUser(TypedDict):
    """
    Internal representation of a user in SDDB including all SP specific pseudonyms.

    The format of this dict is not stable!

    This is a dict representation of RedisUser from `redis_sddb.py`, which is copied from SDDB.
    Keep it in sync!
    """

    id: str  # entryUUID
    api_version: int  # version of format of object in SDDB
    name: str  # UID
    school: str  # OU
    schools: List[str]  # OU1, OU2, ...
    school_authority: str
    modifyTimestamp: str  # "2023-04-28T14:26:40"
    p0001: str  # idBrokerPseudonym0001
    p0002: str  # idBrokerPseudonym0002
    p0003: str  # ...
    p0004: str
    p0005: str
    p0006: str
    p0007: str
    p0008: str
    p0009: str
    p0010: str
    p0011: str
    p0012: str
    p0013: str
    p0014: str
    p0015: str
    p0016: str
    p0017: str
    p0018: str
    p0019: str
    p0020: str
    p0021: str
    p0022: str
    p0023: str
    p0024: str
    p0025: str
    p0026: str
    p0027: str
    p0028: str
    p0029: str
    p0030: str
    data: Dict[str, Any]  # complete DataSource object (Kelvin User) without pseudonyms


class APIQueueEntryType(str, Enum):
    user = "user"
    group = "group"
    sp_mapping = "sp_mapping"


class SdDBClient(abc.ABC):
    @abc.abstractmethod
    def get_user(self, pseudonym: str, service_provider_id: int) -> SddbUser:
        """
        Retrieve a user object from SDDB.

        :param str pseudonym: the users pseudonym (service provider specific)
        :param int service_provider_id: ID of the connecting service provider
        :return: SddbUser object
        :raises NotFoundError: when user was not found.
        """

    @abc.abstractmethod
    def get_users(self, pks: List[str]) -> List[Optional[SddbInternalUser]]:
        """
        Retrieve multiple user objects from SDDB in their in-DB representation.
        Use this to efficiently retrieve group members.

        :param list(str) pks: primary database keys in SDDB
        :return: list of dicts (SddbInternalUser) with the users in-DB representation or None for every
            user not found
        """

    @abc.abstractmethod
    def get_group_by_pseudonym(self, pseudonym: str, service_provider_id: int) -> SddbGroup:
        """
        Retrieve school class using a service provider specific pseudonym.

        :param pseudonym: the groups pseudonym (service provider specific)
        :param int service_provider_id: ID of the connecting service provider
        :return: SddbGroup object, `SddbGroup.pseudonym` will be set to the correct value for the SP
            referenced by `service_provider_id`
        :raises NotFoundError: when group was not found.
        """

    @abc.abstractmethod
    def get_group_by_id(self, group_id: str, service_provider_id: int) -> SddbGroup:
        """
        Retrieve school class using the groups entryUUID.

        :param str group_id: the school class' entryUUID in LDAP
        :param int service_provider_id: ID of the connecting service provider
        :return: SddbGroup object, `SddbGroup.pseudonym` will be set to the correct value for the SP
            referenced by `service_provider_id`
        :raises NotFoundError: when group was not found.
        """

    @abc.abstractmethod
    def get_service_provider_mapping(self) -> SddbServiceProviderMapping:
        """
        Retrieve service provider mapping.

        :return: SddbServiceProviderMapping object
        :raises NotFoundError: when service provider mapping was not found.
        """


class SdDBRestClient(abc.ABC):
    @abc.abstractmethod
    def enqueue(
        self,
        service_provider: str,
        entry_type: APIQueueEntryType,
        pseudonym: str = None,
        entry_uuid: str = None,
    ) -> None:
        """
        Add an entry to SDDBs 'high' priority conversion queue using the SDDB REST API.

        In case the `entry_type` is `sp_mapping`, the values of `service_provider` and `pseudonym` will
        be ignored and can be empty strings.

        :param str entry_uuid: the entry_uuid of the requested object
        :param str service_provider: the connecting service provider
        :param str pseudonym: the pseudonym of the requested object
        :param entry_type: the type of object that was requested / should be enqueued
        """
