from typing import Dict

from fastapi import APIRouter, Depends, HTTPException, Path, Response, status
from ldap3.utils.conv import escape_filter_chars
from pydantic import ValidationError

from id_broker_common.kelvin import kelvin_session
from id_broker_common.ldap_access import ldap_access, wait_for_replication
from provisioning_plugin.dependencies import check_for_authority_admin, get_user_by_id
from provisioning_plugin.models import NonEmptyStr, NoStarStr, SchoolContext, User
from provisioning_plugin.utils import get_pseudonyms_for_providers
from ucsschool.kelvin.client import InvalidRequest, Session, User as KelvinUser

router = APIRouter(tags=["users"], dependencies=[Depends(check_for_authority_admin)])


@router.head("/users/{id}", response_model=None)
@router.get("/users/{id}", response_model=User)
async def get(
    user_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    user: KelvinUser = Depends(get_user_by_id),
) -> User:
    try:
        return User.without_prefix(User.from_kelvin_user(user.as_dict()), school_authority)
    except (ValidationError, ValueError) as exc:
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"The user {user_id!r} in school authority {school_authority!r} is malformed.",
        ) from exc


@router.post("/users", response_model=User, status_code=201)
async def post(
    user_data: User,
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    session: Session = Depends(kelvin_session),
) -> User:
    user_data = User.with_prefix(user_data, school_authority)
    await _check_user_dependencies(user_data.context)
    user = KelvinUser(
        school=user_data.school,
        schools=user_data.schools,
        school_classes=user_data.school_classes,
        workgroups=user_data.workgroups,
        roles=user_data.context[user_data.school].roles,
        name=user_data.user_name,
        firstname=user_data.first_name,
        lastname=user_data.last_name,
        source_uid=school_authority,
        record_uid=user_data.id,
        session=session,
    )
    async for udm_prop, pseudonym in get_pseudonyms_for_providers(user_data.id, school_authority):
        user.udm_properties[udm_prop] = pseudonym
    try:
        await user.save()
    except InvalidRequest as exc:
        if exc.status == status.HTTP_409_CONFLICT:
            await user.reload()
            raise HTTPException(
                status_code=exc.status, detail={"msg": exc.reason, "conflict_id": user.record_uid}
            ) from exc
        else:
            raise HTTPException(status_code=exc.status, detail=exc.reason) from exc
    await wait_for_replication(ldap_access(), user.dn)
    return User.without_prefix(User.from_kelvin_user(user.as_dict()), school_authority)


@router.put("/users/{id}", response_model=User)
async def put(
    user_data: User,
    user_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    user: KelvinUser = Depends(get_user_by_id),
    session: Session = Depends(kelvin_session),
) -> User:
    if user_data.id != user.record_uid:
        raise HTTPException(
            status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
            detail="The user_id must not be changed after creation!",
        )
    user_data = User.with_prefix(user_data, school_authority)
    await _check_user_dependencies(user_data.context)
    user.school = user_data.school
    user.schools = user_data.schools
    user.school_classes = user_data.school_classes
    user.workgroups = user_data.workgroups
    user.roles = user_data.context[user_data.school].roles
    user.name = user_data.user_name
    user.firstname = user_data.first_name
    user.lastname = user_data.last_name
    try:
        await user.save()
        return User.without_prefix(User.from_kelvin_user(user.as_dict()), school_authority)
    except InvalidRequest as exc:
        raise HTTPException(status_code=exc.status, detail=exc.reason) from exc
    except (ValidationError, ValueError) as exc:
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"The user {user_id!r} in school authority {school_authority!r} is malformed!",
        ) from exc


@router.delete("/users/{id}", status_code=status.HTTP_204_NO_CONTENT)
async def delete(
    user_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    user: KelvinUser = Depends(get_user_by_id),
) -> Response:
    await user.delete()
    await wait_for_replication(ldap_access(), user.dn, should_exist=False)
    return Response(status_code=status.HTTP_204_NO_CONTENT)


async def _check_object_exists(ldap_filter: str, obj: str, obj_class: str) -> None:
    ldap = ldap_access()
    exists = await ldap.search(ldap_filter, use_master=True)
    if not exists:
        raise HTTPException(
            status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
            detail=f"{obj_class} {obj!r} does not exist yet.",
        )


async def _check_school_exists(school: str) -> None:
    ldap = ldap_access()
    exists = await ldap.search(
        f"(&(objectClass=ucsschoolOrganizationalUnit)(ou={escape_filter_chars(school)}))",
        use_master=True,
    )
    if not exists:
        raise HTTPException(
            status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
            detail=f"School {school!r} does not exist yet.",
        )


async def _check_school_class_exists(school: str, school_class: str) -> None:
    ldap = ldap_access()
    exists = await ldap.search(
        f"(&(ucsschoolRole=school_class:*)"
        f"(cn={escape_filter_chars(school)}-{escape_filter_chars(school_class)}))",
        use_master=True,
    )
    if not exists:
        raise HTTPException(
            status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
            detail=f"School class {school_class!r} does not exist yet.",
        )


async def _check_workgroup_exists(school: str, workgroup: str) -> None:
    ldap = ldap_access()
    exists = await ldap.search(
        f"(&(ucsschoolRole=workgroup:*)(cn={escape_filter_chars(school)}-{escape_filter_chars(workgroup)}))",
        use_master=True,
    )
    if not exists:
        raise HTTPException(
            status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
            detail=f"Workgroup {workgroup!r} does not exist yet.",
        )


async def _check_user_dependencies(context: Dict[str, SchoolContext]) -> None:
    """Use Ldap to verify existence of schools, school classes & workgroups. May raise HTTPException(422)."""
    for school, school_context in context.items():
        await _check_school_exists(school)
        for school_class in school_context.classes:
            await _check_school_class_exists(school=school, school_class=school_class)
        for workgroup in school_context.workgroups:
            await _check_workgroup_exists(school=school, workgroup=workgroup)
