# -*- coding: utf-8 -*-

# Copyright 2019-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.

import asyncio
import datetime
import json
import os
from pathlib import Path
from typing import Any, Dict, List, Optional, Tuple, Type
from urllib.parse import urljoin

import faker
import pytest
import pytest_asyncio

import ucsschool_id_connector.plugin_loader
from ucsschool.kelvin.client import (
    KelvinObject,
    KelvinResource,
    NoObject,
    School,
    SchoolClass,
    SchoolClassResource,
    SchoolResource,
    Session,
    User,
    UserResource,
    WorkGroup,
    WorkGroupResource,
)
from ucsschool_id_connector.ldap_access import LDAPAccess
from ucsschool_id_connector.models import SchoolAuthorityConfiguration

# load ID Broker plugin
ucsschool_id_connector.plugin_loader.load_plugins()
id_broker = pytest.importorskip("idbroker")
pytestmark = pytest.mark.id_broker

from idbroker.id_broker_client import IDBrokerSchool  # isort:skip  # noqa: E402
from idbroker.initial_sync import get_ous  # isort:skip  # noqa: E402

fake = faker.Faker()


@pytest.fixture(scope="session")
def event_loop(request):
    """Create an instance of the default event loop for each test case."""
    loop = asyncio.get_event_loop_policy().new_event_loop()
    yield loop
    loop.close()


def _id_broker_ip(docker_hostname: str, http_request, file_name: str) -> str:
    url = urljoin(f"https://{docker_hostname}", file_name)
    resp = http_request("get", url, verify=False)
    assert resp.status_code == 200, (resp.status_code, resp.reason, url)
    return resp.text.strip()


@pytest.fixture(scope="session")
def id_broker_ip(docker_hostname: str, http_request) -> str:
    return _id_broker_ip(docker_hostname, http_request, "IP_idbroker_primary.txt")


@pytest.fixture(scope="session")
def id_broker_ip_provisioning(docker_hostname: str, http_request) -> str:
    return _id_broker_ip(docker_hostname, http_request, "IP_idbroker_provisioning.txt")


def _id_broker_fqdn(http_request, ip, docker_hostname):
    resp = http_request(
        "get",
        f"https://Administrator:univention@{docker_hostname}"
        f"/univention/udm/dns/host_record/?property=a&query[]={ip}",
        headers={"Accept": "application/json"},
        verify=False,
    )
    resp_json = resp.json()
    dn: str = resp_json["_embedded"]["udm:object"][0]["dn"]
    relative_domain_name, zone_name = [s.split("=")[1] for s in dn.split(",")[:2]]
    fqdn = f"{relative_domain_name}.{zone_name}"
    return fqdn


@pytest.fixture(scope="session")
def id_broker_fqdn(http_request, id_broker_ip, docker_hostname) -> str:
    return _id_broker_fqdn(http_request, id_broker_ip, docker_hostname)


@pytest.fixture(scope="session")
def id_broker_fqdn_provisioning(http_request, id_broker_ip_provisioning, docker_hostname) -> str:
    return _id_broker_fqdn(http_request, id_broker_ip_provisioning, docker_hostname)


@pytest.fixture(scope="session")
def id_broker_session(id_broker_fqdn, kelvin_session):
    return kelvin_session(id_broker_fqdn)


@pytest.fixture(scope="session")
def school_auth_config_id_broker(id_broker_fqdn_provisioning):
    """
    Fixture to create configurations for replicating to the ID Broker.
    """

    def _school_auth_config_id_broker(
        s_a_name: str, password: str, schools: List[str] = []
    ) -> Dict[str, Any]:
        """
        Generates a configuration for a school authority.

        :return: The school authority configuration in dictionary form
        """
        return {
            "name": s_a_name,
            "active": True,
            "url": f"https://{id_broker_fqdn_provisioning}/",
            "plugins": ["id_broker-users", "id_broker-groups"],
            "plugin_configs": {
                "id_broker": {
                    "password": password,
                    "username": f"provisioning-{s_a_name}",
                    "schools": schools,
                    "version": 1,
                },
            },
        }

    return _school_auth_config_id_broker


def get_id_broker_school_auth():
    """
    get provisioning-admin user on the id broker system.
    """
    s_a_name = os.environ["DOCKER_HOST_NAME"].split(".", 1)[0].capitalize()
    config_file_name = f"{s_a_name}.json"
    config_path = "/var/lib/univention-appcenter/apps/ucsschool-id-connector/conf/school_authorities/"
    with open(os.path.join(config_path, config_file_name), "r") as config_fd:
        config = json.load(config_fd)
    password = config["plugin_configs"]["id_broker"]["password"]
    return s_a_name, password


@pytest.fixture(scope="session")
def id_broker_school_auth_conf(temp_clear_dir, school_auth_config_id_broker) -> Dict[str, Any]:
    s_a_name, password = get_id_broker_school_auth()
    # save existing school authority configurations
    temp_clear_dir(
        Path("/var/lib/univention-appcenter/apps/ucsschool-id-connector/conf/school_authorities/")
    )
    yield school_auth_config_id_broker(s_a_name, password)


@pytest_asyncio.fixture()
def set_school_authority_on_id_connector(id_broker_school_auth_conf, make_school_authority):
    async def _func(schools, active=True):
        id_broker_school_auth_conf["plugin_configs"]["id_broker"]["schools"] = schools
        id_broker_school_auth_conf["active"] = active
        school_authority = await make_school_authority(
            plugin_name="id_broker", **id_broker_school_auth_conf
        )
        return school_authority

    return _func


@pytest_asyncio.fixture
def kelvin_schools_on_sender(create_school, id_connector_host_name):
    async def _func(num_schools: int = 2):
        return [await create_school(id_connector_host_name) for _ in range(num_schools)]

    return _func


@pytest_asyncio.fixture
def delete_school_on_id_broker(wait_for_kelvin_school_on_id_broker):
    async def _func(config, schools):
        school_client = IDBrokerSchool(
            school_authority=config,
            plugin_name="id_broker",
        )
        for _school in schools:
            schools = await get_ous(ou=_school)
            if not schools:
                continue
            school = schools[0]
            name = f"{config.name}-{school.name}"
            school_exists = await school_client.exists(school.id)
            if not school_exists:
                print(f"School {name} was already removed.")
                continue
            await school_client.delete(school.id)
            await wait_for_kelvin_school_on_id_broker(
                name=school, s_a_name=config.name, must_exist=False
            )

    return _func


@pytest.fixture
def delete_school_on_host(docker_hostname: str, http_request):
    def _func(schools):
        ldap_base = os.environ["LDAP_BASE"]
        req_headers = {
            "accept": "application/json",
        }
        for school in schools:
            school_dn = f"ou={school},{ldap_base}"
            url = urljoin(
                f"https://Administrator:univention@{docker_hostname}/univention/udm/container/ou/",
                school_dn,
            )
            http_request("delete", url, verify=False, headers=req_headers, expected_statuses=(204,))

    return _func


@pytest_asyncio.fixture
async def schedule_schools_delete(delete_school_on_id_broker, delete_school_on_host):
    _schools: List[School] = []
    _config: SchoolAuthorityConfiguration = None

    async def _func(school_authority: SchoolAuthorityConfiguration, schools: List[str]):
        nonlocal _schools, _config
        _schools = schools
        _config = school_authority

    yield _func
    await delete_school_on_id_broker(config=_config, schools=_schools)
    delete_school_on_host(schools=_schools)


@pytest.fixture(scope="session")
def kelvin_user_object(docker_hostname, kelvin_session, random_name, source_uid):
    def _func(roles=("student",), schools=("DEMOSCHOOL",), **kwargs) -> User:
        first_name = fake.first_name()
        last_name = fake.last_name()
        return User(
            name=kwargs.get("name", f"test.{first_name[:5]}.{last_name}"[:15]),
            birthday=kwargs.get("birthday", fake.date_of_birth(minimum_age=6, maximum_age=67)),
            expiration_date=kwargs.get(
                "expiration_date", fake.date_between(start_date="+1y", end_date="+10y")
            ),
            disabled=kwargs.get("disabled", False),
            firstname=kwargs.get("firstname", first_name),
            lastname=kwargs.get("lastname", last_name),
            password=kwargs.get("password", fake.password(length=15, special_chars=False)),
            record_uid=kwargs.get(
                "record_uid", f"{first_name[:5]}.{last_name}.{fake.pyint(1000, 9999)}"
            ),
            roles=roles,
            school=schools[0],
            schools=schools,
            workgroups=kwargs.get("workgroups", {}),
            school_classes=kwargs.get("school_classes", {})
            if roles == ("staff",)
            else {school: sorted([random_name(4), random_name(4)]) for school in schools},
            source_uid=kwargs.get("source_uid", source_uid),
            session=kwargs.get("session", kelvin_session(docker_hostname)),
        )

    return _func


@pytest_asyncio.fixture()
async def make_sender_user(
    kelvin_session,
    kelvin_session_kwargs,
    kelvin_user_object,
    docker_hostname,
    school_auth_host_configs,
):
    """
    Fixture factory to create users on the apps host system. They are created
    via the Kelvin-API and automatically removed when the fixture goes out of scope.
    """
    created_users: List[Dict[str, Any]] = []

    async def _make_sender_user(roles=("student",), ous=("DEMOSCHOOL",)):
        """
        Creates a user on the hosts UCS system via Kelvin-API-Client

        :param roles: The new users roles
        :param ous: The new users ous
        :return: The json used to create the user via the API
        """
        user_obj: User = kelvin_user_object(roles, ous, session=kelvin_session(docker_hostname))
        password = user_obj.password
        await user_obj.save()
        user_obj.password = password
        user_obj_as_dict = user_obj.as_dict()
        created_users.append(user_obj_as_dict)
        print("Created new User in source system: {!r}".format(user_obj_as_dict))
        return user_obj_as_dict

    yield _make_sender_user

    for host in (
        docker_hostname,
        school_auth_host_configs["IP_traeger1"],
        school_auth_host_configs["IP_traeger2"],
    ):
        for user_dict in created_users:
            print(f"Deleting user {user_dict['name']!r} from host {host!r}...")
            # Creating a new session per user deletion, because of a problem in httpx/httpcore/h11.
            # I think it has a problem with the disconnect happening by Kelvin with FastAPI delete():
            # h11._util.LocalProtocolError: Too much data for declared Content-Length
            async with Session(**kelvin_session_kwargs(host)) as session:
                try:
                    user = await UserResource(session=session).get(name=user_dict["name"])
                    await user.delete()
                    print(f"Success deleting user {user_dict['name']!r} from host {host!r}.")
                except NoObject:
                    print(f"No user {user_dict['name']!r} on {host!r}.")


@pytest.fixture(scope="session")
def wait_for_kelvin_object_exists():
    """
    Repeat executing `await resource_cls(session=session).method(**method_kwargs)`
    as long as `NoObject` is raised or until the timeout hits.
    """

    async def _func(
        resource_cls: Type[KelvinResource],
        method: str,
        session: Session,
        wait_timeout: int = 100,
        **method_kwargs,
    ) -> KelvinObject:
        end = datetime.datetime.now() + datetime.timedelta(seconds=wait_timeout)
        error: Optional[NoObject] = None
        while datetime.datetime.now() < end:
            resource = resource_cls(session=session)
            func = getattr(resource, method)
            try:
                res = await func(**method_kwargs)
                print(f"OK: received answer for {resource_cls.__name__}.{method}({method_kwargs!r})")
                return res
            except NoObject as exc:
                error = exc
                print(f"Waiting for {resource_cls.__name__}.{method}({method_kwargs!r}): {exc!s}")
                await session.close()
                await asyncio.sleep(1)
                session.open()
        raise AssertionError(f"No object found after {wait_timeout} seconds: {error!s}")

    return _func


@pytest_asyncio.fixture()
def wait_for_kelvin_school_on_id_broker(
    wait_for_kelvin_object_exists, wait_for_kelvin_object_not_exists, id_broker_session
):
    async def _func(name, s_a_name, must_exist=True):
        if must_exist:
            return await wait_for_kelvin_object_exists(
                resource_cls=SchoolResource,
                method="get",
                session=id_broker_session,
                name=f"{s_a_name}-{name}",
            )
        else:
            return await wait_for_kelvin_object_not_exists(
                resource_cls=SchoolResource,
                method="get",
                session=id_broker_session,
                name=f"{s_a_name}-{name}",
            )

    return _func


@pytest_asyncio.fixture()
def wait_for_kelvin_user_on_id_broker(
    wait_for_kelvin_object_exists, wait_for_kelvin_object_not_exists, id_broker_session
):
    async def _func(name, school, s_a_name, must_exist=True):
        if must_exist:
            return await wait_for_kelvin_object_exists(
                resource_cls=UserResource,
                method="get",
                session=id_broker_session,
                name=f"{s_a_name}-{name}",
                school=f"{s_a_name}-{school}",
            )
        else:
            return await wait_for_kelvin_object_not_exists(
                resource_cls=UserResource,
                method="get",
                session=id_broker_session,
                name=f"{s_a_name}-{name}",
                school=f"{s_a_name}-{school}",
            )

    return _func


@pytest_asyncio.fixture()
def wait_for_kelvin_school_class_on_id_broker(
    wait_for_kelvin_object_exists, wait_for_kelvin_object_not_exists, id_broker_session
):
    async def _func(name, school, s_a_name, must_exist=True):
        if must_exist:
            return await wait_for_kelvin_object_exists(
                resource_cls=SchoolClassResource,
                method="get",
                session=id_broker_session,
                name=name,
                school=f"{s_a_name}-{school}",
            )
        else:
            return await wait_for_kelvin_object_not_exists(
                resource_cls=SchoolClassResource,
                method="get",
                session=id_broker_session,
                name=name,
                school=f"{s_a_name}-{school}",
            )

    return _func


@pytest_asyncio.fixture()
def wait_for_kelvin_workgroup_on_id_broker(
    wait_for_kelvin_object_exists, wait_for_kelvin_object_not_exists, id_broker_session
):
    async def _func(name, school, s_a_name, must_exist=True):
        if must_exist:
            return await wait_for_kelvin_object_exists(
                resource_cls=WorkGroupResource,
                method="get",
                session=id_broker_session,
                name=name,
                school=f"{s_a_name}-{school}",
            )
        else:
            return await wait_for_kelvin_object_not_exists(
                resource_cls=WorkGroupResource,
                method="get",
                session=id_broker_session,
                name=name,
                school=f"{s_a_name}-{school}",
            )

    return _func


@pytest_asyncio.fixture()
async def make_kelvin_workgroup_on_id_connector(kelvin_session, id_connector_host_name):
    created_workgroups: List[Tuple[str, str, str]] = []

    async def _func(
        school_name: str, wokrgroup_name: str = None, description: str = None, users: List[str] = None
    ) -> WorkGroup:
        wg_obj = WorkGroup(
            name=wokrgroup_name or fake.user_name(),
            school=school_name,
            description=description or fake.first_name(),
            session=kelvin_session(id_connector_host_name),
            users=users or [],
        )
        await wg_obj.save()
        created_workgroups.append((id_connector_host_name, wg_obj.name, wg_obj.school))
        return wg_obj

    yield _func

    for host, name, school in created_workgroups:
        try:
            _wg = await WorkGroupResource(session=kelvin_session(host)).get(name=name, school=school)
            await _wg.delete()
            print(f"Success deleting work group {name!r} from host {host!r}.")
        except NoObject:
            print(f"No work group {name!r} on {host!r}.")


@pytest_asyncio.fixture()
async def make_kelvin_school_class_on_id_connector(kelvin_session, id_connector_host_name):
    created_school_classes: List[Tuple[str, str, str]] = []

    async def _func(
        school_name: str, school_class_name: str = None, description: str = None, users: List[str] = None
    ) -> SchoolClass:
        sc_obj = SchoolClass(
            name=school_class_name or fake.user_name(),
            school=school_name,
            description=description or fake.first_name(),
            session=kelvin_session(id_connector_host_name),
            users=users or [],
        )
        await sc_obj.save()
        created_school_classes.append((id_connector_host_name, sc_obj.name, sc_obj.school))
        return sc_obj

    yield _func

    for host, name, school in created_school_classes:
        try:
            _sc = await SchoolClassResource(session=kelvin_session(host)).get(name=name, school=school)
            await _sc.delete()
            print(f"Success deleting school class {name!r} from host {host!r}.")
        except NoObject:
            print(f"No school class {name!r} on {host!r}.")


@pytest_asyncio.fixture()
async def make_sender_user_special(
    kelvin_user_object,
    kelvin_session,
    kelvin_session_kwargs,
    docker_hostname,
    id_broker_fqdn,
):
    """
    Fixture factory to create users on the apps host system. They are created
    via the Kelvin-API and removed on both sides when the fixture goes out of scope.
    """
    created_users: List[Dict[str, Any]] = []
    _s_a_name: List[str] = []

    async def _delete_user(username: str, host: str) -> None:
        print(f"Deleting user {username!r} from host {host!r}...")
        # Creating a new session per user deletion, because of a problem in httpx/httpcore/h11.
        # I think it has a problem with the disconnect happening by Kelvin with FastAPI delete():
        # h11._util.LocalProtocolError: Too much data for declared Content-Length
        async with Session(**kelvin_session_kwargs(host)) as session:
            try:
                user = await UserResource(session=session).get(name=username)
                await user.delete()
                print(f"Success deleting user {username!r} from host {host!r}.")
            except NoObject:
                print(f"No user {username!r} on {host!r}.")

    async def _make_sender_user_special(
        roles=("student",), ous=("DEMOSCHOOL",), s_a_name="Traeger1", **kwargs
    ):
        """
        Creates a user on the hosts UCS system via Kelvin-API-Client
        which will be synced to the ID Broker

        :param roles: The new users roles
        :param ous: The new users ous
        :return: The json used to create the user via the API
        """
        nonlocal _s_a_name
        _s_a_name = [s_a_name]
        user_obj: User = kelvin_user_object(
            roles, ous, session=kelvin_session(docker_hostname), **kwargs
        )
        password = user_obj.password
        await user_obj.save()
        user_obj.password = password
        user_obj_as_dict = user_obj.as_dict()
        created_users.append(user_obj_as_dict)
        print("Created new User in source system: {!r}".format(user_obj_as_dict))
        return user_obj_as_dict

    yield _make_sender_user_special

    for user_dict in created_users:
        await _delete_user(user_dict["name"], docker_hostname)
        # if the ID Connector works, there should be nothing to clean up.
        await _delete_user(f"{_s_a_name[0]}-{user_dict['name']}", id_broker_fqdn)


async def get_user_id(name: str = None) -> str:
    ldap = LDAPAccess()
    ldap_filter = f"(uid={name})"
    ldap_objects = await ldap.search(filter_s=ldap_filter, attributes=["entryUUID"])
    return ldap_objects[0].entryUUID.value


async def get_workgroup_id(ou: str, name: str = None) -> str:
    ldap = LDAPAccess()
    ldap_filter = f"(&(ucsschoolRole=workgroup:school:{ou})(cn={ou}-{name}))"
    ldap_objects = await ldap.search(filter_s=ldap_filter, attributes=["entryUUID"])
    return ldap_objects[0].entryUUID.value


async def get_class_id(ou: str, name: str = None) -> str:
    ldap = LDAPAccess()
    ldap_filter = f"(&(ucsschoolRole=school_class:school:{ou})(cn={ou}-{name}))"
    ldap_objects = await ldap.search(filter_s=ldap_filter, attributes=["entryUUID"])
    return ldap_objects[0].entryUUID.value


async def get_ou_id(
    ou: str,
) -> str:
    ldap = LDAPAccess()
    ldap_filter = f"(&(objectClass=ucsschoolOrganizationalUnit)(ou={ou}))"
    ldap_objects = await ldap.search(filter_s=ldap_filter, attributes=["entryUUID"])
    return ldap_objects[0].entryUUID.value
