import re

import pytest
from fastapi import HTTPException

from id_broker_common.kelvin import search_kelvin_obj
from id_broker_common.utils import remove_prefix
from provisioning_plugin import dependencies
from provisioning_plugin.dependencies import member_ids_to_usernames
from provisioning_plugin.utils import get_roles_for_school, groups_of_school, users_of_school
from tests.unittests.utils import (
    DictObject,
    KelvinSessionMock,
    kelvin_school_class,
    kelvin_student,
    kelvin_workgroup,
)
from ucsschool.kelvin.client import InvalidRequest, SchoolClass, User, WorkGroup


class AsyncIteratorMock:
    def __init__(self, seq):
        self.iter = iter(seq)

    def __aiter__(self):
        return self

    async def __anext__(self):
        try:
            return next(self.iter)
        except StopIteration:
            raise StopAsyncIteration


@pytest.mark.asyncio
async def test_search_kelvin_obj(mocker):
    resource_mock = mocker.Mock()
    resource_mock.search = mocker.Mock(return_value=AsyncIteratorMock([1]))
    result = await search_kelvin_obj(resource_mock, name="a", uid=2)
    assert result == 1
    resource_mock.search.assert_called_with(name="a", uid=2)


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "return_value,status_code,exc_type", [([], 404, HTTPException), ([1, 2], 500, HTTPException)]
)
async def test_search_kelvin_obj_http_exception(return_value, status_code, exc_type, mocker):
    resource_mock = mocker.Mock()
    resource_mock.Meta.kelvin_object.__name__ = "Resource"
    resource_mock.search = mocker.Mock(return_value=AsyncIteratorMock(return_value))
    with pytest.raises(exc_type) as exc_info:
        await search_kelvin_obj(resource_mock)
    assert exc_info.type == exc_type
    assert exc_info.value.status_code == status_code


@pytest.mark.asyncio
async def test_search_kelvin_obj_invalid_request(mocker):
    resource_mock = mocker.Mock()
    resource_mock.search = mocker.Mock(side_effect=InvalidRequest(status=422, reason="Unprocessable"))
    with pytest.raises(HTTPException) as exc_info:
        await search_kelvin_obj(resource_mock)
    assert exc_info.type == HTTPException
    assert exc_info.value.status_code == 422
    assert exc_info.value.detail == "Unprocessable"


@pytest.mark.parametrize(
    "prefix,value,expected",
    [
        ("TEST-", "TEST-string", "string"),
        ("OTHER-PRE-FIX", "OTHER-PRE-FIX-FIX-FIX", "-FIX-FIX"),
        ("Test-", "Test-", ""),
        ("", "TEST-string", "TEST-string"),
    ],
)
def test_remove_school_authority_prefix(prefix, value, expected):
    assert remove_prefix(value, prefix) == expected


def test_remove_school_authority_prefix_exc():
    with pytest.raises(ValueError):
        remove_prefix("RANDOM_STR", "PREFIX-")


@pytest.mark.parametrize(
    "roles,expected",
    [
        ([], set()),
        (["teacher:school:DEMOSCHOOL"], {"teacher"}),
        [["student:school:DEMOSCHOOL", "teacher:school:OTHERSCHOOL"], {"student"}],
        (["legal_guardian:school:DEMOSCHOOL"], {"legal_guardian"}),
        (
            ["teacher:school:DEMOSCHOOL", "staff:school:DEMOSCHOOL", "student:school:OTHERSCHOOL"],
            {"teacher", "staff"},
        ),
        (["teacher:other:DEMOSCHOOL"], set()),
        (["teacher:school:DEMOSCHOOL".upper()], set()),
    ],
)
def test_get_roles_for_school(roles, expected):
    assert get_roles_for_school(roles, "DEMOSCHOOL") == expected


@pytest.mark.parametrize("role", ["", ":", "::", ":::", "teacher:school:"])
def test_get_roles_for_schools_exc(role):
    with pytest.raises(ValueError):
        get_roles_for_school([role], "DEMOSCHOOL")


@pytest.mark.asyncio
async def test_member_ids_to_usernames_no_members():
    assert await member_ids_to_usernames(members=set(), school_authority="TEST-AUTHORITY") == []


@pytest.mark.asyncio
async def test_member_ids_to_usernames(monkeypatch):
    class Member(DictObject):
        def __init__(self, uid, source_uid):
            super().__init__()
            self.uid = DictObject(value=uid)
            self.source_uid = source_uid
            self.ucsschoolRole = DictObject(value="foo")

    class ldap_access:
        async def search(self, filter_s, **kwargs):
            source_uid = re.search(r"ucsschoolSourceUID=([^)]*)", filter_s)
            if source_uid:
                return [
                    Member(uid=record_id, source_uid=source_uid.group())
                    for record_id in re.findall(r"ucsschoolRecordUID=([^)]*)", filter_s)
                ]

    members = {f"id-{i}" for i in range(10)}
    monkeypatch.setattr(
        dependencies,
        "ldap_access",
        ldap_access,
    )
    assert sorted(
        await member_ids_to_usernames(members=members, school_authority="TEST-AUTHORITY")
    ) == sorted(members)


@pytest.mark.asyncio
async def test_groups_of_school():
    kelvin_session_mock = KelvinSessionMock()
    kelvin_session_mock.configure_mock(
        **{
            "get.return_value": [kelvin_school_class(), kelvin_workgroup()],
            "__aenter__.return_value": kelvin_session_mock,
        }
    )
    res = await groups_of_school(session=kelvin_session_mock, school="test-school")
    assert isinstance(res[0], SchoolClass)
    assert isinstance(res[1], SchoolClass)
    assert isinstance(res[2], WorkGroup)
    assert isinstance(res[3], WorkGroup)


@pytest.mark.asyncio
async def test_users_of_school():
    kelvin_session_mock = KelvinSessionMock()
    kelvin_session_mock.configure_mock(
        **{
            "get.return_value": [kelvin_student(), kelvin_student()],
            "__aenter__.return_value": kelvin_session_mock,
        }
    )
    res = await users_of_school(session=kelvin_session_mock, school="test-school")
    assert isinstance(res[0], User)
    assert isinstance(res[1], User)
