from unittest.mock import AsyncMock

import pytest

import provisioning_plugin.dependencies
from id_broker_common.kelvin import kelvin_session
from provisioning_plugin.models import SchoolClass
from provisioning_plugin.routes.v1 import school_classes
from ucsschool.kelvin.client.exceptions import InvalidRequest

from ..utils import (
    CLASS_SIZE,
    DEFAULT_SCHOOL_AUTHORITY,
    get_preconfigured_session_mock,
    kelvin_school_class,
    kelvin_student,
)


@pytest.mark.parametrize("method", ["GET", "HEAD"])
def test_school_class_by_id(func_client, monkeypatch, method, broker_school_class):
    exp = SchoolClass.with_prefix(SchoolClass(**broker_school_class.dict()), DEFAULT_SCHOOL_AUTHORITY)
    monkeypatch.setattr(SchoolClass, "from_kelvin_school_class", AsyncMock(return_value=exp))
    _client = func_client(
        custom_overrides={
            provisioning_plugin.dependencies.get_school_class_by_id: kelvin_school_class,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    query = _client.get if method == "GET" else _client.head
    response = query(url=f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/classes/{broker_school_class.id}")
    assert response.status_code == 200
    if method == "GET":
        res = response.json()
        res["members"] = sorted(res["members"])
        assert res == broker_school_class
    else:
        assert response.status_code == 200, response.__dict__
        assert response.content == b""


@pytest.mark.parametrize(
    "method,status_code",
    [
        ("POST", 201),
        ("PUT", 200),
    ],
)
def test_post_school_class(
    func_client, monkeypatch, method, status_code, broker_school_class, async_iterable
):
    monkeypatch.setattr(
        school_classes,
        "member_ids_to_usernames",
        AsyncMock(return_value=[kelvin_student(i) for i in range(CLASS_SIZE)]),
    )
    _client = func_client(
        custom_overrides={
            kelvin_session: get_preconfigured_session_mock(
                {
                    "post.return_value": kelvin_school_class(),
                }
            ),
            provisioning_plugin.dependencies.get_school_class_by_id: kelvin_school_class,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    monkeypatch.setattr(
        school_classes,
        "get_pseudonyms_for_providers",
        async_iterable,
    )
    url = f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/classes"
    query = None
    if method == "POST":
        query = _client.post
    elif method == "PUT":
        query = _client.put
        url += f"/{broker_school_class.id}"
    response = query(
        url,
        json=broker_school_class,
    )
    print(response.__dict__)
    assert response.status_code == status_code
    res = response.json()
    res["members"] = sorted(res["members"])
    assert res == broker_school_class


@pytest.mark.parametrize(
    "method",
    [
        "POST",
        "PUT",
    ],
)
def test_put_post_school_class_missing_member(
    func_client, monkeypatch, method, broker_school_class, async_iterable
):
    class ldap_access:
        async def search(self, filter_s, **kwargs):
            return []

    monkeypatch.setattr(
        provisioning_plugin.dependencies,
        "ldap_access",
        ldap_access,
    )

    _client = func_client(
        custom_overrides={
            kelvin_session: get_preconfigured_session_mock(
                {
                    "post.return_value": kelvin_school_class(),
                }
            ),
            provisioning_plugin.dependencies.get_school_class_by_id: kelvin_school_class,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    monkeypatch.setattr(
        school_classes,
        "get_pseudonyms_for_providers",
        async_iterable,
    )
    url = f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/classes"
    query = None
    change = "created"
    if method == "POST":
        query = _client.post
    elif method == "PUT":
        query = _client.put
        url += f"/{broker_school_class.id}"
        change = "modified"
    response = query(
        url,
        json=broker_school_class,
    )
    assert response.status_code == 422, response.json()
    assert response.json()["detail"] in [
        f"The school class '{broker_school_class['name']}' in school "
        f"'{DEFAULT_SCHOOL_AUTHORITY}-{broker_school_class['school']}' in school authority "
        f"'{DEFAULT_SCHOOL_AUTHORITY}' could not be {change}: "
        f"\"Object '{member}' in school authority '{DEFAULT_SCHOOL_AUTHORITY}' not found.\""
        for member in broker_school_class["members"]
    ]


def test_post_conflict_kelvin_request(func_client, monkeypatch, broker_school_class, async_iterable):
    async def post_mock(*args, **kwargs):
        raise InvalidRequest(msg="test", status=409, reason="test-reason")

    async def reload_mock(*args, **kwargs):
        return

    monkeypatch.setattr(
        school_classes,
        "member_ids_to_usernames",
        AsyncMock(return_value=[kelvin_student(i) for i in range(CLASS_SIZE)]),
    )
    _client = func_client(
        custom_overrides={
            kelvin_session: get_preconfigured_session_mock({"post": post_mock}),
            provisioning_plugin.dependencies.get_school_class_by_id: kelvin_school_class,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    monkeypatch.setattr(
        school_classes.KelvinSchoolClass,
        "reload",
        reload_mock,
    )
    monkeypatch.setattr(
        school_classes,
        "get_pseudonyms_for_providers",
        async_iterable,
    )
    url = f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/classes"
    response = _client.post(
        url,
        json=broker_school_class,
    )
    assert response.status_code == 409, response.json()
    assert response.json() == {
        "detail": {"conflict_id": broker_school_class["id"], "msg": "test-reason"}
    }
