import pytest
from pydantic import ValidationError

import provisioning_plugin.models
from provisioning_plugin.models import School, SchoolClass, SchoolContext, User, WorkGroup
from tests.integration_tests.utils import DEFAULT_SCHOOL_AUTHORITY
from ucsschool.kelvin.client import User as KelvinUser


@pytest.fixture()
def reference_user():
    return User(
        id="m.m",
        first_name="Max",
        last_name="Mustermann",
        user_name="max.m",
        context={
            "School1": SchoolContext(roles={"teacher"}, classes={"1a"}, workgroups={"Wg1"}),
            "School2": SchoolContext(roles={"teacher"}, classes={"2b", "3c"}),
        },
    )


@pytest.fixture()
def reference_school_class():
    return SchoolClass(
        name="1a",
        description="Class 1a",
        school="DEMOSCHOOL",
        members={"u1-id", "u2-id", "u3-id"},
        id="1a-id",
    )


@pytest.fixture()
def reference_workgroup():
    return WorkGroup(
        name="Wg1",
        description="Workgroup 1",
        school="DEMOSCHOOL",
        members={"u1-id", "u2-id", "u3-id"},
        id="Wg1-id",
    )


def test_user_empty_context():
    with pytest.raises(ValidationError, match="The context must not be empty!"):
        User(id="m.m", first_name="Max", last_name="Mustermann", user_name="max.m", context={})


def test_user_non_uniform_roles():
    with pytest.raises(ValidationError, match="All schools must have the same set of roles!"):
        User(
            id="m.m",
            first_name="Max",
            last_name="Mustermann",
            user_name="max.m",
            context={
                "School1": SchoolContext(roles={"teacher"}),
                "School2": SchoolContext(roles={"student"}),
            },
        )


@pytest.mark.asyncio
async def test_user_from_kelvin_user(reference_user):
    prefix = f"{DEFAULT_SCHOOL_AUTHORITY}-"
    school1 = f"{prefix}School1"
    school2 = f"{prefix}School2"
    assert (
        await User.from_kelvin_user(
            KelvinUser(
                record_uid="m.m",
                firstname="Max",
                lastname="Mustermann",
                name=f"{prefix}max.m",
                schools=[school1, school2],
                school_classes={school1: ["1a"], school2: ["2b", "3c"]},
                workgroups={school1: ["Wg1"], school2: []},
                ucsschool_roles=[f"teacher:school:{school1}", f"teacher:school:{school2}"],
                roles=["teacher"],
            ),
            DEFAULT_SCHOOL_AUTHORITY,
        )
        == reference_user
    )


def test_user_prop_ucsschool_roles(reference_user):
    assert (
        reference_user.ucsschool_roles.sort()
        == ["teacher:school:School1", "teacher:school:School2"].sort()
    )


def test_user_prop_school_classes(reference_user):
    school_classes = reference_user.school_classes
    assert list(school_classes.keys()).sort() == ["School1", "School2"].sort()
    assert school_classes["School1"] == ["1a"]
    assert school_classes["School2"].sort() == ["2b", "3c"].sort()


def test_user_prop_workgroups(reference_user):
    workgroups = reference_user.workgroups
    assert list(workgroups.keys()).sort() == ["School1", "School2"].sort()
    assert workgroups["School1"] == ["Wg1"]
    assert workgroups["School2"].sort() == [].sort()


def test_user_prop_school(reference_user):
    assert reference_user.school == "School1"


def test_school_from_kelvin_school():
    assert School.from_kelvin_school(
        dict(name="TEST", display_name="Display Test", udm_properties={"ucsschoolRecordUID": "TEST_ID"})
    ) == School(name="TEST", display_name="Display Test", id="TEST_ID")


def test_school_without_prefix():
    assert School.without_prefix(
        School(name="TEST-S1", display_name="S1", id="TEST_ID"), "TEST"
    ) == School(name="S1", display_name="S1", id="TEST_ID")


def test_school_with_prefix():
    assert School.with_prefix(School(name="S1", display_name="S1", id="TEST_ID"), "TEST") == School(
        name="TEST-S1", display_name="S1", id="TEST_ID"
    )


@pytest.mark.asyncio
async def test_school_class_from_kelvin_school_class(mocker, monkeypatch, reference_school_class):
    async def retrieve_users_in_parallel_mock(user_names, *, parallelism, session):
        return [mocker.Mock(record_uid=f"{user_name}-id") for user_name in user_names]

    monkeypatch.setattr(
        provisioning_plugin.models, "retrieve_users_in_parallel", retrieve_users_in_parallel_mock
    )
    mocker.patch("provisioning_plugin.models.kelvin_access_parallelism_setting", return_value=4)
    assert (
        await SchoolClass.from_kelvin_school_class(
            {
                "name": "1a",
                "school": "DEMOSCHOOL",
                "description": "Class 1a",
                "users": ["u1", "u2", "u3"],
                "udm_properties": {"ucsschoolRecordUID": "1a-id"},
            },
            session=mocker.Mock(),
        )
        == reference_school_class
    )


@pytest.mark.asyncio
async def test_workgroup_from_kelvin_workgroup(mocker, monkeypatch, reference_workgroup):
    async def retrieve_users_in_parallel_mock(user_names, *, parallelism, session):
        return [mocker.Mock(record_uid=f"{user_name}-id") for user_name in user_names]

    monkeypatch.setattr(
        provisioning_plugin.models, "retrieve_users_in_parallel", retrieve_users_in_parallel_mock
    )
    mocker.patch("provisioning_plugin.models.kelvin_access_parallelism_setting", return_value=4)
    assert (
        await WorkGroup.from_kelvin_workgroup(
            {
                "name": "Wg1",
                "school": "DEMOSCHOOL",
                "description": "Workgroup 1",
                "users": ["u1", "u2", "u3"],
                "udm_properties": {"ucsschoolRecordUID": "Wg1-id"},
            },
            session=mocker.Mock(),
        )
        == reference_workgroup
    )


@pytest.mark.asyncio
async def test_workgroup_from_kelvin_workgroup_no_fill_members(mocker, reference_workgroup):
    wg = reference_workgroup
    wg.members = set()
    assert (
        await WorkGroup.from_kelvin_workgroup(
            {
                "name": "Wg1",
                "school": "DEMOSCHOOL",
                "description": "Workgroup 1",
                "users": ["u1", "u2", "u3"],
                "udm_properties": {"ucsschoolRecordUID": "Wg1-id"},
            },
            session=mocker.Mock(),
            fill_members=False,
        )
        == wg
    )


def test_school_class_without_prefix(reference_school_class):
    assert (
        SchoolClass.without_prefix(
            SchoolClass(
                name="1a",
                description="Class 1a",
                school="TEST-DEMOSCHOOL",
                members={"u1-id", "u2-id", "u3-id"},
                id="1a-id",
            ),
            "TEST",
        )
        == reference_school_class
    )


def test_school_class_with_prefix(reference_school_class):
    assert SchoolClass.with_prefix(reference_school_class, "TEST") == SchoolClass(
        name="1a",
        description="Class 1a",
        school="TEST-DEMOSCHOOL",
        members={"u1-id", "u2-id", "u3-id"},
        id="1a-id",
    )


def test_workgroup_without_prefix(reference_workgroup):
    assert (
        WorkGroup.without_prefix(
            WorkGroup(
                name="Wg1",
                description="Workgroup 1",
                school="TEST-DEMOSCHOOL",
                members={"u1-id", "u2-id", "u3-id"},
                id="Wg1-id",
            ),
            "TEST",
        )
        == reference_workgroup
    )


def test_workgroup_with_prefix(reference_workgroup):
    assert WorkGroup.with_prefix(reference_workgroup, "TEST") == WorkGroup(
        name="Wg1",
        description="Workgroup 1",
        school="TEST-DEMOSCHOOL",
        members={"u1-id", "u2-id", "u3-id"},
        id="Wg1-id",
    )
