from typing import AsyncIterable, Iterable, Set, Tuple

from id_broker_common.pseudonyms import generate_pseudonym
from id_broker_common.pseudonyms_udm_rest import (  # noqa: F401
    get_admin_connection_kwargs,
    get_service_provider_secret,
    get_settings_data_content,
)
from id_broker_common.utils import ldap_settings  # noqa: F401
from ucsschool.apis.utils import get_logger
from ucsschool.kelvin.client import (
    NoObject as KelvinNoObject,
    SchoolClassResource,
    Session,
    UserResource,
    WorkGroupResource,
)
from udm_rest_client import UDM, NoObject

logger = get_logger()


def get_roles_for_school(roles: Iterable[str], school: str) -> Set[str]:
    """
    Takes a list of ucsschool_roles and returns a list of user roles for the given school.

    >>> get_roles_for_school(["teacher:school:School1", "student:school:School2"], "School1")
    {'teacher'}

    :param roles: The list of ucsschool_roles to filter
    :param school: The school to filter the roles for
    :return: The list of user roles for the given school
    :raises ValueError: If any of the role strings is malformed
    """
    filtered_roles = set()
    for role in roles:
        role_parts = role.split(":")
        if len(role_parts) != 3 or not all(role_parts):
            raise ValueError(f"The role {role} is malformed!")
        if (
            role_parts[1] == "school"
            and role_parts[2] == school
            and role_parts[0] in ("student", "teacher", "legal_guardian", "staff")
        ):
            filtered_roles.add(role_parts[0])
    return filtered_roles


async def get_pseudonyms_for_providers(
    entry_uuid: str, school_authority: str
) -> AsyncIterable[Tuple[str, str]]:
    """
    Generates pseudonyms for all service providers, which are registered
    and sets the corresponding UDM property.
    """
    ldap_base = ldap_settings().ldap_base
    dn = f"cn=IDBrokerServiceProviderMappings,cn=IDBrokerSettings,cn=univention,{ldap_base}"
    provider_to_udm_prop = await get_settings_data_content(dn)
    for sp, udm_prop in provider_to_udm_prop.items():
        # todo could be cached for optimization
        service_provider_salt = await get_service_provider_secret(sp)
        yield udm_prop, generate_pseudonym(service_provider_salt, entry_uuid, school_authority)


async def users_of_school(school: str, session: Session):
    return [obj async for obj in UserResource(session=session).search(school=school)]


async def groups_of_school(school: str, session: Session):
    school_classes = [obj async for obj in SchoolClassResource(session=session).search(school=school)]
    work_groups = [obj async for obj in WorkGroupResource(session=session).search(school=school)]
    return school_classes + work_groups


async def remove_school_objects(school: str, session: Session):
    users = await users_of_school(school=school, session=session)
    for user in users:
        try:
            await user.delete()
        except KelvinNoObject:
            logger.warning(f"User {user.name} already deleted. Skip it.")
    groups = await groups_of_school(school=school, session=session)
    for group in groups:
        try:
            await group.delete()
        except KelvinNoObject:
            logger.warning(f"Group {group.name} already deleted. Skip it.")
    ldap_setting = ldap_settings()
    ldap_base = ldap_setting.ldap_base
    # On the ID Broker we don't change those names, so we don't have to get our hands
    # dirty and parse env or ucr values.
    dns = [
        f"cn=OU{school}-Member-Verwaltungsnetz,cn=ucsschool,cn=groups,{ldap_base}",
        f"cn=OU{school}-Member-Edukativnetz,cn=ucsschool,cn=groups,{ldap_base}",
        f"cn=OU{school}-Klassenarbeit,cn=ucsschool,cn=groups,{ldap_base}",
        f"cn=OU{school}-DC-Verwaltungsnetz,cn=ucsschool,cn=groups,{ldap_base}",
        f"cn=OU{school}-DC-Edukativnetz,cn=ucsschool,cn=groups,{ldap_base}",
        f"cn=admins-{school},cn=ouadmins,cn=groups,{ldap_base}",
    ]
    async with UDM(**get_admin_connection_kwargs()) as udm:
        group_mod = udm.get("groups/group")
        for dn in dns:
            try:
                group = await group_mod.get(dn)
                await group.delete()
            except NoObject:
                logger.warning(f"Group {group.name} already deleted. Skip it.")
        ou_mod = udm.get("container/ou")
        try:
            ou = await ou_mod.get(f"ou={school},{ldap_base}")
            await ou.delete()
        except NoObject:
            logger.warning(f"Ou {ou.name} already deleted. Skip it.")
