from __future__ import annotations

import asyncio
from typing import Any, Dict, List, Set

from fastapi import HTTPException
from ldap3.utils.conv import escape_filter_chars
from pydantic import BaseModel, PrivateAttr, constr, validator
from starlette import status

from id_broker_common.kelvin import kelvin_access_parallelism_setting, retrieve_users_in_parallel
from id_broker_common.ldap_access import ldap_access
from id_broker_common.utils import remove_prefix
from provisioning_plugin.utils import get_roles_for_school, logger
from ucsschool.kelvin.client import Session, User as KelvinUser

from .settings import PLUGIN_SETTINGS_FILE

NonEmptyStr = constr(min_length=1)
NoStarStr = constr(regex=r"^[^*]+$")


class SchoolContext(BaseModel):
    classes: Set[NonEmptyStr] = set()
    workgroups: Set[NonEmptyStr] = set()
    roles: Set[NonEmptyStr] = set()


class User(BaseModel):
    id: NonEmptyStr
    first_name: NonEmptyStr
    last_name: NonEmptyStr
    user_name: NonEmptyStr
    context: Dict[NonEmptyStr, SchoolContext]
    legal_guardians: List[NonEmptyStr] = []
    legal_wards: List[NonEmptyStr] = []
    _prefix: str = PrivateAttr(default="")

    class Config:
        schema_extra = {
            "example": {
                "id": "d8aa843b-9628-42b8-9637-d198c745efa5",
                "user_name": "m.mustermann",
                "first_name": "Max",
                "last_name": "Mustermann",
                "legal_guardians": [],
                "legal_wards": [],
                "context": {
                    "School1": {"classes": ["1a"], "workgroups": ["WG1"], "roles": ["teacher", "staff"]}
                },
            }
        }

    @validator("context")
    def context_not_empty(cls, value: Dict[str, SchoolContext]):
        if not value:
            raise ValueError("The context must not be empty!")
        return value

    @validator("context")
    def uniform_ucsschool_role_in_all_schools(cls, value: Dict[str, SchoolContext]):
        """
        Validates that all SchoolContexts have the same set of roles.

        Users with different roles in different schools are not yet supported in the Kelvin API.
        """
        role_sets = [school_context.roles for school_context in value.values()]
        if any(roles != role_sets[0] for roles in role_sets):
            raise ValueError("All schools must have the same set of roles!")
        return value

    @classmethod
    async def _get_ldap_user_attribute(
        cls, school_authority: str, search_field: str, value: str, result_field: str
    ) -> str:
        ldap = ldap_access()
        filter_s = (
            f"(&(objectClass=organizationalPerson)"
            f"(ucsschoolSourceUID={escape_filter_chars(school_authority)})"
            f"({search_field}={escape_filter_chars(value)}))"
        )
        results = await ldap.search(filter_s, attributes=[result_field], use_master=True)
        if len(results) == 1:
            return results[0][result_field].value
        elif not results:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail=f"User {value!r} in school authority {school_authority!r} not found.",
            )
        else:
            logger.error("More than 1 result when searching LDAP with filter %r: %r.", filter_s, results)
            raise HTTPException(
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
                detail=f"Error retrieving user {value!r} in school authority {school_authority!r}.",
            )

    @classmethod
    async def names_to_uuids(cls, school_authority: str, names: List[str]) -> List[str]:
        if not names:
            return []
        tasks = (
            cls._get_ldap_user_attribute(
                school_authority,
                "uid",
                name,
                "ucsschoolRecordUID",
            )
            for name in names
        )
        return await asyncio.gather(*tasks)

    @classmethod
    async def uuids_to_names(cls, school_authority: str, uuids: List[str]) -> List[str]:
        if not uuids:
            return []
        tasks = (
            cls._get_ldap_user_attribute(
                school_authority,
                "ucsschoolRecordUID",
                uuid,
                "uid",
            )
            for uuid in uuids
        )
        return await asyncio.gather(*tasks)

    @classmethod
    async def from_kelvin_user(cls, user: KelvinUser, school_authority: NonEmptyStr) -> User:
        """
        Takes a KelvinUser and returns a User from it.

        :param user: The dictionary to create the User from
        :param school_authority: Identifier of the school authority this object originates from.
        :return: The user object created from the provided data
        :raises ValueError: If any of the school roles is malformed
        :raises ValidationError: If the user is malformed
        """

        prefix = f"{school_authority}-"

        user_data: Dict[str, Any] = {
            # id unique in concert with the source_uid generated from the school_authority in the resource URL
            "id": user.record_uid,
            "first_name": user.firstname,
            "last_name": user.lastname,
            "user_name": remove_prefix(user.name, prefix),
        }
        context = {
            remove_prefix(school, prefix): SchoolContext(
                classes=user.school_classes.get(school, []),
                workgroups=user.workgroups.get(school, []),
                roles=get_roles_for_school(user.ucsschool_roles, school),
            )
            for school in user.schools
        }
        for param, role in (("legal_guardians", "student"), ("legal_wards", "legal_guardian")):
            user_names = getattr(user, param, [])
            user_uuids = await cls.names_to_uuids(school_authority, user_names)
            user_data[param] = user_uuids
        user_data["context"] = context
        return cls(**user_data)

    def set_prefix(self, prefix):
        self._prefix = prefix

    @property
    def ucsschool_roles(self) -> List[str]:
        """
        The ucsschool_roles generated from the school_context roles.
        """
        return [
            f"{role}:school:{self._prefix}{school}"
            for school, school_context in self.context.items()
            for role in school_context.roles
        ]

    @property
    def school_classes(self) -> Dict[str, List[str]]:
        """
        The school_classes in the form that is present in the Kelvin API.
        """
        return {
            f"{self._prefix}{school}": list(school_context.classes)
            for school, school_context in self.context.items()
        }

    @property
    def workgroups(self) -> Dict[str, List[str]]:
        """
        The workgroups in the form that is present in the Kelvin API.
        """
        return {
            f"{self._prefix}{school}": list(school_context.workgroups)
            for school, school_context in self.context.items()
        }

    @property
    def roles(self) -> List[str]:
        """
        The roles of the primary school in the form that is present in the Kelvin API.
        """
        return list(list(self.context.values())[0].roles)

    @property
    def name(self) -> str:
        """
        The name in the form that is present in the Kelvin API.
        """
        return f"{self._prefix}{self.user_name}"

    @property
    def school(self) -> str:
        """
        The users "main school".
        This is just the first school present in the context field. It is needed since Kelvin needs a specific
        main school.
        """
        # The primary school should be the 1st in the context dict (which is always ordered in cpython3),
        # as that is what the ID connector did. This way the user will hopefully have the same primary
        # school at the school authority and at the ID Broker. That is not strictly necessary, as the
        # Self-Disclosure API does not expose that, but it'll be less confusing, when comparing users.
        primary_school = list(self.context.keys())[0]
        return f"{self._prefix}{primary_school}"

    @property
    def schools(self) -> List[str]:
        """
        A list of all schools of the user.
        """
        return sorted(f"{self._prefix}{s}" for s in self.context.keys())


class School(BaseModel):
    id: NonEmptyStr
    name: NonEmptyStr
    display_name: NonEmptyStr

    class Config:
        schema_extra = {
            "example": {
                "id": "6762f079-bb7d-4a92-9bce-1c3b2d2efad3",
                "name": "DEMOSCHOOL",
                "display_name": "Demo school",
            }
        }

    @classmethod
    def from_kelvin_school(cls, school: Dict[str, Any]) -> School:
        """
        Takes a dictionary of the form that is generated by KelvinSchool.as_dict
        and returns a school from it.

        :param school: The dictionary to create the school from
        :return: The school object created from the provided data
        :raises ValidationError: If the school is malformed
        """
        school_data = {
            "id": school.get("udm_properties", {}).get("ucsschoolRecordUID", ""),
            "name": school.get("name", ""),
            "display_name": school.get("display_name", ""),
        }
        return cls(**school_data)

    @classmethod
    def without_prefix(cls, school: School, school_authority: str) -> School:
        """
        Creates a school from the provided school object and removes the school authority prefix.
        """
        prefix = f"{school_authority}-"
        school_data = school.dict()
        school_data["name"] = remove_prefix(school_data["name"], prefix)
        return cls(**school_data)

    @classmethod
    def with_prefix(cls, school: School, school_authority: str) -> School:
        """
        Creates a school from the provided school object and adds the school authority prefix.
        """
        prefix = f"{school_authority}-"
        school_data = school.dict()
        school_data["name"] = f"{prefix}{school_data['name']}"
        return cls(**school_data)


class SchoolClass(BaseModel):
    id: NonEmptyStr
    name: NonEmptyStr
    description: str
    school: NonEmptyStr
    members: Set[NonEmptyStr]

    class Config:
        schema_extra = {
            "example": {
                "id": "503f1278-9cda-49f6-a749-3af022a7d32b",
                "name": "1a",
                "description": "Class 1a of Demoschool",
                "school": "Demoschool",
                "members": [
                    "d8aa843b-9628-42b8-9637-d198c745efa5",
                    "0e49f8b0-9c98-4716-9fe1-23d4466d9af8",
                    "86f753e0-50b3-44cb-807e-2c2556b0e6ca",
                ],
            }
        }

    @classmethod
    async def from_kelvin_school_class(
        cls, school_class: Dict[str, Any], *, session: Session, fill_members=True
    ) -> SchoolClass:
        """
        Takes a dictionary of the form that is generated by KelvinSchoolClass.as_dict
        and returns a SchoolClass.

        :param school_class: The data to construct the school class from
        :param session: A Kelvin client Session object
        :param bool fill_members: whether the `members` attribute should be filled with pseudonyms
        :return: The constructed school class
        :raises NoObject: If any of the member users cannot be found
        :raises ValidationError: If no valid SchoolClass can be constructed
        """
        if fill_members and school_class["users"]:
            users: List[KelvinUser] = await retrieve_users_in_parallel(
                school_class["users"],
                parallelism=kelvin_access_parallelism_setting(PLUGIN_SETTINGS_FILE),
                session=session,
            )
            members = [user.record_uid for user in users]
        else:
            members = []

        school_class_data = {
            "id": school_class.get("udm_properties", {}).get("ucsschoolRecordUID", ""),
            "name": school_class.get("name", ""),
            "school": school_class.get("school", ""),
            "description": school_class.get("description", "") or "",
            "members": members,
        }
        return cls(**school_class_data)

    @classmethod
    def without_prefix(cls, school_class: SchoolClass, school_authority: str) -> SchoolClass:
        """
        Creates a school class from the provided school class object and removes the school authority prefix.
        """
        prefix = f"{school_authority}-"
        school_class_data = school_class.dict()
        school_class_data["school"] = remove_prefix(school_class_data["school"], prefix)
        return cls(**school_class_data)

    @classmethod
    def with_prefix(cls, school_class: SchoolClass, school_authority: str) -> SchoolClass:
        """
        Creates a school class from the provided school class object and adds the school authority prefix.
        """
        prefix = f"{school_authority}-"
        school_class_data = school_class.dict()

        school_class_data["school"] = f"{prefix}{school_class_data['school']}"
        return cls(**school_class_data)


class WorkGroup(BaseModel):
    id: NonEmptyStr
    name: NonEmptyStr
    description: str
    school: NonEmptyStr
    members: Set[NonEmptyStr]

    class Config:
        schema_extra = {
            "example": {
                "id": "503f1278-9cda-49f6-a749-3af022a7d32b",
                "name": "W1",
                "description": "Workgroup1 of Demoschool",
                "school": "Demoschool",
                "members": [
                    "d8aa843b-9628-42b8-9637-d198c745efa5",
                    "0e49f8b0-9c98-4716-9fe1-23d4466d9af8",
                    "86f753e0-50b3-44cb-807e-2c2556b0e6ca",
                ],
            }
        }

    @classmethod
    async def from_kelvin_workgroup(
        cls, workgroup: Dict[str, Any], *, session: Session, fill_members=True
    ) -> WorkGroup:
        """
        Takes a dictionary of the form that is generated by KelvinWorkGroup.as_dict
        and returns a WorkGroup.

        :param workgroup: The data to construct the school class from
        :param session: A Kelvin client Session object
        :param bool fill_members: whether the `members` attribute should be filled with pseudonyms
        :return: The constructed school class
        :raises NoObject: If any of the member users cannot be found
        :raises ValidationError: If no valid WorkGroup can be constructed
        """
        if fill_members and workgroup["users"]:
            users: List[KelvinUser] = await retrieve_users_in_parallel(
                workgroup["users"],
                parallelism=kelvin_access_parallelism_setting(PLUGIN_SETTINGS_FILE),
                session=session,
            )
            members = [user.record_uid for user in users]
        else:
            members = []

        workgroup_data = {
            "id": workgroup.get("udm_properties", {}).get("ucsschoolRecordUID", ""),
            "name": workgroup.get("name", ""),
            "school": workgroup.get("school", ""),
            "description": workgroup.get("description", "") or "",
            "members": members,
        }
        return cls(**workgroup_data)

    @classmethod
    def without_prefix(cls, workgroup: WorkGroup, school_authority: str) -> WorkGroup:
        """
        Creates a workgroup from the provided workgroup object and removes the school authority prefix.
        """
        prefix = f"{school_authority}-"
        workgroup_data = workgroup.dict()
        workgroup_data["school"] = remove_prefix(workgroup_data["school"], prefix)
        return cls(**workgroup_data)

    @classmethod
    def with_prefix(cls, workgroup: WorkGroup, school_authority: str) -> WorkGroup:
        """
        Creates a workgroup from the provided workgroup object and adds the school authority prefix.
        """
        prefix = f"{school_authority}-"
        workgroup_data = workgroup.dict()

        workgroup_data["school"] = f"{prefix}{workgroup_data['school']}"
        return cls(**workgroup_data)
