import asyncio
from typing import Dict, List, Set, cast

from fastapi import Depends, HTTPException, Path
from ldap3.utils.conv import escape_filter_chars
from starlette import status

from id_broker_common.kelvin import get_kelvin_obj, kelvin_session, search_kelvin_obj
from id_broker_common.ldap_access import ldap_access
from id_broker_common.utils import remove_prefix
from provisioning_plugin.models import NonEmptyStr, NoStarStr
from ucsschool.apis.models import AuthenticatedUser
from ucsschool.apis.opa import OPAClient, opa_instance
from ucsschool.apis.utils import auth_manager, get_logger
from ucsschool.kelvin.client import (
    School as KelvinSchool,
    SchoolClass as KelvinSchoolClass,
    SchoolClassResource,
    SchoolResource,
    Session,
    User as KelvinUser,
    UserResource,
    WorkGroup as KelvinWorkGroup,
    WorkGroupResource,
)

logger = get_logger()


async def check_for_authority_admin(
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    policy_instance: OPAClient = Depends(opa_instance),
    user: AuthenticatedUser = Depends(auth_manager("provisioning", AuthenticatedUser)),
) -> bool:
    """
    Checks if the user is a school authority admin via OPA.
    """
    return await policy_instance.check_policy_true_or_raise(
        "provisioning_plugin/school_authority_admin",
        {"username": user.username, "school_authority": school_authority},
    )


async def _get_obj_attr_by_id(
    obj_id: NoStarStr,
    school_authority: NonEmptyStr,
    object_class: NonEmptyStr,
    name_attr: NonEmptyStr,
) -> Dict[str, str]:
    """Get 'name' and 'school' from `name_attr` and `ucsschoolRole` of LDAP object."""
    ldap = ldap_access()
    filter_s = (
        f"(&(objectClass={object_class})"
        f"(ucsschoolSourceUID={escape_filter_chars(school_authority)})"
        f"(ucsschoolRecordUID={escape_filter_chars(obj_id)}))"
    )
    results = await ldap.search(filter_s, attributes=[name_attr, "ucsschoolRole"], use_master=True)
    if len(results) == 1:
        result = results[0]
        name = result[name_attr].value
        if isinstance(result.ucsschoolRole.value, str):
            school_value = result.ucsschoolRole.value
        else:
            school_value = [s for s in result.ucsschoolRole.value if ":school:" in s][0]
        school = school_value.split(":")[-1]
    elif len(results) > 1:
        logger.error("More than 1 result when searching LDAP with filter %r: %r.", filter_s, results)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error retrieving object {obj_id!r} in school authority {school_authority!r}.",
        )
    else:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail=f"Object {obj_id!r} in school authority {school_authority!r} not found.",
        )
    res = {"name": name, "school": school}
    logger.debug("obj_id: %r (%r) => %r", obj_id, school_authority, res)
    return res


async def get_school_class_by_id(
    class_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    session: Session = Depends(kelvin_session),
) -> KelvinSchoolClass:
    attrs = await _get_obj_attr_by_id(class_id, school_authority, "ucsschoolGroup", "cn")
    school = attrs["school"]
    name = remove_prefix(attrs["name"], f"{school}-")
    return cast(
        KelvinSchoolClass,
        await get_kelvin_obj(
            SchoolClassResource(session=session),
            name=name,
            school=school,
        ),
    )


async def get_workgroup_by_id(
    workgroup_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    session: Session = Depends(kelvin_session),
) -> KelvinWorkGroup:
    attrs = await _get_obj_attr_by_id(workgroup_id, school_authority, "ucsschoolGroup", "cn")
    school = attrs["school"]
    name = remove_prefix(attrs["name"], f"{school}-")
    return cast(
        KelvinWorkGroup,
        await get_kelvin_obj(
            WorkGroupResource(session=session),
            name=name,
            school=school,
        ),
    )


async def find_school_name_by_id_in_primary_ldap(
    school_id: NonEmptyStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
) -> str:
    """

    :param str school_id: Unique ID of LDAP object on school authority side
    :param str school_authority: Identifier of the school authority this object originates from
    :return: name of OU
    """
    res = await _get_obj_attr_by_id(school_id, school_authority, "ucsschoolOrganizationalUnit", "ou")
    return res["name"]


async def get_school_by_id(
    school_id: NonEmptyStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    session: Session = Depends(kelvin_session),
) -> KelvinSchool:
    school_name = await find_school_name_by_id_in_primary_ldap(school_id, school_authority)
    return cast(
        KelvinSchool,
        await get_kelvin_obj(
            SchoolResource(session=session),
            name=school_name,
        ),
    )


async def get_user_by_id(
    user_id: NonEmptyStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    session: Session = Depends(kelvin_session),
) -> KelvinUser:
    return cast(
        KelvinUser,
        await search_kelvin_obj(
            UserResource(session=session),
            record_uid=user_id,
            source_uid=school_authority,
        ),
    )


async def member_ids_to_usernames(members: Set[str], school_authority: str) -> List[str]:
    if not members:
        return []

    tasks = (
        _get_obj_attr_by_id(
            object_class="organizationalPerson",
            school_authority=school_authority,
            obj_id=record_uid,
            name_attr="uid",
        )
        for record_uid in members
    )
    ldap_members = await asyncio.gather(*tasks)
    return [member["name"] for member in ldap_members]
