import logging
import random
from typing import Dict, List

import requests
import requests.packages.urllib3
from locust import events, task
from requests.cookies import RequestsCookieJar
from urllib3.exceptions import InsecureRequestWarning

from performance_tests.common import TestData, get_schools, get_users_with_passwords
from performance_tests.keycloak.common import SCHOOL_AUTHORITIES_TRAEGER_MAPPING, TestApp, TestAppError

requests.packages.urllib3.disable_warnings(category=InsecureRequestWarning)

school_authorities: List[str] = []  # ["Traeger1", "Traeger"]
id_broker_school_users: Dict[str, Dict[str, Dict[str, str]]] = {}  # {sch_auth: {ou: {uid: uuid, ...}}}
id_broker_schools: Dict[str, Dict[str, str]] = {}  # {sch_auth: {ou: uuid, ...}}
NUM_SCHOOLS_PER_SCHOOL_AUTHORITY = 5  # loading all OUs from test DB takes to long


@events.init.add_listener
def on_init(environment, **kwargs):
    data = TestData()
    school_authorities.extend(data.school_authorities)
    id_broker_schools.update(get_schools(data))
    id_broker_school_users.update(get_users_with_passwords(data, NUM_SCHOOLS_PER_SCHOOL_AUTHORITY))


class TestAppClient(TestApp):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.url = f"https://{self.host}/{self.endpoint}"

    @task
    def access_token_get(self) -> None:
        traeger = random.choice(list(id_broker_school_users))
        school = random.choice(list(id_broker_school_users[traeger]))
        user = random.choice(list(id_broker_school_users[traeger][school]))
        password = id_broker_school_users[traeger][school][user]
        self.client.cookies = RequestsCookieJar()
        idp_hint = SCHOOL_AUTHORITIES_TRAEGER_MAPPING[traeger]["idp_hint"]
        logging.debug(f"testing with user {user}")

        # 1. univention-test-app/login
        params = {"kc_idp_hint": idp_hint, "pkce": "y"}
        with self.client.get(
            self.url,
            verify=False,
            params=params,
            name="univention-test-app/login",
            catch_response=True,
            allow_redirects=False,
        ) as response:
            try:
                redirect_url = response.headers["Location"]
            except KeyError as exc:
                response.failure(exc)
                return

        # 2. login.kc1.broker.local/auth/realms/ID-Broker/protocol/openid-connect/auth
        #    login.kc1.broker.local/auth/realms/ID-Broker/broker/traeger2/login
        name = "/auth/realms/ID-Broker/auth/login"
        with self.client.get(redirect_url, verify=False, catch_response=True, name=name) as response:
            try:
                redirect_url = self.extract_auth_redirect_url(response.text)
                saml_relay_state = self.extract_saml_relay_state(response.text)
                saml_request = self.extract_saml_request(response.text)
            except TestAppError as exc:
                response.failure(exc)
                return

        # 3. simplesamlphp/saml2/idp/SSOService.php
        name = "simplesamlphp/saml2/idp/SSOService.php"
        data = {
            "SAMLRequest": saml_request,
            "RelayState": saml_relay_state,
        }
        with self.client.post(
            redirect_url, verify=False, data=data, name=name, catch_response=True
        ) as response:
            try:
                saml_auth_state = self.extract_saml_auth_state(response.text)
                response_url = response.url
            except TestAppError as exc:
                response.failure(exc)
                return

        # 4. simplesamlphp/saml2/idp/SSOService.php
        name = "simplesamlphp/saml2/idp/SSOService.php"
        data = {"AuthState": saml_auth_state, "password": password, "username": user}
        with self.client.post(
            response_url, verify=False, data=data, name=name, catch_response=True
        ) as response:
            try:
                saml_response = self.extract_saml_response(response.text)
                saml_relay_state = self.extract_saml_relay_state(response.text)
                kc_redirect_url = self.extract_auth_redirect_url(response.text)
            except TestAppError as exc:
                response.failure(exc)
                return

        # 5. auth/realms/ID-Broker/broker/traeger1/endpoint
        data = {"SAMLResponse": saml_response, "RelayState": saml_relay_state}
        name = "/auth/realms/ID-Broker/broker/{traeger}/endpoint"
        with self.client.post(
            kc_redirect_url,
            name=name,
            data=data,
            verify=False,
            catch_response=True,
            allow_redirects=False,
        ) as response:
            try:
                location = response.headers["Location"]
            except (TestAppError, KeyError) as exc:
                response.failure(exc)
                return

        # 6. univention-test-app/authorize
        name = "univention-test-app/authorize"
        with self.client.get(location, name=name, verify=False, catch_response=True) as response:
            try:
                self.extrace_access_token(response.text)
            except TestAppError as exc:
                logging.exception(exc)
                logging.error(response.text)
                response.failure(exc)
                return
