import datetime
import json
import logging
import os
import random
import re
import time
from json import JSONDecodeError
from pathlib import Path
from typing import Any, Dict, Iterable, List, Set, Tuple, Union, cast

import jwt
import requests.packages.urllib3
from diskcache import Index
from gevent.lock import BoundedSemaphore
from locust import HttpUser
from locust.clients import HttpSession as LocustClient, ResponseContextManager as LucustCMResponse
from pydantic import BaseModel
from urllib3.exceptions import InsecureRequestWarning

PROVISIONING_SCHOOL_URL = "/ucsschool/apis/provisioning/v1/{school_authority}/schools/{id}"
PROVISIONING_CLASS_URL = "/ucsschool/apis/provisioning/v1/{school_authority}/classes/{id}"
PROVISIONING_USER_URL = "/ucsschool/apis/provisioning/v1/{school_authority}/users/{id}"
UCSSCHOOL_APIS_TOKEN_URL = "/ucsschool/apis/auth/token"
BASE_DIR = Path("/var/lib/id-broker-performance-tests")
DEFAULT_DATA_DIR = BASE_DIR / "test-data-db"
USED_SCHOOLS_LOG = BASE_DIR / "used_schools.txt"
ENV_DATA_DIR = "ID_BROKER_TEST_DATA"

try:
    PROVISIONING_USER_TRAEGER1_SECRET = os.environ["UCS_ENV_PROVISIONING_USER_TRAEGER1_SECRET"]
    PROVISIONING_USER_TRAEGER2_SECRET = os.environ["UCS_ENV_PROVISIONING_USER_TRAEGER2_SECRET"]
except KeyError as err:
    raise ValueError(f"Missing environment variable {err}!")


requests.packages.urllib3.disable_warnings(category=InsecureRequestWarning)
JSON_FILENAME_PATTERN = re.compile(r"(?P<sa>.+?)-(?P<ou>.+)_(?P<type>.+).json")


class SchoolAuthorityData(BaseModel):
    name: str  # "Traeger1"
    prov_api_admins: Dict[str, str]  # {"provisioning-Traeger1": "secret"}
    schools: Dict[str, str]  # {"TEST-01": "abc-123", "TEST-02": "456-efg"}
    number_of_schools: int


class GroupData(BaseModel):
    id: str
    name: str  # "1a"
    memberUid: List[str]  # ["alice", "bob"]
    pseudonyms: Dict[str, str]  # {"idBrokerPseudonym0001": "012-345"}


class UserData(BaseModel):
    id: str
    name: str  # "alice"
    password: str  # "secret"
    ucsschoolRole: List[str]  # ["student"], ["teacher"] or ["legal_guardian"]
    pseudonyms: Dict[str, str]  # {"idBrokerPseudonym0001": "012-345"}


class SchoolData(BaseModel):
    id: str
    name: str  # "TEST-01"
    pseudonyms: Dict[str, str]  # {"idBrokerPseudonym0001": "012-345"}
    groups: Dict[str, GroupData]  # {"1a": <GroupData>}
    users: Dict[str, UserData]  # {"alice": <UserData>}


class TestData:
    """
    >>> from pathlib import Path
    >>> from performance_tests.common import TestData
    >>> data = TestData(Path("test-data-db"))
    >>> data.school_authorities
    ['Traeger1', 'Traeger2']
    >>> data.school_authority("Traeger1")
    SchoolAuthorityData(
        name='Traeger1',
        prov_api_admins={'provisioning-Traeger1': 'univentionunivention'},
        schools={"TEST-01": "abc-123", "TEST-02": "456-efg"})
    >>> data.school("Traeger1", "TEST-72")
    SchoolData(
        id='768d6b20-7437-4d7f-a225-5deb308a5bde',
        name='TEST-72',
        pseudonyms={'idBrokerPseudonym0029': '5a370e9b-213a...},
        groups={'1a': GroupData(id='e82069eb-259b...', name='1a', memberUid=['Traeger1-TEST-72-volle.k', ...])},
        users={'bogumil.plue': UserData(
            id='40b6fb5d-aa26...',
            name='bogumil.plue',
            password='univentionunivention',
            ucsschoolRole=['student'],
            pseudonyms={'idBrokerPseudonym0029': '4e3c002a...'}))
    """

    def __init__(self, db_path: Path = None):
        """
        The database directory is search in the following order:
        1. parameter `db_path`
        2. environment variable `ID_BROKER_TEST_DATA`
        3. `/var/lib/id-broker-performance-tests/test-data-db` (`DEFAULT_DATA_DIR`)
        """
        path = self._data_dir(db_path)
        self.db = Index(str(path))

    @property
    def school_authorities(self) -> List[str]:
        """List of school authority names: ["Traeger1", "Traeger2"]"""
        return self.db["SCHOOL_AUTHORITIES"]

    def school_authority(self, name: str) -> SchoolAuthorityData:
        """Data about a school authority (e.g. "Traeger1")."""
        data = self.db[name]
        data["number_of_schools"] = len(data["schools"].keys())
        return SchoolAuthorityData(name=name, **data)

    def school(self, school_authority_name: str, school_name: str) -> SchoolData:
        """Data about a school in a school authority (e.g. "TEST-01" of "Traeger1")."""
        school_data = self.db[f"{school_authority_name}-{school_name}"]
        group_data = school_data.pop("groups")
        groups = {}
        for group_name, group_attrs in group_data.items():
            pseudonyms = {k: v for k, v in group_attrs.items() if k.startswith("idBrokerPseudonym")}
            groups[group_name] = GroupData(
                id=group_attrs["id"],
                name=group_name,
                memberUid=group_attrs["memberUid"],
                pseudonyms=pseudonyms,
            )
        user_data = school_data.pop("users")
        users = {}
        for user_name, user_attrs in user_data.items():
            pseudonyms = {k: v for k, v in user_attrs.items() if k.startswith("idBrokerPseudonym")}
            users[user_name] = UserData(
                id=user_attrs["id"],
                name=user_name,
                password=user_attrs["password"],
                pseudonyms=pseudonyms,
                ucsschoolRole=user_attrs["ucsschoolRole"],
            )
        pseudonyms = {k: v for k, v in school_data.items() if k.startswith("idBrokerPseudonym")}
        return SchoolData(
            name=school_name, pseudonyms=pseudonyms, groups=groups, users=users, **school_data
        )

    @staticmethod
    def _data_dir(db_path: Path = None) -> Path:
        """
        If `db_path` is not set
        1. try environment variable `ID_BROKER_TEST_DATA`
        2. try `/var/lib/id-broker-performance-tests/test-data-db`
        3. raise ValueError
        """
        if db_path:
            return db_path
        try:
            _db_path = os.environ[ENV_DATA_DIR]
            return Path(_db_path)
        except KeyError:
            pass
        if DEFAULT_DATA_DIR.exists():
            return DEFAULT_DATA_DIR
        raise ValueError(
            f"Either set parameter 'db_path', store the test DB in '{DEFAULT_DATA_DIR!s}' or export a path "
            f"to the environment variable {ENV_DATA_DIR!r}."
        )


def json_to_dict_cache(source_path: Path, target_path: Path) -> None:
    """
        Load `Traeger*-{ou}_{groups,schools,users}.json` and write data into a `diskcache.Index` structure.

        Top level database keys are the names of the school authorities (e.g. "Traeger1") with their own data
        (currently only the admin users for the Provisioning API) and the names of the schools that belong to it.
        The names of the school authorities are in a list found by the key "SCHOOL_AUTHORITIES".
        The schools data and the data of their users and groups can be accessed through top level keys named after
        the school prefixed with the school authority name (e.g. "Traeger1-TEST-01").

        {
            "SCHOOL_AUTHORITIES": ["Traeger1", "Traeger"],     # all school authority names
            "Traeger1": {                                      # school authority name
                "prov_api_admins": {                           # credentials for the Provisioning API
                    "provisioning-Traeger1": "secret",         # username & password for P.API for "Traeger1"
                },
                "schools": {"TEST-01": "xyz-123", ...},        # schools and their IDs of this school authority
            },
            "Traeger1-TEST-01": {                              # school name (prefixed with the school auth name)
                "id": "abc-def",                               # entryUUID of school "TEST-01"
                "idBrokerPseudonym0001": "012-345",            # pseudonym #1 of school "TEST-01"
                ...
                "groups": {                                    # groups of school "TEST-01"
                    "1a": {                                    # group name
                        "id": "ghi-jkl",                       # entryUUID of group "1a"
                        "memberUid": ["alice", "bob"],         # members of group "1a"
                        "idBrokerPseudonym0001": "678-901",    # pseudonym #1 of group "1a"
                    }
                },
                "users": {                                     # users of school "TEST-01"
                    "alice": {                                 # username
                        "id": "mno-pqr",                       # entryUUID of user "alice"
                        "password": "secret"                   # password of user "alice"
                        "ucsschoolRole": ["student"],          # role(s) of user "alice"
                        "idBrokerPseudonym0001": "234-567",    # pseudonym #1 of user "alice"
                        ...                                    # more pseudonyms
                    }
                },
            },
            "Traeger1-TEST-02": { ... },                       # school name (prefixed with the school auth name)
        }
    }
    """
    school_authorities = sorted(
        {
            JSON_FILENAME_PATTERN.match(path.name)["sa"]
            for path in sorted(source_path.glob("Traeger*.json"))
        }
    )
    print(f"Found school authorities: {school_authorities!r}.")
    db = Index(str(target_path))
    db["SCHOOL_AUTHORITIES"] = school_authorities
    for school_authority in school_authorities:
        print(f"Reading school authority: {school_authority!r}...")
        traeger = {"prov_api_admins": {}, "schools": {}}
        school = {"groups": {}, "users": {}}
        for path in sorted(source_path.glob(f"{school_authority}-*.json")):
            print(f"Reading file {path.name!r}...")
            m = JSON_FILENAME_PATTERN.match(path.name)
            school_authority = m["sa"]  # "Traeger1"
            school_name = m["ou"]  # "TEST-1"
            object_type = m["type"]  # "groups" / "schools" / "users"
            long_school_name = f"{school_authority}-{school_name}"
            data = json.load(path.open())
            if object_type == "groups":
                for classes_dict in data[school_authority].values():
                    for long_name, class_data in classes_dict.items():
                        class_name = long_name[len(f"{school_authority}-{school_name}-") :]
                        school["groups"][class_name] = class_data
                        school["groups"][class_name]["memberUid"] = [
                            uid[len(f"{school_authority}-{school_name}-") :]
                            for uid in class_data["memberUid"]
                        ]
            elif object_type == "schools":
                school_data = data[school_authority][long_school_name]
                school.update(school_data)
                traeger["schools"][school_name] = school_data["id"]
            elif object_type == "users":
                for key in data[school_authority].keys():
                    if key.startswith("provisioning-"):
                        traeger["prov_api_admins"][key] = data[school_authority][key]
                    else:
                        for long_name, user_data in data[school_authority][key].items():
                            user_name = long_name[len(f"{school_authority}-{school_name}-") :]
                            school["users"][user_name] = user_data
            else:
                raise RuntimeError(f"Unknown object_type={object_type!r}.")
            db[long_school_name] = school
        db[school_authority] = traeger
    db.cache.close()


def get_provisioning_api_admins(data: TestData) -> Dict[str, Tuple[str, str]]:
    """
    Get Provisioning API school authority admin users and passwords.

    :param data: `TestData` instance
    :returns mapping school authority -> (username, password): {"Traeger1": (username, password)}
    """
    res = {}
    for school_authority in data.school_authorities:
        school_authority_data = data.school_authority(school_authority)
        prov_api_admins = list(school_authority_data.prov_api_admins.items())
        credentials = tuple(random.choice(prov_api_admins))
        if credentials[0] == "provisioning-Traeger1":
            credentials = (credentials[0], PROVISIONING_USER_TRAEGER1_SECRET)
        if credentials[0] == "provisioning-Traeger2":
            credentials = (credentials[0], PROVISIONING_USER_TRAEGER2_SECRET)
        res[school_authority] = credentials
    return res


def get_classes(
    data: TestData, schools_per_authority: int = 5, school_authorities: List[str] = None
) -> Dict[str, Dict[str, Dict[str, str]]]:
    """
    Get school class names and IDs.

    :param data: `TestData` instance
    :param schools_per_authority: to reduce startup time, load only that many schools for each authority
    :param school_authorities: optional list of school authorities for which to get school class data
    :returns mapping school authority -> OU -> class -> ID: {"Traeger1": {"TEST-101": {"1a": "abc-123"}}}
    """
    school_authorities = school_authorities or data.school_authorities
    res = {}
    for school_authority in school_authorities:
        res[school_authority] = {}
        school_authority_data = data.school_authority(school_authority)
        schools = random.choices(list(school_authority_data.schools.keys()), k=schools_per_authority)
        for school in schools:
            school_data = data.school(school_authority, school)
            res[school_authority][school] = {
                group_data.name: group_data.id for group_data in school_data.groups.values()
            }
    return res


def get_users(
    data: TestData, schools_per_authority: int = 5, school_authorities: List[str] = None
) -> Dict[str, Dict[str, Dict[str, str]]]:
    """
    Get usernames and IDs.

    :param data: `TestData` instance
    :param schools_per_authority: to reduce startup time, load only that many schools for each authority
    :param school_authorities: optional list of school authorities for which to get school class data
    :returns mapping school authority -> OU -> class -> ID: {"Traeger1": {"TEST-101": {"alice": "def-123"}}}
    """
    school_authorities = school_authorities or data.school_authorities
    res = {}
    previously_used_ous = UsedSchools.load_previously_used()
    for school_authority in school_authorities:
        res[school_authority] = {}
        school_authority_data = data.school_authority(school_authority)
        schools = random.choices(list(school_authority_data.schools.keys()), k=schools_per_authority)
        for school in schools:
            previously_used_ous_for_authority = [
                used for used in previously_used_ous if used[0] == school_authority
            ]
            while (school_authority, school) in previously_used_ous:
                school = random.choice(list(school_authority_data.schools.keys()))
                if len(previously_used_ous_for_authority) >= school_authority_data.number_of_schools:
                    break
            school_data = data.school(school_authority, school)
            res[school_authority][school] = {
                user_data.name: user_data.id for user_data in school_data.users.values()
            }
    return res


def get_users_with_passwords(
    data: TestData, schools_per_authority: int = 5
) -> Dict[str, Dict[str, Dict[str, str]]]:
    """
    Get usernames and passwords.

    :param data: `TestData` instance
    :param schools_per_authority: to reduce startup time, load only that many schools for each authority
    :returns mapping school authority -> OU -> class ->
                ID: {"Traeger1": {"TEST-101": {"TEST-101-alice": "password"}}}
    """
    res = {}
    for school_authority in data.school_authorities:
        res[school_authority] = {}
        school_authority_data = data.school_authority(school_authority)
        schools = random.choices(list(school_authority_data.schools.keys()), k=schools_per_authority)
        for school in schools:
            school_data = data.school(school_authority, school)
            for user_data in school_data.users.values():
                res[school_authority][school] = {
                    f"{school}-{user_data.name}": user_data.password
                    for user_data in school_data.users.values()
                }
    return res


def get_schools(data: TestData) -> Dict[str, Dict[str, str]]:
    """
    Get school names and IDs.

    :param data: `TestData` instance
    :returns mapping school authority -> mapping school -> ID: {"Traeger1": {"TEST-01": "ghi-123"}}
    """
    res = {}
    previously_used_ous = UsedSchools.load_previously_used()
    for school_authority in data.school_authorities:
        used_ous = {ou for s_a, ou in previously_used_ous if s_a == school_authority}
        school_authority_data = data.school_authority(school_authority)
        res[school_authority] = school_authority_data.schools
        for ou in used_ous:
            if ou in res[school_authority]:
                del res[school_authority][ou]  # remove OUs whose objects were altered in previous tests
    return res


def get_member_data(
    data: TestData, schools: Iterable[str]
) -> Dict[str, Dict[str, Dict[str, List[str]]]]:
    """
    Get school classes and members.

    :param data: `TestData` instance
    :param schools: only collect classes and members for those OUs (will try same names in all authorities)
    :returns mapping school authority -> school -> class -> members: {"Traeger1": {"TEST-01": {"1a": ["alice"]}}}
    """
    res = {}
    for school_authority in data.school_authorities:
        res[school_authority] = {}
        for school in schools:
            try:
                school_data = data.school(school_authority, school)
            except IndexError:
                logging.warning(
                    "*** Ignoring unknown school %r in authority %r.", school, school_authority
                )
                continue
            res[school_authority][school] = {
                group_name: group_data.memberUid for group_name, group_data in school_data.groups.items()
            }
    return res


def get_role_data(data: TestData, schools: Iterable[str]) -> Dict[str, Dict[str, Dict[str, List[str]]]]:
    """
    Get school classes and members.

    :param data: `TestData` instance
    :param schools: only collect roles for user in those OUs (will try same names in all authorities)
    :returns mapping school authority -> school -> user -> roles: {"Traeger1": {"TEST-01": {"alice": ["student"]}}}
    """
    res = {}
    for school_authority in data.school_authorities:
        res[school_authority] = {}
        for school in schools:
            try:
                school_data = data.school(school_authority, school)
            except IndexError:
                logging.warning(
                    "*** Ignoring unknown school %r in authority %r.", school, school_authority
                )
                continue
            res[school_authority][school] = {
                username: user_data.ucsschoolRole for username, user_data in school_data.users.items()
            }
    return res


class ResponseError(Exception):
    ...


class TokenError(Exception):
    ...


class ProvisioningSchool(BaseModel):
    school_authority: str
    id: str
    name: str
    display_name: str = ""

    def __str__(self):
        return f"{self.__class__.__name__}(s_auth={self.school_authority!r}, id={self.id!r}, name={self.name!r})"


class ProvisioningSchoolClass(BaseModel):
    school_authority: str
    id: str
    name: str
    description: str = ""
    school: str
    members: List[str] = []

    def __str__(self):
        return f"{self.__class__.__name__}(s_auth={self.school_authority!r}, id={self.id!r}, name={self.name!r})"


class ProvisioningSchoolContext(BaseModel):
    classes: List[str]
    roles: List[str]


class ProvisioningUser(BaseModel):
    school_authority: str
    id: str
    first_name: str
    last_name: str = ""
    user_name: str
    context: Dict[str, ProvisioningSchoolContext]

    def __str__(self):
        return (
            f"{self.__class__.__name__}(s_auth={self.school_authority!r}, id={self.id!r}, "
            f"user_name={self.user_name!r})"
        )


class FastAPIClient(HttpUser):
    abstract = True
    auth_token_url = ""

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.auth_username = ""
        self.auth_password = ""
        self._access_token = ""  # do not access this var directly, use the "self.access_token" property
        self._update_time = 0.0
        self._sem_token = BoundedSemaphore()  # guard access to self._access_token

    def on_start(self) -> None:
        if not self.auth_username or not self.auth_password:
            raise RuntimeError("self.auth_username or self.auth_password not set.")
        _ = self.access_token  # side effect: fetch token

    @property
    def access_token(self) -> str:
        return self._get_access_token(self.client, self.host)

    @access_token.setter
    def access_token(self, token: str) -> None:
        self._set_access_token(token)

    def _head(self, *args, **kwargs) -> LucustCMResponse:
        kwargs.setdefault("headers", {}).update(
            {"accept": "application/json", "Authorization": self.access_token}
        )
        logging.debug("_head() args=%r kwargs=%r", args, kwargs)
        with self.client.head(*args, catch_response=True, verify=False, **kwargs) as response:
            return cast(LucustCMResponse, response)

    def _get_json(
        self, *args, **kwargs
    ) -> Tuple[Union[Dict[str, Any], List[Dict[str, Any]]], LucustCMResponse]:
        kwargs.setdefault("headers", {}).update(
            {"accept": "application/json", "Authorization": self.access_token}
        )
        logging.debug("_get_json() args=%r kwargs=%r", args, kwargs)
        with self.client.get(*args, catch_response=True, verify=False, **kwargs) as response:
            response = cast(LucustCMResponse, response)
            response_json = self._response_to_json(response)
            return response_json, response

    def _get_json_list(self, *args, **kwargs) -> Tuple[List[Dict[str, Any]], LucustCMResponse]:
        res, response = self._get_json(*args, **kwargs)
        if not res:
            res = []
        return res, response

    def _post_json(self, *args, **kwargs) -> Tuple[Dict[str, Any], LucustCMResponse]:
        kwargs.setdefault("headers", {}).update(
            {"accept": "application/json", "Authorization": self.access_token}
        )
        logging.debug("_post_json() args=%r kwargs=%r", args, kwargs)
        with self.client.post(*args, catch_response=True, verify=False, **kwargs) as response:
            response = cast(LucustCMResponse, response)
            response_json = self._response_to_json(response)
            return response_json, response

    def _put_json(self, *args, **kwargs) -> Tuple[Dict[str, Any], LucustCMResponse]:
        kwargs.setdefault("headers", {}).update(
            {"accept": "application/json", "Authorization": self.access_token}
        )
        logging.debug("_put_json() args=%r kwargs=%r", args, kwargs)
        with self.client.put(*args, catch_response=True, verify=False, **kwargs) as response:
            response = cast(LucustCMResponse, response)
            response_json = self._response_to_json(response)
            return response_json, response

    def _delete(self, *args, **kwargs) -> LucustCMResponse:
        kwargs.setdefault("headers", {}).update(
            {"accept": "application/json", "Authorization": self.access_token}
        )
        logging.debug("_delete() args=%r kwargs=%r", args, kwargs)
        with self.client.delete(*args, catch_response=True, verify=False, **kwargs) as response:
            response = cast(LucustCMResponse, response)
            if response.status_code != 204:
                msg = f"status code: {response.status_code} Response text: {response.text}"
                response.failure(msg)
                raise ResponseError(msg)
            return response

    def _response_to_json(
        self, response: LucustCMResponse
    ) -> Union[Dict[str, Any], List[Dict[str, Any]]]:
        if response.status_code == 401:
            msg = f"status code: {response.status_code} Response text: {response.text}"
            response.failure(msg)
            self.access_token = ""
            raise ResponseError(msg)
        elif response.status_code < 200 or response.status_code > 299:
            msg = f"status code: {response.status_code} Response text: {response.text}"
            response.failure(msg)
            raise ResponseError(msg)
        try:
            return response.json()
        except JSONDecodeError as exc:
            msg = f"Response could not be decoded as JSON: {exc!s}"
            response.failure(msg)
            raise ResponseError(msg) from exc

    @staticmethod
    def _get_token_expiry(token: str) -> datetime.datetime:
        """Get the time at which `token` expires."""
        actual_token = token.rsplit(" ", 1)[-1]
        payload = jwt.decode(actual_token, algorithm="HS256", options={"verify_signature": False})
        exp: str = payload.get("exp")
        ts = int(exp)
        return datetime.datetime.fromtimestamp(ts)

    def _get_access_token(self, http_client, server) -> str:
        with self._sem_token:
            if not self._access_token or time.time() >= self._update_time:
                logging.info(f"Fetching access token for {self.auth_username!r} from {server!r}...")
                url = f"https://{server}{self.auth_token_url}"
                self._access_token = self._get_fastapi_token(
                    http_client, url, self.auth_username, self.auth_password
                )
                self._update_time = time.time() + 3600 - 10  # refresh (shortly before) every hour
        return self._access_token

    def _set_access_token(self, token: str) -> None:
        with self._sem_token:
            self._access_token = token

    @staticmethod
    def _get_fastapi_token(client: LocustClient, url: str, username: str, password: str) -> str:
        """Get an access token from the server."""
        headers = {"accept": "application/json", "Content-Type": "application/x-www-form-urlencoded"}
        data = {"username": username, "password": password}
        with client.post(url, data=data, headers=headers, catch_response=True, verify=False) as response:
            response = cast(LucustCMResponse, response)
            try:
                response_json = response.json()
            except JSONDecodeError as exc:
                raise TokenError(
                    f"Fetching token for {username!r}: response could not be decoded as JSON: {exc!s}\n"
                    f"response.text: {response.text!r}"
                ) from exc
            try:
                return f"{response_json['token_type']} {response_json['access_token']}"
            except KeyError as exc:
                raise TokenError(
                    f"Fetching token for {username!r}: response did not have expected content: {response_json!r}"
                ) from exc

    def _head_resource(self, url: str, _url_name: str = None) -> LucustCMResponse:
        return self._head(url, name=_url_name)

    def _get_resource(self, url: str, _url_name: str = None) -> Dict[str, Any]:
        res, _ = self._get_json(url, name=_url_name)
        return res

    def _get_resource_list(
        self, url: str, _url_name: str = None, params: Dict[str, Any] = None
    ) -> List[Dict[str, Any]]:
        res, _ = self._get_json_list(url, name=_url_name, params=params)
        return res

    def _create_resource(self, url: str, _url_name: str = None, **body) -> Dict[str, Any]:
        res, _ = self._post_json(url, name=_url_name, json=body)
        return res

    def _delete_resource(self, url: str, _url_name: str = None) -> LucustCMResponse:
        return self._delete(url, name=_url_name)

    def _update_resource(self, url: str, _url_name: str = None, **body) -> Dict[str, Any]:
        res, _ = self._put_json(url, name=_url_name, json=body)
        return res


class ProvisioningClient(FastAPIClient):
    abstract = True
    auth_token_url = UCSSCHOOL_APIS_TOKEN_URL

    def school_exists(self, school_authority: str, obj_id: str) -> bool:
        url = f"https://{self.host}{PROVISIONING_SCHOOL_URL.format(school_authority=school_authority, id=obj_id)}"
        res = self._head_resource(url, _url_name=PROVISIONING_SCHOOL_URL)
        return res.status_code == 200

    def school_get(self, school_authority: str, obj_id: str) -> ProvisioningSchool:
        url = f"https://{self.host}{PROVISIONING_SCHOOL_URL.format(school_authority=school_authority, id=obj_id)}"
        res = self._get_resource(url, _url_name=PROVISIONING_SCHOOL_URL)
        return ProvisioningSchool(school_authority=school_authority, **res)

    def school_create(
        self, school_authority: str, obj_id: str, name: str, display_name: str = ""
    ) -> ProvisioningSchool:
        url_path = PROVISIONING_SCHOOL_URL.format(school_authority=school_authority, id="").rstrip("/")
        url = f"https://{self.host}{url_path}"
        data = {
            "id": obj_id,
            "name": name,
            "display_name": display_name or f"Display name: {name}",
        }
        res = self._create_resource(url, _url_name=PROVISIONING_SCHOOL_URL, **data)
        return ProvisioningSchool(school_authority=school_authority, **res)

    def school_class_get(self, school_authority: str, obj_id: str) -> ProvisioningSchoolClass:
        url_path = PROVISIONING_CLASS_URL.format(school_authority=school_authority, id=obj_id)
        url = f"https://{self.host}{url_path}"
        res = self._get_resource(url, _url_name=PROVISIONING_CLASS_URL)
        return ProvisioningSchoolClass(school_authority=school_authority, **res)

    def school_class_exists(self, school_authority: str, obj_id: str) -> bool:
        url_path = PROVISIONING_CLASS_URL.format(school_authority=school_authority, id=obj_id)
        url = f"https://{self.host}{url_path}"
        res = self._head_resource(url, _url_name=PROVISIONING_CLASS_URL)
        return res.status_code == 200

    def school_class_create(
        self,
        school_authority: str,
        obj_id: str,
        name: str,
        school: str,
        description: str = "",
        members: List[str] = None,
    ) -> ProvisioningSchoolClass:
        url_path = PROVISIONING_CLASS_URL.format(school_authority=school_authority, id="").rstrip("/")
        url = f"https://{self.host}{url_path}"
        data = {
            "id": obj_id,
            "name": name,
            "school": school,
            "description": description or f"Description: {name}",
            "members": members or [],
        }
        res = self._create_resource(url, _url_name=PROVISIONING_CLASS_URL, **data)
        return ProvisioningSchoolClass(school_authority=school_authority, **res)

    def school_class_update(
        self,
        school_authority: str,
        obj_id: str,
        name: str,
        school: str,
        description: str = "",
        members: List[str] = None,
    ) -> ProvisioningSchoolClass:
        url_path = PROVISIONING_CLASS_URL.format(school_authority=school_authority, id=obj_id)
        url = f"https://{self.host}{url_path}"
        data = {
            "id": obj_id,
            "name": name,
            "school": school,
            "description": description or f"Description: {name}",
            "members": members or [],
        }
        res = self._update_resource(url, _url_name=PROVISIONING_CLASS_URL, **data)
        return ProvisioningSchoolClass(school_authority=school_authority, **res)

    def school_class_remove(self, school_authority: str, obj_id: str) -> None:
        url_path = PROVISIONING_CLASS_URL.format(school_authority=school_authority, id=obj_id)
        url = f"https://{self.host}{url_path}"
        self._delete_resource(url, _url_name=PROVISIONING_CLASS_URL)

    def user_get(self, school_authority: str, obj_id: str) -> ProvisioningUser:
        url_path = PROVISIONING_USER_URL.format(school_authority=school_authority, id=obj_id)
        url = f"https://{self.host}{url_path}"
        res = self._get_resource(url, _url_name=PROVISIONING_USER_URL)
        return ProvisioningUser(school_authority=school_authority, **res)

    def user_exists(self, school_authority: str, obj_id: str) -> bool:
        url_path = PROVISIONING_USER_URL.format(school_authority=school_authority, id=obj_id)
        url = f"https://{self.host}{url_path}"
        res = self._head_resource(url, _url_name=PROVISIONING_USER_URL)
        return res.status_code == 200

    def user_create(
        self,
        school_authority: str,
        obj_id: str,
        first_name: str,
        last_name: str,
        user_name: str,
        context: Dict[str, ProvisioningSchoolContext],
    ) -> ProvisioningUser:
        url_path = PROVISIONING_USER_URL.format(school_authority=school_authority, id="").rstrip("/")
        url = f"https://{self.host}{url_path}"
        data = {
            "id": obj_id,
            "first_name": first_name,
            "last_name": last_name,
            "user_name": user_name,
            "context": {ou: ou_context.dict() for ou, ou_context in context.items()},
        }
        res = self._create_resource(url, _url_name=PROVISIONING_USER_URL, **data)
        return ProvisioningUser(school_authority=school_authority, **res)

    def user_update(
        self,
        school_authority: str,
        obj_id: str,
        first_name: str,
        last_name: str,
        user_name: str,
        context: Dict[str, ProvisioningSchoolContext],
    ) -> ProvisioningUser:
        url_path = PROVISIONING_USER_URL.format(school_authority=school_authority, id=obj_id)
        url = f"https://{self.host}{url_path}"
        data = {
            "id": obj_id,
            "first_name": first_name,
            "last_name": last_name,
            "user_name": user_name,
            "context": {ou: ou_context.dict() for ou, ou_context in context.items()},
        }
        res = self._update_resource(url, _url_name=PROVISIONING_USER_URL, **data)
        return ProvisioningUser(school_authority=school_authority, **res)

    def user_remove(self, school_authority: str, obj_id: str) -> None:
        url_path = PROVISIONING_USER_URL.format(school_authority=school_authority, id=obj_id)
        url = f"https://{self.host}{url_path}"
        self._delete_resource(url, _url_name=PROVISIONING_USER_URL)


class UsedSchools:
    _schools: Set[Tuple[str, str]] = set()
    _sem_schools = BoundedSemaphore()  # guards access to _schools
    _sem_storage = BoundedSemaphore()  # guards access to _storage
    _storage = USED_SCHOOLS_LOG

    def __init__(self):
        self.load_previously_used()

    @classmethod
    def add(cls, school_authority: str, school: str) -> None:
        with cls._sem_schools:
            cls._schools.add((school_authority, school))

    @classmethod
    def clear(cls) -> None:
        with cls._sem_schools:
            cls._schools.clear()

    @property
    def schools(self) -> Set[Tuple[str, str]]:
        with self._sem_schools:
            return self._schools.copy()

    @classmethod
    def persist(cls) -> None:
        cls.load_previously_used()
        with cls._sem_storage, cls._storage.open("w") as fp:
            fp.writelines(f"{s_a}-{ou}\n" for s_a, ou in sorted(cls._schools))

    @classmethod
    def load_previously_used(cls) -> Set[Tuple[str, str]]:
        with cls._sem_schools:
            loaded = cls._load_previously_used_from_storage()
            cls._schools.update(loaded)
        return cls._schools

    @classmethod
    def _load_previously_used_from_storage(cls) -> Set[Tuple[str, str]]:
        res = set()
        with cls._sem_storage:
            try:
                for line in cls._storage.open("r"):
                    s_a, ou = line.split("-", 1)
                    res.add((s_a, ou.strip()))
            except FileNotFoundError:
                return res
        return res

    @classmethod
    def length(cls, sa) -> int:
        i = 0
        for school in cls._schools:
            if school[0] == sa:
                i += 1
        return i
