# -*- coding: utf-8 -*-

# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.
from typing import Iterable, List

from idbroker.utils import obj_event_on_configured_schools
from ucsschool_id_connector.models import ListenerObject, SchoolAuthorityConfiguration
from ucsschool_id_connector.plugins import hook_impl, plugin_manager
from ucsschool_id_connector.queues import InQueue, OutQueue
from ucsschool_id_connector.utils import ConsoleAndFileLogging

# That are all existing plugins for the id broker. If names change, this should be updated!
ID_BROKER_PLUGIN_NAMES = {"id_broker-users", "id_broker-groups"}


class IDBrokerDistributionImpl:
    def __init__(self):
        self.logger = ConsoleAndFileLogging.get_logger(self.__class__.__name__)

    @staticmethod
    def _get_id_broker_school_authorities(
        out_queues: List[OutQueue],
    ) -> Iterable[SchoolAuthorityConfiguration]:
        configurations = []
        for out_queue in out_queues:
            if any(plugin in ID_BROKER_PLUGIN_NAMES for plugin in out_queue.school_authority.plugins):
                configurations.append(out_queue.school_authority)
        return configurations

    @hook_impl
    async def school_authorities_to_distribute_to(
        self, obj: ListenerObject, in_queue: InQueue
    ) -> Iterable[str]:
        """
        We ignore the global school_authority mapping and check the schools configured in the plugin.
        """
        configurations = self._get_id_broker_school_authorities(in_queue.out_queues)
        authorities = []
        for sac in configurations:
            if self.obj_should_be_distributed(obj, sac):
                self.logger.debug(f"Active ID Broker configuration found: {sac.name}.")
                authorities.append(sac.name)
        return authorities

    def obj_should_be_distributed(
        self,
        obj: ListenerObject,
        sac: SchoolAuthorityConfiguration,
    ) -> bool:
        if not sac:
            self.logger.debug("No configuration")
            return False
        if not sac.active:
            self.logger.debug("Configuration not active")
            return False
        if "schools" not in sac.plugin_configs["id_broker"]:
            self.logger.warning("schools missing in configuration")
            return False
        if not sac.plugin_configs["id_broker"]["schools"]:
            self.logger.debug("No schools configured")
            return False
        if obj_event_on_configured_schools(obj, sac):
            self.logger.debug(f"{obj} has active school")
            return True
        self.logger.debug(f"No active school for {obj}")
        return False


plugin_manager.register(IDBrokerDistributionImpl())
