from typing import Callable
from unittest.mock import AsyncMock, MagicMock

DEFAULT_SCHOOL_AUTHORITY = "test_authority"
PREFIX = f"{DEFAULT_SCHOOL_AUTHORITY}-"
CLASS_SIZE = 5


class DictObject(dict):
    def __getattr__(self, item):
        try:
            return self[item]
        except KeyError:
            return super().__getattr__(item)

    def __setattr__(self, key, value):
        self[key] = value

    def delete(self):
        return


class Dictable(DictObject):
    """We need to mock a kelvin user object, which has an as_dict method"""

    def as_dict(self):
        return self

    def dict(self):
        return self

    async def save(self):
        return

    async def delete(self):
        return


class KelvinSessionMock(AsyncMock):
    # For convenience return a magic mock for any url
    # this can also be done with chained calls
    # The return value can be configured with get_preconfigured_session_mock
    urls = MagicMock()


def get_preconfigured_session_mock(config=None) -> Callable:
    """
    Mock a kelvin session with chained calls to do e.g. self.session.post()
    fast-api overrides have to be callables, so we simply return a
    >>> callable_mock = get_preconfigured_session_mock({'post.return_value': "someobject"})})
    >>> callable_mock()

    :param config:
    :return: callable kelvin session mock
    """
    if config is None:
        config = {}
    my_mock = KelvinSessionMock()
    my_mock.configure_mock(**config)

    async def call_me():
        return my_mock

    return call_me


def kelvin_school(i: int = 1) -> Dictable:
    return Dictable(
        name=f"{PREFIX}School{i}",
        display_name=f"School{i} Display",
        udm_properties={"ucsschoolRecordUID": f"School{i}-id"},
    )


def kelvin_school_class(i: int = 1) -> Dictable:
    return Dictable(
        name=f"{i}a",
        description=f"{i}a of School1",
        school=f"{PREFIX}School1",
        users=[f"{PREFIX}student{i}" for i in range(CLASS_SIZE)],
        udm_properties={"ucsschoolRecordUID": f"{i}a-id"},
    )


def kelvin_workgroup(i: int = 1) -> Dictable:
    return Dictable(
        name=f"Wg{i}",
        description=f"Wg{i} of School1",
        school=f"{PREFIX}School1",
        users=[f"{PREFIX}student{i}" for i in range(CLASS_SIZE)],
        udm_properties={"ucsschoolRecordUID": f"Wg{i}-id"},
    )


def kelvin_student(i: int = 1) -> Dictable:
    return Dictable(
        name=f"{PREFIX}student{i}",
        roles=["student"],
        firstname=f"stu{i}",
        lastname="dent",
        school_classes={f"{PREFIX}School1": ["1a"]},
        workgroups={f"{PREFIX}School1": ["Wg1"]},
        school=f"{PREFIX}School1",
        schools=[f"{PREFIX}School1"],
        ucsschool_roles=[f"student:school:{PREFIX}School1"],
        record_uid=f"student{i}",
        birthday=None,
        disabled=False,
        email=None,
        expiration_date=None,
        source_uid="TESTID",
        dn="cn=container,dc=base",
    )
