import uuid

import pytest
import requests
from tenacity import Retrying, stop_after_attempt, wait_fixed

from provisioning_plugin.models import SchoolContext, User
from ucsschool.kelvin.client import NoObject, User as KelvinUser, UserResource

from ..utils import (
    DEFAULT_SCHOOL_AUTHORITY,
    DEFAULT_SCHOOL_NAME,
    EXPECTED_HEADER_NAME,
    must_run_in_docker,
)

pytestmark = must_run_in_docker


@pytest.mark.asyncio
async def test_users_get_legal_guardian(
    create_school, create_user, url_fragment, auth_headers, kelvin_session_obj
):
    await create_school()
    legal_guardian = await create_user(roles=["legal_guardian"])
    user = await create_user(roles=["student"], legal_guardians=[legal_guardian.name])
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    response = requests.get(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/{user.record_uid}",
        headers=headers,
    )
    assert response.status_code == 200, response.__dict__
    response_user = User(**response.json())
    assert response_user == await User.from_kelvin_user(
        user, DEFAULT_SCHOOL_AUTHORITY
    ), response.__dict__
    legal_guardians = response_user.legal_guardians
    assert len(legal_guardians) == 1


@pytest.mark.asyncio
async def test_users_get(create_school, create_user, url_fragment, auth_headers):
    await create_school()
    user = await create_user()
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.get(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/{user.record_uid}",
        headers=headers,
    )
    assert response.status_code == 200, response.__dict__
    assert User(**response.json()) == await User.from_kelvin_user(
        user, DEFAULT_SCHOOL_AUTHORITY
    ), response.__dict__
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id


@pytest.mark.asyncio
async def test_users_head(create_school, create_user, url_fragment, auth_headers):
    await create_school()
    user = await create_user()
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.head(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/{user.record_uid}",
        headers=headers,
    )
    assert response.status_code == 200, response.__dict__
    assert response.content == b""
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id


@pytest.mark.asyncio
async def test_school_classes_get_wrong_auth(create_school, create_user, url_fragment, auth_headers):
    await create_school(DEFAULT_SCHOOL_AUTHORITY)
    await create_school("TEST2")
    user = await create_user()
    headers = await auth_headers("TEST2")
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.get(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/{user.record_uid}",
        headers=headers,
    )
    assert response.status_code == 403, response.__dict__
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id


@pytest.mark.asyncio
@pytest.mark.parametrize(
    "username,expected_code",
    [
        pytest.param(lambda faker: faker.pystr(1, 7), 201, id="username normal"),
        pytest.param(
            lambda faker: faker.pystr(None, 100 - len(f"{DEFAULT_SCHOOL_AUTHORITY}-")),
            201,
            id="username max length",
        ),
        pytest.param(
            lambda faker: faker.pystr(None, 101 - len(f"{DEFAULT_SCHOOL_AUTHORITY}-")),
            400,
            id="username too long",
        ),
    ],
)
async def test_users_post(
    create_school,
    create_school_class,
    create_workgroup,
    url_fragment,
    auth_headers,
    kelvin_session_obj,
    faker,
    user_cleaner,
    username,
    expected_code,
):
    await create_school()
    school_class = await create_school_class()
    workgroup = await create_workgroup()
    new_user = User(
        id=faker.pystr(1, 7),
        first_name=faker.first_name(),
        last_name=faker.last_name(),
        user_name=username(faker),
        context={
            DEFAULT_SCHOOL_NAME: SchoolContext(
                roles={"teacher"},
                classes={school_class.name},
                workgroups={workgroup.name},
            )
        },
    )
    user_cleaner(new_user.id, DEFAULT_SCHOOL_AUTHORITY)
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.post(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/",
        data=new_user.json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == expected_code, response.__dict__
    if expected_code == 400:
        return
    for attempt in Retrying(stop=stop_after_attempt(12), wait=wait_fixed(10)):
        with attempt:
            user = await UserResource(session=kelvin_session_obj).get(
                name=f"{DEFAULT_SCHOOL_AUTHORITY}-{new_user.user_name}"
            )
    assert await User.from_kelvin_user(user, DEFAULT_SCHOOL_AUTHORITY) == new_user
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id


@pytest.mark.asyncio
async def test_users_post_legal_guardians(
    create_school,
    create_school_class,
    create_workgroup,
    create_user,
    url_fragment,
    auth_headers,
    kelvin_session_obj,
    faker,
    user_cleaner,
):
    await create_school()
    school_class = await create_school_class()
    workgroup = await create_workgroup()
    legal_guardian = await create_user(roles=["legal_guardian"])
    legal_guardian_uuids = await User.names_to_uuids(DEFAULT_SCHOOL_AUTHORITY, [legal_guardian.name])
    new_user = User(
        id=faker.pystr(1, 7),
        first_name=faker.first_name(),
        last_name=faker.last_name(),
        user_name=faker.pystr(1, 7),
        legal_guardians=legal_guardian_uuids,
        context={
            DEFAULT_SCHOOL_NAME: SchoolContext(
                roles={"student"},
                classes={school_class.name},
                workgroups={workgroup.name},
            )
        },
    )
    user_cleaner(new_user.id, DEFAULT_SCHOOL_AUTHORITY)
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    response = requests.post(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/",
        data=new_user.json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == 201, response.__dict__
    for attempt in Retrying(stop=stop_after_attempt(12), wait=wait_fixed(10)):
        with attempt:
            user: KelvinUser = await UserResource(session=kelvin_session_obj).get(
                name=f"{DEFAULT_SCHOOL_AUTHORITY}-{new_user.user_name}"
            )
    assert await User.from_kelvin_user(user, DEFAULT_SCHOOL_AUTHORITY) == new_user
    assert user.legal_guardians == [legal_guardian.name]


@pytest.mark.asyncio
async def test_users_post_legal_guardians_wrong_role(
    create_school,
    create_school_class,
    create_workgroup,
    create_user,
    url_fragment,
    auth_headers,
    kelvin_session_obj,
    faker,
    user_cleaner,
):
    await create_school()
    school_class = await create_school_class()
    workgroup = await create_workgroup()
    legal_guardian = await create_user(roles=["legal_guardian"])
    legal_guardian_uuids = await User.names_to_uuids(DEFAULT_SCHOOL_AUTHORITY, [legal_guardian.name])
    new_user = User(
        id=faker.pystr(1, 7),
        first_name=faker.first_name(),
        last_name=faker.last_name(),
        user_name=faker.pystr(1, 7),
        legal_guardians=legal_guardian_uuids,
        context={
            DEFAULT_SCHOOL_NAME: SchoolContext(
                roles={"teacher"},
                classes={school_class.name},
                workgroups={workgroup.name},
            )
        },
    )
    user_cleaner(new_user.id, DEFAULT_SCHOOL_AUTHORITY)
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    response = requests.post(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/",
        data=new_user.json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == 400, response.__dict__
    assert response.json() == {
        "detail": "Parameter 'legal_guardians' only valid for user-role 'student'"
    }


@pytest.mark.asyncio
async def test_users_post_legal_wards(
    create_school,
    create_school_class,
    create_workgroup,
    create_user,
    url_fragment,
    auth_headers,
    kelvin_session_obj,
    faker,
    user_cleaner,
):
    await create_school()
    school_class = await create_school_class()
    workgroup = await create_workgroup()
    student = await create_user(roles=["student"])
    student_uuids = await User.names_to_uuids(DEFAULT_SCHOOL_AUTHORITY, [student.name])
    new_user = User(
        id=faker.pystr(1, 7),
        first_name=faker.first_name(),
        last_name=faker.last_name(),
        user_name=faker.pystr(1, 7),
        legal_wards=student_uuids,
        context={
            DEFAULT_SCHOOL_NAME: SchoolContext(
                roles={"legal_guardian"},
                classes={school_class.name},
                workgroups={workgroup.name},
            )
        },
    )
    user_cleaner(new_user.id, DEFAULT_SCHOOL_AUTHORITY)
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    response = requests.post(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/",
        data=new_user.json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == 201, response.__dict__
    for attempt in Retrying(stop=stop_after_attempt(12), wait=wait_fixed(10)):
        with attempt:
            user: KelvinUser = await UserResource(session=kelvin_session_obj).get(
                name=f"{DEFAULT_SCHOOL_AUTHORITY}-{new_user.user_name}"
            )
    assert await User.from_kelvin_user(user, DEFAULT_SCHOOL_AUTHORITY) == new_user
    assert user.legal_wards == [student.name]


@pytest.mark.asyncio
async def test_users_post_legal_wards_wrong_role(
    create_school,
    create_school_class,
    create_workgroup,
    create_user,
    url_fragment,
    auth_headers,
    kelvin_session_obj,
    faker,
    user_cleaner,
):
    await create_school()
    school_class = await create_school_class()
    workgroup = await create_workgroup()
    student = await create_user(roles=["student"])
    student_uuids = await User.names_to_uuids(DEFAULT_SCHOOL_AUTHORITY, [student.name])
    new_user = User(
        id=faker.pystr(1, 7),
        first_name=faker.first_name(),
        last_name=faker.last_name(),
        user_name=faker.pystr(1, 7),
        legal_wards=student_uuids,
        context={
            DEFAULT_SCHOOL_NAME: SchoolContext(
                roles={"teacher"},
                classes={school_class.name},
                workgroups={workgroup.name},
            )
        },
    )
    user_cleaner(new_user.id, DEFAULT_SCHOOL_AUTHORITY)
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    response = requests.post(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/",
        data=new_user.json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == 400, response.__dict__
    assert response.json() == {
        "detail": "Parameter 'legal_wards' only valid for user-role 'legal_guardian'"
    }


@pytest.mark.asyncio
async def test_users_put(
    create_school,
    create_school_class,
    create_workgroup,
    create_user,
    url_fragment,
    auth_headers,
    kelvin_session_obj,
    faker,
):
    await create_school()
    school_class = await create_school_class()
    workgroup = await create_workgroup()
    user = await create_user(
        school_classes={DEFAULT_SCHOOL_NAME: [school_class.name]},
        workgroups={DEFAULT_SCHOOL_NAME: [workgroup.name]},
    )
    new_user = await User.from_kelvin_user(user, DEFAULT_SCHOOL_AUTHORITY)
    new_user.first_name = faker.first_name()
    new_user.user_name = faker.pystr(1, 7)
    new_user.context[DEFAULT_SCHOOL_NAME].roles = {"teacher", "staff"}
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.put(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/{user.record_uid}",
        data=new_user.json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == 200, response.__dict__
    for attempt in Retrying(stop=stop_after_attempt(12), wait=wait_fixed(10)):
        with attempt:
            kelvin_user = await UserResource(session=kelvin_session_obj).get(
                name=f"{DEFAULT_SCHOOL_AUTHORITY}-{new_user.user_name}"
            )
    assert await User.from_kelvin_user(kelvin_user, DEFAULT_SCHOOL_AUTHORITY) == new_user
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id


@pytest.mark.asyncio
async def test_users_put_legal_guardians(
    create_user,
    url_fragment,
    auth_headers,
    kelvin_session_obj,
):
    user = await create_user(roles=["student"])
    new_user = await User.from_kelvin_user(user, DEFAULT_SCHOOL_AUTHORITY)
    legal_guardian = await create_user(roles=["legal_guardian"])
    legal_guardian_uuids = await User.names_to_uuids(DEFAULT_SCHOOL_AUTHORITY, [legal_guardian.name])
    new_user.legal_guardians = legal_guardian_uuids
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    response = requests.put(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/{user.record_uid}",
        data=new_user.json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == 200, response.__dict__
    for attempt in Retrying(stop=stop_after_attempt(12), wait=wait_fixed(10)):
        with attempt:
            kelvin_user: KelvinUser = await UserResource(session=kelvin_session_obj).get(
                name=f"{DEFAULT_SCHOOL_AUTHORITY}-{new_user.user_name}"
            )
    assert await User.from_kelvin_user(kelvin_user, DEFAULT_SCHOOL_AUTHORITY) == new_user
    assert kelvin_user.legal_guardians == [legal_guardian.name]


@pytest.mark.asyncio
async def test_users_put_legal_guardians_wrong_role(
    create_user,
    url_fragment,
    auth_headers,
    kelvin_session_obj,
):
    user = await create_user(roles=["teacher"])
    new_user = await User.from_kelvin_user(user, DEFAULT_SCHOOL_AUTHORITY)
    legal_guardian = await create_user(roles=["legal_guardian"])
    legal_guardian_uuids = await User.names_to_uuids(DEFAULT_SCHOOL_AUTHORITY, [legal_guardian.name])
    new_user.legal_guardians = legal_guardian_uuids
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    response = requests.put(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/{user.record_uid}",
        data=new_user.json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == 400, response.__dict__
    assert response.json() == {
        "detail": "Parameter 'legal_guardians' only valid for user-role 'student'"
    }


@pytest.mark.asyncio
async def test_users_put_legal_wards(
    create_user,
    url_fragment,
    auth_headers,
    kelvin_session_obj,
):
    user = await create_user(roles=["legal_guardian"])
    new_user = await User.from_kelvin_user(user, DEFAULT_SCHOOL_AUTHORITY)
    student = await create_user(roles=["student"])
    student_uuids = await User.names_to_uuids(DEFAULT_SCHOOL_AUTHORITY, [student.name])
    new_user.legal_wards = student_uuids
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    response = requests.put(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/{user.record_uid}",
        data=new_user.json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == 200, response.__dict__
    for attempt in Retrying(stop=stop_after_attempt(12), wait=wait_fixed(10)):
        with attempt:
            kelvin_user: KelvinUser = await UserResource(session=kelvin_session_obj).get(
                name=f"{DEFAULT_SCHOOL_AUTHORITY}-{new_user.user_name}"
            )
    assert await User.from_kelvin_user(kelvin_user, DEFAULT_SCHOOL_AUTHORITY) == new_user
    assert kelvin_user.legal_wards == [student.name]


@pytest.mark.asyncio
async def test_users_put_legal_wards_wrong_role(
    create_user,
    url_fragment,
    auth_headers,
    kelvin_session_obj,
):
    user = await create_user(roles=["teacher"])
    new_user = await User.from_kelvin_user(user, DEFAULT_SCHOOL_AUTHORITY)
    student = await create_user(roles=["student"])
    student_uuids = await User.names_to_uuids(DEFAULT_SCHOOL_AUTHORITY, [student.name])
    new_user.legal_wards = student_uuids
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    response = requests.put(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/{user.record_uid}",
        data=new_user.json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == 400, response.__dict__
    assert response.json() == {
        "detail": "Parameter 'legal_wards' only valid for user-role 'legal_guardian'"
    }


@pytest.mark.asyncio
async def test_users_delete(create_school, create_user, url_fragment, auth_headers, kelvin_session_obj):
    await create_school()
    user = await create_user()
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.delete(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/{user.record_uid}",
        headers=headers,
    )
    assert response.status_code == 204, response.__dict__
    for attempt in Retrying(stop=stop_after_attempt(12), wait=wait_fixed(10)):
        with attempt:
            with pytest.raises(NoObject):
                await UserResource(session=kelvin_session_obj).get(name=user.name)
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id


@pytest.mark.asyncio
async def test_users_no_id_change(
    create_school,
    create_school_class,
    create_user,
    url_fragment,
    auth_headers,
    kelvin_session_obj,
):
    await create_school()
    user = await create_user()
    new_user = await User.from_kelvin_user(user, DEFAULT_SCHOOL_AUTHORITY)
    new_user.id = "CHANGED ID"
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.put(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/{user.record_uid}",
        data=new_user.json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == 422, response.__dict__
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id
