import asyncio
import os
import random
import warnings
import zlib
from typing import Dict, List

import pytest
from ldap3 import BASE as SCOPE_BASE

from id_broker_common.kelvin import (
    kelvin_on_primary_session_kwargs,
    kelvin_session,
    kelvin_session_kwargs,
    load_kelvin_session_kwargs,
)
from id_broker_common.utils import ldap_credentials, ldap_settings
from provisioning_plugin.settings import PLUGIN_SETTINGS_FILE
from ucsschool.apis.utils import LDAPAccess
from ucsschool.kelvin.client import (
    InvalidRequest,
    NoObject,
    School as KelvinSchool,
    SchoolClass as KelvinSchoolClass,
    Session as KelvinSession,
    User as KelvinUser,
    UserResource,
    WorkGroup as KelvinWorkGroup,
)

from .utils import DEFAULT_PASSWORD, DEFAULT_SCHOOL_AUTHORITY, DEFAULT_SCHOOL_NAME, get_access_token


@pytest.fixture(autouse=True)
def faker_seed():
    return random.randint(0, 10000)


@pytest.fixture(scope="session")
def event_loop():
    loop = asyncio.get_event_loop()
    yield loop
    loop.close()


@pytest.fixture()
def create_school(kelvin_session_obj, wait_for_dn_exists):
    # kelvin_session_obj() will setup kelvin_session_kwargs
    async def _create_school(
        school_authority: str = DEFAULT_SCHOOL_AUTHORITY,
        name: str = DEFAULT_SCHOOL_NAME,
        display_name: str = None,
    ):
        if display_name is None:
            display_name = name
        async with KelvinSession(
            **kelvin_on_primary_session_kwargs(PLUGIN_SETTINGS_FILE), timeout=120
        ) as session:
            school = KelvinSchool(
                name=f"{school_authority}-{name}",
                display_name=display_name,
                session=session,
                educational_servers=[f"dc-{zlib.crc32(school_authority.encode('UTF-8'))}"],
                udm_properties={
                    "ucsschoolRecordUID": f"record_{school_authority}_{name}",
                    "ucsschoolSourceUID": school_authority,
                },
            )
            try:
                await school.save()
            except InvalidRequest as exc:
                if exc.status != 409:
                    raise
            await school.reload()
        await wait_for_dn_exists(school.dn)
        return school

    return _create_school


@pytest.fixture
async def user_cleaner(kelvin_session_obj):
    users = []

    def _user_cleaner(record_uid, source_uid):
        users.append((record_uid, source_uid))

    yield _user_cleaner

    for record, source in users:
        searched_users = [
            user
            async for user in UserResource(session=kelvin_session_obj).search(
                record_uid=record, source_uid=source
            )
        ]
        if len(searched_users) == 0:
            warnings.warn(f"The user {record}, {source} could not be found and thus was not cleaned up.")
        for kelvin_user in searched_users:
            await kelvin_user.delete()


@pytest.fixture()
def create_user(kelvin_session_obj, faker, user_cleaner, wait_for_dn_exists):
    async def _create_user(
        user_name: str = None,
        password: str = DEFAULT_PASSWORD,
        first_name: str = None,
        last_name: str = None,
        school_authority: str = DEFAULT_SCHOOL_AUTHORITY,
        school: str = DEFAULT_SCHOOL_NAME,
        roles: List[str] = None,
        school_classes: Dict[str, List[str]] = None,
        workgroups: Dict[str, List[str]] = None,
        legal_guardians: List[str] = None,
        legal_wards: List[str] = None,
    ):
        if user_name is None:
            user_name = faker.pystr(1, 7)
        if first_name is None:
            first_name = faker.first_name()
        if last_name is None:
            last_name = faker.last_name()
        if school_classes is None:
            school_classes = {}
        if workgroups is None:
            workgroups = {}
        if roles is None:
            roles = ["teacher"]
        if legal_guardians is None:
            legal_guardians = []
        if legal_wards is None:
            legal_wards = []
        _user_name = (
            user_name
            if user_name == f"provisioning-{school_authority}"
            else f"{school_authority}-{user_name}"
        )
        user = KelvinUser(
            firstname=first_name,
            lastname=last_name,
            name=_user_name,
            school=f"{school_authority}-{school}",
            schools=[f"{school_authority}-{school}"],
            roles=roles,
            password=password,
            source_uid=school_authority,
            record_uid=f"record_{_user_name}",
            session=kelvin_session_obj,
            school_classes={
                f"{school_authority}-{school}": school_classes
                for school, school_classes in school_classes.items()
            },
            workgroups={
                f"{school_authority}-{school}": workgroups for school, workgroups in workgroups.items()
            },
            legal_guardians=legal_guardians,
            legal_wards=legal_wards,
        )
        try:
            await user.save()
        except InvalidRequest as exc:
            if exc.status != 409:
                raise
            warnings.warn(f"recycling existing user {user.record_uid} {user.source_uid}")
        await wait_for_dn_exists(user.dn)
        user_cleaner(user.record_uid, user.source_uid)
        return user

    return _create_user


@pytest.fixture()
async def create_school_class(kelvin_session_obj, faker, wait_for_dn_exists):
    created_school_classes = []

    async def _create_school_class(
        school_authority: str = DEFAULT_SCHOOL_AUTHORITY,
        name: str = None,
        school: str = DEFAULT_SCHOOL_NAME,
        descr: str = "",
        members: List[str] = None,
    ):
        if name is None:
            name = faker.pystr(1, 10)
        if members is None:
            members = []
        school_class = KelvinSchoolClass(
            name=name,
            school=f"{school_authority}-{school}",
            description=descr,
            users=members,
            session=kelvin_session_obj,
            udm_properties={
                "ucsschoolRecordUID": f"record_class_{school_authority}_{name}",
                "ucsschoolSourceUID": school_authority,
            },
        )
        await school_class.save()
        await school_class.reload()
        await wait_for_dn_exists(school_class.dn)
        created_school_classes.append(school_class)
        return school_class

    yield _create_school_class

    for school_class in created_school_classes:
        try:
            await school_class.delete()
        except NoObject:
            pass


@pytest.fixture()
async def create_workgroup(kelvin_session_obj, faker, wait_for_dn_exists):
    created_workgroups: List[KelvinWorkGroup] = []

    async def _create_workgroup(
        school_authority: str = DEFAULT_SCHOOL_AUTHORITY,
        name: str = None,
        school: str = DEFAULT_SCHOOL_NAME,
        descr: str = "",
        members: List[str] = None,
    ):
        if name is None:
            name = faker.pystr(1, 10)
        if members is None:
            members = []
        workgroup = KelvinWorkGroup(
            name=name,
            school=f"{school_authority}-{school}",
            description=descr,
            users=members,
            session=kelvin_session_obj,
            udm_properties={
                "ucsschoolRecordUID": f"record_workgroup_{school_authority}_{name}",
                "ucsschoolSourceUID": school_authority,
            },
        )
        await workgroup.save()
        await workgroup.reload()
        await wait_for_dn_exists(workgroup.dn)
        created_workgroups.append(workgroup)
        return workgroup

    yield _create_workgroup

    for workgroup in created_workgroups:
        try:
            await workgroup.delete()
        except NoObject:
            pass


@pytest.fixture(scope="session")
def url_fragment():
    return f"http://{os.environ['DOCKER_HOST_NAME']}/ucsschool/apis"


@pytest.fixture()
async def kelvin_session_obj():
    kelvin_session_kwargs.update(load_kelvin_session_kwargs(PLUGIN_SETTINGS_FILE))
    session = await kelvin_session()
    return session


@pytest.fixture()
def auth_headers(create_user, create_school):
    async def _auth_headers(school_authority: str):
        await create_user(
            user_name=f"provisioning-{school_authority}", school_authority=school_authority
        )
        token = get_access_token(f"provisioning-{school_authority}")
        return {"Authorization": f"Bearer {token}"}

    return _auth_headers


@pytest.fixture()
def wait_for_dn_exists():
    """
    Wait until `dn` exists in LDAP or timeout hits.
    """

    async def _func(dn: str, timeout: int = 120, exists: bool = True) -> None:
        ldap = LDAPAccess(ldap_settings(), ldap_credentials())
        for i in range(timeout):
            print(f"wait for object {dn}")
            result = await ldap.search("(objectClass=*)", search_base=dn, search_scope=SCOPE_BASE)
            if len(result) != 0:
                break
            print(f"{dn} not found, retry ...")
            await asyncio.sleep(1)
        else:
            if exists:
                raise AssertionError(f"{dn} not found after {timeout} seconds!")

    return _func
