#!/usr/share/ucs-test/runner /usr/share/ucs-test/playwright
## -*- coding: utf-8 -*-
## desc: Check test app login, get access token and verify self disclosure user metadata
## tags: [ucsschool-id-broker]
## packages: [id-broker-provisioning-api-plugin]
## exposure: safe


import json
import os
import time

import ldap
import pytest
import requests
from playwright.sync_api import Page
from utils import get_traeger_ldap_admin_con, load_test_app_page

import univention.testing.strings as uts
from univention.admin.rest.client import UDM
from univention.testing.utils import retry_on_error


@pytest.fixture(scope="session")
def browser_context_args(browser_context_args, playwright):
    return {"ignore_https_errors": True, "locale": "en-US"}


@pytest.fixture(scope="session")
def t1_password():
    with open("/var/lib/ucs-test/Traeger1.secret", "r") as fd:
        return fd.read().strip()


@pytest.fixture(scope="session")
def t2_password():
    with open("/var/lib/ucs-test/Traeger2.secret", "r") as fd:
        return fd.read().strip()


@pytest.fixture
def victim_t1(t1_password):
    server = os.environ.get("UCS_ENV_TRAEGER1_FQDN")
    kelvin = f"https://{server}/ucsschool/kelvin"
    token = requests.post(
        f"{kelvin}/token",
        data={
            "grant_type": "password",
            "username": "Administrator",
            "password": t1_password,
        },
        verify=False,
    ).json()["access_token"]
    impersonated_user_t1 = requests.post(
        f"{kelvin}/v1/users/",
        json={
            "name": uts.random_username(),
            "firstname": "Bob",
            "lastname": "Innocent",
            "school": f"{kelvin}/v1/schools/ou1",
            "password": "univention123456789",
            "roles": [f"{kelvin}/v1/roles/teacher"],
            "record_uid": "impersonated_user_t1",
        },
        verify=False,
        headers={"Authorization": f"Bearer {token}"},
    ).json()
    assert "dn" in impersonated_user_t1
    yield impersonated_user_t1
    token = requests.post(
        f"{kelvin}/token",
        data={
            "grant_type": "password",
            "username": "Administrator",
            "password": t1_password,
        },
        verify=False,
    ).json()["access_token"]
    requests.delete(
        f"{kelvin}/v1/users/{impersonated_user_t1['name']}",
        verify=False,
        headers={"Authorization": f"Bearer {token}"},
    )


@pytest.fixture
def victim_t2(t2_password):
    server = os.environ.get("UCS_ENV_TRAEGER2_FQDN")
    kelvin = f"https://{server}/ucsschool/kelvin"
    token = requests.post(
        f"{kelvin}/token",
        data={
            "grant_type": "password",
            "username": "Administrator",
            "password": t2_password,
        },
        verify=False,
    ).json()["access_token"]
    impersonated_user_t2 = requests.post(
        f"{kelvin}/v1/users/",
        json={
            "name": uts.random_username(),
            "firstname": "Alice",
            "lastname": "Innocent",
            "school": f"{kelvin}/v1/schools/ou1",
            "password": "univention123456789",
            "roles": [f"{kelvin}/v1/roles/teacher"],
            "record_uid": "impersonated_user_t2",
        },
        verify=False,
        headers={"Authorization": f"Bearer {token}"},
    ).json()
    assert "dn" in impersonated_user_t2
    yield impersonated_user_t2
    token = requests.post(
        f"{kelvin}/token",
        data={
            "grant_type": "password",
            "username": "Administrator",
            "password": t2_password,
        },
        verify=False,
    ).json()["access_token"]
    requests.delete(
        f"{kelvin}/v1/users/{impersonated_user_t2['name']}",
        verify=False,
        headers={"Authorization": f"Bearer {token}"},
    )


@pytest.fixture
def attacker_t2(t2_password):
    server = os.environ.get("UCS_ENV_TRAEGER2_FQDN")
    lo_t2 = get_traeger_ldap_admin_con("Traeger2")
    # Avoid id connector erorrs due to the duplicate entry_uuid by using a non school user
    udm_t2 = UDM.http(
        f"http://{server}/univention/udm/",
        "Administrator",
        t2_password,
    )
    user_module_t2 = udm_t2.get("users/user")
    obj = user_module_t2.new()
    obj.properties["username"] = uts.random_username()
    obj.properties["lastname"] = "Impersonator"
    obj.properties["firstname"] = "Eve"
    obj.properties["password"] = "univention123456789"
    obj.save()
    impersonating_user = {"dn": obj.dn, "name": obj.properties["username"]}
    entry_uuid_old = lo_t2.get(impersonating_user["dn"], ("entryUUID",))["entryUUID"][0]
    yield impersonating_user
    entry_uuid_new = lo_t2.get(impersonating_user["dn"], ("entryUUID",))["entryUUID"][0]
    lo_t2.modify(
        impersonating_user["dn"],
        [("entryUUID", entry_uuid_new, entry_uuid_old)],
        serverctrls=[ldap.controls.simple.RelaxRulesControl()],
    )
    udm_t2 = UDM.http(
        f"http://{server}/univention/udm/",
        "Administrator",
        t2_password,
    )
    user_module_t2 = udm_t2.get("users/user")
    obj = user_module_t2.get(impersonating_user["dn"])
    obj.delete()


@pytest.fixture
def impersonation_users(udm, attacker_t2, victim_t1, victim_t2):
    udm_mod = udm.get("users/user")
    impersonation_users = {
        "impersonated_user_t1": victim_t1,
        "impersonated_user_t2": victim_t2,
        "impersonating_user": attacker_t2,
    }
    for user in [
        f"Traeger2-{victim_t2['name']}",
        f"Traeger1-{victim_t1['name']}",
    ]:
        print(f"Waiting for {user}")
        retry_on_error(lambda: list(udm_mod.search(f"uid={user}"))[0])
    time.sleep(6)  # sleep for sddb
    return impersonation_users


@pytest.mark.parametrize(
    "uid,kdc_idp_hint",
    [
        ("Traeger1-stud1", "traeger1"),
        ("Traeger1-stud1", "traeger1-kc-saml"),
        ("Traeger1-stud1", "traeger1-simple-saml"),
        ("Traeger1-teach1", "traeger1"),
        ("Traeger1-teach1", "traeger1-kc-saml"),
        ("Traeger1-teach1", "traeger1-simple-saml"),
        ("Traeger2-stud1", "traeger2"),
        ("Traeger2-teach1", "traeger2"),
    ],
)
def test_app_login(page: Page, udm, uid, kdc_idp_hint) -> None:
    traeger = uid.split("-")[0]
    username = uid.split("-", 1)[1]
    password = "univention"

    # Check the time on all systems, if this fails repeatedly!
    load_test_app_page(page, username, password, kdc_idp_hint)
    user_metadata = json.loads(page.locator("#userMetadata").inner_text().replace("'", '"'))
    user_groups = json.loads(page.locator("#groups").inner_text().replace("'", '"'))
    user_resolved_groups = json.loads(page.locator("#groupsResolved").inner_text().replace("'", '"'))

    udm_mod = udm.get("users/user")
    broker_user = list(udm_mod.search(f"uid={uid}"))[0]

    # check self disclosure user metadata
    assert user_metadata["username"] == "{}.{}".format(
        broker_user.props.firstname, broker_user.props.lastname
    )
    assert user_metadata["firstname"] == broker_user.props.firstname
    assert user_metadata["lastname"] == broker_user.props.lastname
    assert user_metadata["school_id"] in [
        school.replace(f"{traeger}-", "") for school in broker_user.props.school
    ]
    assert user_metadata["school_authority"] == broker_user.props.ucsschoolSourceUID
    assert user_metadata["type"] in [role.split(":")[0] for role in broker_user.props.ucsschoolRole]

    # check self disclosure user groups
    for group in user_groups["groups"]:
        cn = f"cn={group['school_id']}-{group['name']}"
        assert any(_group.startswith(cn) for _group in broker_user.props.groups)
        dn = [_group for _group in broker_user.props.groups if _group.startswith(cn)][0]
        if "cn=klassen" in dn:
            assert group["type"] == "school_class"
        else:
            assert group["type"] == "workgroup"
        if "stud" in uid:
            assert user_resolved_groups[group["group_id"]]["detail"] == "Forbidden"
        else:
            assert "students" in user_resolved_groups[group["group_id"]]
            assert "teachers" in user_resolved_groups[group["group_id"]]


def test_user_impersonation_single_idp(page: Page, udm, impersonation_users):
    # Test that logins from traegerX are trusted unconditional for users from traegerX
    lo_t2 = get_traeger_ldap_admin_con("Traeger2")
    traeger = "Traeger2"
    kdc_idp_hint = traeger.lower()
    password = "univention123456789"

    # Lets use a user from traeger2 to impersonate another user of traeger2.
    # We can do this by manipulating the entryUUID.
    # We fully trust the IdP of traeger2 for users from traeger2, so this should work.
    # This first test is to make sure this works in theory.

    entry_uuid_target = lo_t2.get(impersonation_users["impersonated_user_t2"]["dn"], ("entryUUID",))[
        "entryUUID"
    ][0]
    entry_uuid = lo_t2.get(impersonation_users["impersonating_user"]["dn"], ("entryUUID",))["entryUUID"][
        0
    ]

    lo_t2.modify(
        impersonation_users["impersonating_user"]["dn"],
        [("entryUUID", entry_uuid, entry_uuid_target)],
        serverctrls=[ldap.controls.simple.RelaxRulesControl()],
    )

    # Check the time on all systems, if this fails repeatedly!
    load_test_app_page(page, impersonation_users["impersonating_user"]["name"], password, kdc_idp_hint)
    user_metadata = json.loads(page.locator("#userMetadata").inner_text().replace("'", '"'))

    udm_mod = udm.get("users/user")
    # See, we used impersonating_user to login but the broker returns impersonated_user_t2
    broker_user = list(
        udm_mod.search(f"uid={traeger}-{impersonation_users['impersonated_user_t2']['name']}")
    )[0]
    assert user_metadata["username"] == "{}.{}".format(
        broker_user.props.firstname, broker_user.props.lastname
    )
    assert user_metadata["firstname"] == broker_user.props.firstname
    assert user_metadata["lastname"] == broker_user.props.lastname
    assert user_metadata["school_id"] in [
        school.replace(f"{traeger}-", "") for school in broker_user.props.school
    ]
    assert user_metadata["school_authority"] == broker_user.props.ucsschoolSourceUID
    assert user_metadata["type"] in [role.split(":")[0] for role in broker_user.props.ucsschoolRole]


def test_user_impersonation_cross_idp(page: Page, udm, impersonation_users):
    # Test that logins from traegerX are only trusted for users from traegerX and not for any other
    lo_t1 = get_traeger_ldap_admin_con("Traeger1")
    lo_t2 = get_traeger_ldap_admin_con("Traeger2")
    traeger = "Traeger2"
    kdc_idp_hint = traeger.lower()
    password = "univention123456789"

    entry_uuid = lo_t2.get(impersonation_users["impersonating_user"]["dn"], ("entryUUID",))["entryUUID"][
        0
    ]

    entry_uuid_target = lo_t1.get(impersonation_users["impersonated_user_t1"]["dn"], ("entryUUID",))[
        "entryUUID"
    ][0]
    lo_t2.modify(
        impersonation_users["impersonating_user"]["dn"],
        [("entryUUID", entry_uuid, entry_uuid_target)],
        serverctrls=[ldap.controls.simple.RelaxRulesControl()],
    )
    load_test_app_page(
        page,
        impersonation_users["impersonating_user"]["name"],
        password,
        kdc_idp_hint,
        login_success=False,
    )
