from fastapi import APIRouter, Depends, HTTPException, Path, Request, status
from pydantic import ValidationError
from starlette.responses import Response

from id_broker_common.kelvin import kelvin_session
from provisioning_plugin.dependencies import (
    check_for_authority_admin,
    get_workgroup_by_id,
    member_ids_to_usernames,
)
from provisioning_plugin.models import NonEmptyStr, NoStarStr, WorkGroup
from provisioning_plugin.utils import get_pseudonyms_for_providers
from ucsschool.kelvin.client import InvalidRequest, NoObject, Session, WorkGroup as KelvinWorkGroup

router = APIRouter(tags=["workgroups"], dependencies=[Depends(check_for_authority_admin)])


@router.head("/workgroups/{id}", response_model=None)
@router.get("/workgroups/{id}", response_model=WorkGroup)
async def get(
    request: Request,
    workgroup_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    workgroup: KelvinWorkGroup = Depends(get_workgroup_by_id),
    session: Session = Depends(kelvin_session),
) -> WorkGroup:
    try:
        prov_class = await WorkGroup.from_kelvin_workgroup(
            workgroup.as_dict(), session=session, fill_members=request.method == "GET"
        )
        return WorkGroup.without_prefix(prov_class, school_authority)
    except (ValidationError, ValueError, NoObject) as exc:
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"The workgroup {workgroup_id!r} in school authority {school_authority!r} is "
            f"malformed.",
        ) from exc


@router.post("/workgroups", response_model=WorkGroup, status_code=201)
async def post(
    workgroup_data: WorkGroup,
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    session: Session = Depends(kelvin_session),
) -> WorkGroup:
    workgroup_data = WorkGroup.with_prefix(workgroup_data, school_authority)
    try:
        usernames = await member_ids_to_usernames(
            members=workgroup_data.members, school_authority=school_authority
        )
    except HTTPException as exc:
        if exc.status_code == status.HTTP_404_NOT_FOUND:
            raise HTTPException(
                status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
                detail=f"The workgroup {workgroup_data.name!r} in school {workgroup_data.school!r} "
                f"in school authority {school_authority!r} could not be created: {exc.detail!r}",
            )
        raise
    workgroup = KelvinWorkGroup(
        name=workgroup_data.name,
        school=workgroup_data.school,
        users=usernames,
        description=workgroup_data.description,
        udm_properties={
            "ucsschoolRecordUID": workgroup_data.id,
            "ucsschoolSourceUID": school_authority,
        },
        session=session,
        create_share=False,
    )
    async for udm_prop, pseudonym in get_pseudonyms_for_providers(workgroup_data.id, school_authority):
        workgroup.udm_properties[udm_prop] = pseudonym
    try:
        await workgroup.save()
    except InvalidRequest as exc:
        if exc.status == status.HTTP_409_CONFLICT:
            await workgroup.reload()
            raise HTTPException(
                status_code=exc.status,
                detail={
                    "msg": exc.reason,
                    "conflict_id": workgroup.udm_properties["ucsschoolRecordUID"],
                },
            ) from exc
        else:
            raise HTTPException(status_code=exc.status, detail=exc.reason) from exc
    # To avoid retrieving all members again, at this point we trust that the data was saved correctly
    # It would be safer to use the data from the response. However, that would have a big performance
    # impact. So we'll leave this as it is.
    return WorkGroup.without_prefix(workgroup_data, school_authority)


@router.put("/workgroups/{id}", response_model=WorkGroup)
async def put(
    workgroup_data: WorkGroup,
    workgroup_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    workgroup: KelvinWorkGroup = Depends(get_workgroup_by_id),
):
    workgroup_data = WorkGroup.with_prefix(workgroup_data, school_authority)
    if workgroup_data.school != workgroup.school:
        raise HTTPException(
            status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
            detail="The school must not be changed after creation.",
        )
    workgroup.name = workgroup_data.name
    workgroup.description = workgroup_data.description
    try:
        usernames = await member_ids_to_usernames(
            members=workgroup_data.members, school_authority=school_authority
        )
    except HTTPException as exc:
        if exc.status_code == 404:
            raise HTTPException(
                status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
                detail=f"The workgroup {workgroup_data.name!r} in school {workgroup_data.school!r} "
                f"in school authority {school_authority!r} could not be modified: {exc.detail!r}",
            )
        raise
    workgroup.users = usernames
    try:
        await workgroup.save()
    except InvalidRequest as exc:
        raise HTTPException(status_code=exc.status, detail=exc.reason) from exc
    return WorkGroup.without_prefix(workgroup_data, school_authority)


@router.delete("/workgroups/{id}", status_code=status.HTTP_204_NO_CONTENT)
async def delete(
    workgroup_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    workgroup: KelvinWorkGroup = Depends(get_workgroup_by_id),
) -> Response:
    await workgroup.delete()
    return Response(status_code=status.HTTP_204_NO_CONTENT)
