import asyncio

from fastapi import APIRouter, Depends, HTTPException, Path, Response, status
from ldap3.utils.conv import escape_filter_chars
from pydantic import ValidationError

from id_broker_common.kelvin import kelvin_session
from id_broker_common.ldap_access import ldap_access, wait_for_replication
from provisioning_plugin.dependencies import check_for_authority_admin, get_user_by_id
from provisioning_plugin.models import NonEmptyStr, NoStarStr, User
from provisioning_plugin.utils import get_pseudonyms_for_providers
from ucsschool.kelvin.client import InvalidRequest, Session, User as KelvinUser

router = APIRouter(tags=["users"], dependencies=[Depends(check_for_authority_admin)])


@router.head("/users/{id}", response_model=None)
@router.get("/users/{id}", response_model=User)
async def get(
    user_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    user: KelvinUser = Depends(get_user_by_id),
) -> User:
    try:
        return await User.from_kelvin_user(user, school_authority)
    except (ValidationError, ValueError) as exc:
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"The user {user_id!r} in school authority {school_authority!r} is malformed.",
        ) from exc


@router.post("/users", response_model=User, status_code=201)
async def post(
    user_data: User,
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    session: Session = Depends(kelvin_session),
) -> User:
    user_data.set_prefix(f"{school_authority}-")
    await _check_user_dependencies(user_data)
    user = KelvinUser(
        school=user_data.school,
        schools=user_data.schools,
        school_classes=user_data.school_classes,
        workgroups=user_data.workgroups,
        roles=user_data.roles,
        name=user_data.name,
        firstname=user_data.first_name,
        lastname=user_data.last_name,
        source_uid=school_authority,
        record_uid=user_data.id,
        session=session,
    )
    # Check that legal-guardian parameters are only used with the correct role & convert UUIDs to user-names
    for param, role in (("legal_guardians", "student"), ("legal_wards", "legal_guardian")):
        user_uuids = getattr(user_data, param, None)
        if user_uuids and role not in user.roles:
            error_msg = f"Parameter {param!r} only valid for user-role {role!r}"
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail=error_msg,
            )
        user_names = await User.uuids_to_names(school_authority, user_uuids)
        setattr(user, param, user_names)
    async for udm_prop, pseudonym in get_pseudonyms_for_providers(user_data.id, school_authority):
        user.udm_properties[udm_prop] = pseudonym
    try:
        await user.save()
    except InvalidRequest as exc:
        if exc.status == status.HTTP_409_CONFLICT:
            await user.reload()
            raise HTTPException(
                status_code=exc.status, detail={"msg": exc.reason, "conflict_id": user.record_uid}
            ) from exc
        else:
            raise HTTPException(status_code=exc.status, detail=exc.reason) from exc
    await wait_for_replication(ldap_access(), user.dn)
    return await User.from_kelvin_user(user, school_authority)


@router.put("/users/{id}", response_model=User)
async def put(
    user_data: User,
    user_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    user: KelvinUser = Depends(get_user_by_id),
    session: Session = Depends(kelvin_session),
) -> User:
    if user_data.id != user.record_uid:
        raise HTTPException(
            status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
            detail="The user_id must not be changed after creation!",
        )
    user_data.set_prefix(f"{school_authority}-")
    await _check_user_dependencies(user_data)
    user.school = user_data.school
    user.schools = user_data.schools
    user.school_classes = user_data.school_classes
    user.workgroups = user_data.workgroups
    user.roles = user_data.roles
    user.name = user_data.name
    user.firstname = user_data.first_name
    user.lastname = user_data.last_name
    # Check that legal-guardian parameters are only used with the correct role & convert UUIDs to user-names
    for param, role in (("legal_guardians", "student"), ("legal_wards", "legal_guardian")):
        user_uuids = getattr(user_data, param, None)
        if user_uuids and role not in user.roles:
            error_msg = f"Parameter {param!r} only valid for user-role {role!r}"
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail=error_msg,
            )
        user_names = await User.uuids_to_names(school_authority, user_uuids)
        setattr(user, param, user_names)
    try:
        await user.save()
        return await User.from_kelvin_user(user, school_authority)
    except InvalidRequest as exc:
        raise HTTPException(status_code=exc.status, detail=exc.reason) from exc
    except (ValidationError, ValueError) as exc:
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"The user {user_id!r} in school authority {school_authority!r} is malformed!",
        ) from exc


@router.delete("/users/{id}", status_code=status.HTTP_204_NO_CONTENT)
async def delete(
    user_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    user: KelvinUser = Depends(get_user_by_id),
) -> Response:
    await user.delete()
    await wait_for_replication(ldap_access(), user.dn, should_exist=False)
    return Response(status_code=status.HTTP_204_NO_CONTENT)


async def _check_school_exists(school: str) -> None:
    ldap = ldap_access()
    exists = await ldap.search(
        f"(&(objectClass=ucsschoolOrganizationalUnit)(ou={escape_filter_chars(school)}))",
        use_master=True,
    )
    if not exists:
        raise HTTPException(
            status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
            detail=f"School {school!r} does not exist yet.",
        )


async def _check_school_class_exists(school: str, school_class: str) -> None:
    ldap = ldap_access()
    exists = await ldap.search(
        f"(&(ucsschoolRole=school_class:*)"
        f"(cn={escape_filter_chars(school)}-{escape_filter_chars(school_class)}))",
        use_master=True,
    )
    if not exists:
        raise HTTPException(
            status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
            detail=f"School class {school_class!r} does not exist yet.",
        )


async def _check_workgroup_exists(school: str, workgroup: str) -> None:
    ldap = ldap_access()
    exists = await ldap.search(
        f"(&(ucsschoolRole=workgroup:*)(cn={escape_filter_chars(school)}-{escape_filter_chars(workgroup)}))",
        use_master=True,
    )
    if not exists:
        raise HTTPException(
            status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
            detail=f"Workgroup {workgroup!r} does not exist yet.",
        )


async def _check_user_exists(user: str) -> None:
    ldap = ldap_access()
    exists = await ldap.search(
        f"(&(objectClass=organizationalPerson)(ucsschoolRecordUID={escape_filter_chars(user)}))",
        use_master=True,
    )
    if not exists:
        raise HTTPException(
            status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
            detail=f"User {user!r} does not exist yet.",
        )


async def _check_user_dependencies(user: User) -> None:
    """
    Use Ldap to verify existence of schools, school classes, workgroups & connected users.
    May raise HTTPException(422).
    """
    dependency_check_tasks = []
    for school in user.schools:
        dependency_check_tasks.append(_check_school_exists(school))
    for school, school_classes in user.school_classes.items():
        for school_class in school_classes:
            dependency_check_tasks.append(
                _check_school_class_exists(school=school, school_class=school_class)
            )
    for school, workgroups in user.workgroups.items():
        for workgroup in workgroups:
            dependency_check_tasks.append(_check_workgroup_exists(school=school, workgroup=workgroup))
    for legal_guardian in user.legal_guardians:
        dependency_check_tasks.append(_check_user_exists(user=legal_guardian))
    for legal_ward in user.legal_wards:
        dependency_check_tasks.append(_check_user_exists(user=legal_ward))
    await asyncio.gather(*dependency_check_tasks)
