import logging
import random
from typing import Dict, List, Tuple

from faker import Faker
from locust import events, task

from performance_tests.common import (
    ProvisioningClient,
    ProvisioningSchoolContext,
    ResponseError,
    TestData,
    UsedSchools,
    get_classes,
    get_provisioning_api_admins,
    get_schools,
)

school_authorities: List[str] = []  # ["Traeger1", "Traeger"]
user_credentials: Dict[str, Tuple[str, str]] = {}  # [(username, password), ..]
id_broker_schools: Dict[str, Dict[str, str]] = {}  # {sch_auth: {ou: uuid, ...}}
id_broker_school_classes: Dict[str, Dict[str, Dict[str, str]]] = {}  # {sch_auth: {ou: {cl: uuid, ...}}}
NUM_SCHOOLS_PER_SCHOOL_AUTHORITY = 5  # loading all OUs from test DB takes to long
fake = Faker()
used_schools = UsedSchools()


@events.init.add_listener
def on_init(environment, **kwargs):
    data = TestData()
    school_authorities.extend(data.school_authorities)
    user_credentials.update(get_provisioning_api_admins(data))
    id_broker_schools.update(get_schools(data))
    id_broker_school_classes.update(get_classes(data, NUM_SCHOOLS_PER_SCHOOL_AUTHORITY))


class ProvisioningAPIClient(ProvisioningClient):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.school_authority = random.choice(school_authorities)
        logging.info("Testing with school authority: %r", self.school_authority)
        self.auth_username, self.auth_password = user_credentials[self.school_authority]
        self.schools = list(id_broker_school_classes[self.school_authority].keys())
        logging.info("Testing with schools: %r.", sorted(self.schools))

    @task
    def user_create(self) -> None:
        school = random.choice(self.schools)
        used_schools.add(self.school_authority, school)
        username = f"test.{fake.user_name()[:10]}{fake.pyint(1000, 9999)}"
        entry_uuid = fake.uuid4()
        roles = [random.choice(["student", "teacher", "legal_guardian"])]
        classes = random.choices(
            list(id_broker_school_classes[self.school_authority][school].keys()),
            k=1 if "student" in roles else 2,
        )
        try:
            super().user_create(
                school_authority=self.school_authority,
                obj_id=entry_uuid,
                first_name=fake.first_name(),
                last_name=fake.last_name(),
                user_name=username,
                context={school: ProvisioningSchoolContext(classes=classes, roles=roles)},
            )
        except ResponseError as exc:
            logging.error(
                "%s: school_authority=%r name=%r entry_uuid=%r -> %r",
                self.greenlet.name,
                self.school_authority,
                username,
                entry_uuid,
                exc,
            )


@events.test_stop.add_listener
def _(environment, **_kwargs):
    logging.info("Used schools: %r", sorted(f"{s_a}-{ou}" for s_a, ou in used_schools.schools))
    used_schools.persist()
