import logging
import random
from copy import deepcopy
from typing import Dict, List, Tuple

from gevent.lock import BoundedSemaphore
from locust import events, task

from performance_tests.common import (
    ProvisioningClient,
    ResponseError,
    TestData,
    UsedSchools,
    get_provisioning_api_admins,
    get_users,
)

school_authorities: List[str] = []  # ["Traeger1", "Traeger"]
user_credentials: Dict[str, Tuple[str, str]] = {}  # [(username, password), ..]
id_broker_school_users: Dict[str, Dict[str, Dict[str, str]]] = {}  # {sch_auth: {ou: {uid: uuid, ...}}}
sem_id_broker_school_users = BoundedSemaphore()
NUM_SCHOOLS_PER_SCHOOL_AUTHORITY = 10  # loading all OUs from test DB takes to long
used_schools = UsedSchools()
consumed_schools = deepcopy(used_schools._schools)
sem_schools = BoundedSemaphore()


@events.init.add_listener
def on_init(environment, **kwargs):
    data = TestData()
    school_authorities.extend(data.school_authorities)
    user_credentials.update(get_provisioning_api_admins(data))
    with sem_id_broker_school_users:
        id_broker_school_users.update(get_users(data, NUM_SCHOOLS_PER_SCHOOL_AUTHORITY))


class ProvisioningAPIClient(ProvisioningClient):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._name = id(self)
        self.school_authority = random.choice(school_authorities)
        logging.info("Testing with school authority: %r", self.school_authority)
        self.auth_username, self.auth_password = user_credentials[self.school_authority]
        self.schools = list(id_broker_school_users[self.school_authority].keys())
        self.school = self.get_fresh_school()

    def get_fresh_school(self) -> str:
        logging.info("%s: Looking for a fresh school...", self._name)
        with sem_schools:
            with sem_id_broker_school_users:
                schools = [
                    ou
                    for ou in id_broker_school_users[self.school_authority].keys()
                    if (self.school_authority, ou) not in consumed_schools
                ]
            logging.info(
                "%s: Unconsumed schools in id_broker_school_classes[%r]: %r",
                self._name,
                self.school_authority,
                schools,
            )
            if len(schools) < 30 and used_schools.length(self.school_authority) <= 400:
                logging.info("%s: Getting new schools...", self._name)
                schools = self.get_fresh_schools()
            school = random.choice(schools)
            logging.info("%s: Using school: %r", self._name, school)
            used_schools.add(self.school_authority, school)
        with sem_id_broker_school_users:
            len_users = len(id_broker_school_users[self.school_authority][school])
        logging.info("%s: Testing with school %r incl. %d users.", self._name, school, len_users)
        return school

    def get_fresh_schools(self) -> List[str]:
        schools = []
        while not schools:
            logging.info("%s: Loading classes from test data...", self._name)
            new_users = get_users(
                data=TestData(), schools_per_authority=1, school_authorities=[self.school_authority]
            )
            new_schools = sorted(new_users[self.school_authority])
            logging.info(
                "%s: Got users for %r schools: %r", self._name, self.school_authority, new_schools
            )
            schools = [
                ou for ou in new_schools if (self.school_authority, ou) not in used_schools.schools
            ]
            logging.info("%s: Unused schools in those: %r", self._name, schools)
            if not schools:
                continue  # needs break condition: when all schools are used -> loop forever
            with sem_id_broker_school_users:
                id_broker_school_users[self.school_authority][schools[0]] = new_users[
                    self.school_authority
                ][schools[0]]
        return schools

    @task
    def user_delete(self) -> None:
        if used_schools.length(self.school_authority) >= 400:
            return
        while True:
            with sem_id_broker_school_users:
                users = id_broker_school_users[self.school_authority][self.school]
            if random.choice(range(10)) == 0:
                logging.info("%s: Random new school...", self._name)
                self.school = self.get_fresh_school()
                continue
            if not users:
                logging.info(
                    "%s: Found no users in %r school %r.", self._name, self.school_authority, self.school
                )
                consumed_schools.add((self.school_authority, self.school))
                self.school = self.get_fresh_school()
                continue
            name, entry_uuid = random.choice(list(users.items()))
            with sem_id_broker_school_users:
                del id_broker_school_users[self.school_authority][self.school][name]
            break
        try:
            if (self.school_authority, self.school) not in used_schools.schools:
                used_schools.add(self.school_authority, self.school)
            super().user_remove(school_authority=self.school_authority, obj_id=entry_uuid)
        except ResponseError as exc:
            logging.error(
                "%s: school_authority=%r name=%r entry_uuid=%r -> %r",
                self.greenlet.name,
                self.school_authority,
                name,
                entry_uuid,
                exc,
            )


@events.test_stop.add_listener
def _(environment, **_kwargs):
    logging.info("Used schools: %r", sorted(f"{s_a}-{ou}" for s_a, ou in used_schools.schools))
    used_schools.persist()
