import logging
import random
from typing import Dict, List, Tuple

from faker import Faker
from locust import events, task

from performance_tests.common import (
    ProvisioningClient,
    ResponseError,
    TestData,
    UsedSchools,
    get_provisioning_api_admins,
    get_schools,
)

school_authorities: List[str] = []  # ["Traeger1", "Traeger"]
user_credentials: Dict[str, Tuple[str, str]] = {}  # [(username, password), ..]
id_broker_schools: Dict[str, Dict[str, str]] = {}  # {sch_auth: {ou: uuid, ...}}
fake = Faker()
used_schools = UsedSchools()


def get_schools_subset(data: TestData, number: int) -> Dict[str, Dict[str, str]]:
    schools = get_schools(data)
    tmp = {}

    for traeger in schools:
        tmp[traeger] = random.sample(schools[traeger].keys(), k=(400 - number))

    for t in tmp:
        for s in tmp[t]:
            del schools[t][s]

    return schools


@events.init.add_listener
def on_init(environment, **kwargs):
    data = TestData()
    school_authorities.extend(data.school_authorities)
    user_credentials.update(get_provisioning_api_admins(data))
    id_broker_schools.update(get_schools_subset(data, 50))


class ProvisioningAPIClient(ProvisioningClient):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.school_authority = random.choice(school_authorities)
        logging.info("Testing with school authority: %r", self.school_authority)
        self.auth_username, self.auth_password = user_credentials[self.school_authority]
        self.schools = list(id_broker_schools[self.school_authority].items())
        logging.info("Testing with schools: %r.", sorted(self.schools))

    @task
    def school_class_create(self) -> None:
        school, _ = random.choice(self.schools)
        used_schools.add(self.school_authority, school)
        name = f"test.{fake.user_name()[:10]}{fake.pyint(1000, 9999)}"
        entry_uuid = fake.uuid4()
        try:
            super().school_class_create(
                school_authority=self.school_authority,
                obj_id=entry_uuid,
                name=name,
                school=school,
                description="Performance Test temp class",
                members=[],  # ID connector will always create empty classes and add users afterwards
            )
        except ResponseError as exc:
            logging.error(
                "%s: school_authority=%r name=%r entry_uuid=%r -> %r",
                self.greenlet.name,
                self.school_authority,
                name,
                entry_uuid,
                exc,
            )


@events.test_stop.add_listener
def _(environment, **_kwargs):
    logging.info("Used schools: %r", sorted(f"{s_a}-{ou}" for s_a, ou in used_schools.schools))
    used_schools.persist()
